import stripAnsi from "strip-ansi";
import { LabelName, LinkType, Stage, Status, } from "./model.js";
export class AllureCommandStepExecutable {
    name = "";
    attachments = [];
    metadata = {};
    constructor(name) {
        this.name = name;
    }
    static toExecutableItem(runtime, stepMetadata) {
        const executable = {
            ...stepMetadata,
            attachments: [],
            steps: [],
        };
        if (stepMetadata.attachments?.length > 0) {
            stepMetadata.attachments.forEach((attachment) => {
                const attachmentContent = Buffer.from(attachment.content, attachment.encoding);
                const attachmentFilename = runtime.writeAttachment(attachmentContent, attachment.type, attachment.encoding);
                executable.attachments.push({
                    name: attachment.name,
                    type: attachment.type,
                    source: attachmentFilename,
                });
            });
        }
        if (stepMetadata.steps?.length > 0) {
            executable.steps = stepMetadata.steps.map((nestedStep) => AllureCommandStepExecutable.toExecutableItem(runtime, nestedStep));
        }
        return executable;
    }
    label(label, value) {
        if (!this.metadata.labels) {
            this.metadata.labels = [];
        }
        this.metadata.labels.push({
            name: label,
            value,
        });
    }
    link(url, name, type) {
        if (!this.metadata.links) {
            this.metadata.links = [];
        }
        this.metadata.links.push({
            name,
            url,
            type,
        });
    }
    parameter(name, value, options) {
        if (!this.metadata.parameter) {
            this.metadata.parameter = [];
        }
        this.metadata.parameter.push({
            name,
            value: JSON.stringify(value),
            excluded: options?.excluded || false,
            mode: options?.mode,
        });
    }
    epic(epic) {
        this.label(LabelName.EPIC, epic);
    }
    feature(feature) {
        this.label(LabelName.FEATURE, feature);
    }
    story(story) {
        this.label(LabelName.STORY, story);
    }
    suite(name) {
        this.label(LabelName.SUITE, name);
    }
    parentSuite(name) {
        this.label(LabelName.PARENT_SUITE, name);
    }
    subSuite(name) {
        this.label(LabelName.SUB_SUITE, name);
    }
    owner(owner) {
        this.label(LabelName.OWNER, owner);
    }
    severity(severity) {
        this.label(LabelName.SEVERITY, severity);
    }
    tag(tag) {
        this.label(LabelName.TAG, tag);
    }
    issue(name, url) {
        this.link(url, name, LinkType.ISSUE);
    }
    tms(name, url) {
        this.link(url, name, LinkType.TMS);
    }
    attach(content, type) {
        const isBuffer = Buffer.isBuffer(content);
        this.attachments.push({
            name: "attachment",
            content: isBuffer ? content.toString("base64") : content,
            encoding: isBuffer ? "base64" : "utf8",
            type,
        });
    }
    description(content) {
        this.metadata.description = content;
    }
    async step(name, body) {
        if (!this.metadata.steps) {
            this.metadata.steps = [];
        }
        const nestedStep = new AllureCommandStepExecutable(name);
        await nestedStep.run(body, async ({ labels = [], links = [], parameter = [], steps = [] }) => {
            this.metadata.labels = (this.metadata.labels || []).concat(labels);
            this.metadata.links = (this.metadata.links || []).concat(links);
            this.metadata.parameter = (this.metadata.parameter || []).concat(parameter);
            this.metadata.steps = (this.metadata.steps || []).concat(steps);
        });
    }
    async start(body) {
        return await new Promise((resolve) => this.run(body, async (result) => resolve(result)));
    }
    async run(body, messageEmitter) {
        const startDate = new Date().getTime();
        try {
            await body.call(this, this);
            const { steps = [], description = "", descriptionHtml = "", ...metadata } = this.metadata;
            await messageEmitter({
                ...metadata,
                steps: [
                    {
                        name: this.name,
                        start: startDate,
                        stop: new Date().getTime(),
                        stage: Stage.FINISHED,
                        status: Status.PASSED,
                        statusDetails: {},
                        attachments: this.attachments,
                        parameters: [],
                        steps,
                        description,
                    },
                ],
            });
        }
        catch (e) {
            const { steps = [], description = "", descriptionHtml = "", ...metadata } = this.metadata;
            await messageEmitter({
                ...metadata,
                steps: [
                    {
                        name: this.name,
                        start: startDate,
                        stop: new Date().getTime(),
                        stage: Stage.FINISHED,
                        status: Status.BROKEN,
                        statusDetails: {
                            message: stripAnsi((e.message || "")),
                            trace: stripAnsi((e.stack || "")),
                        },
                        attachments: this.attachments,
                        parameters: [],
                        steps,
                        description,
                    },
                ],
            });
            throw e;
        }
    }
}
//# sourceMappingURL=AllureCommandStep.js.map