// src/index.ts
import url from "node:url";
import os from "node:os";
import WDIOReporter from "@wdio/reporter";
import junitReportBuilder from "junit-report-builder";

// src/utils.ts
import stringify from "json-stringify-safe";
var OBJLENGTH = 10;
var ARRLENGTH = 10;
var STRINGLIMIT = 1e3;
var STRINGTRUNCATE = 200;
var limit = function(rawVal) {
  if (!rawVal) {
    return rawVal;
  }
  let val = JSON.parse(stringify(rawVal));
  const type = Object.prototype.toString.call(val);
  if (type === "[object String]") {
    if (val.length > 100 && isBase64(val)) {
      return `[base64] ${val.length} bytes`;
    }
    if (val.length > STRINGLIMIT) {
      return val.substring(0, STRINGTRUNCATE) + ` ... (${val.length - STRINGTRUNCATE} more bytes)`;
    }
    return val;
  } else if (type === "[object Array]") {
    const length = val.length;
    if (length > ARRLENGTH) {
      val = val.slice(0, ARRLENGTH);
      val.push(`(${length - ARRLENGTH} more items)`);
    }
    return val.map(limit);
  } else if (type === "[object Object]") {
    const keys = Object.keys(val);
    const removed = [];
    for (let i = 0, l = keys.length; i < l; i++) {
      if (i < OBJLENGTH) {
        val[keys[i]] = limit(val[keys[i]]);
      } else {
        delete val[keys[i]];
        removed.push(keys[i]);
      }
    }
    if (removed.length) {
      val._ = keys.length - OBJLENGTH + " more keys: " + JSON.stringify(removed);
    }
    return val;
  }
  return val;
};
function isBase64(str) {
  if (typeof str !== "string") {
    throw new Error("Expected string but received invalid type.");
  }
  const len = str.length;
  const notBase64 = /[^A-Z0-9+/=]/i;
  if (!len || len % 4 !== 0 || notBase64.test(str)) {
    return false;
  }
  const firstPaddingChar = str.indexOf("=");
  return firstPaddingChar === -1 || firstPaddingChar === len - 1 || firstPaddingChar === len - 2 && str[len - 1] === "=";
}

// src/common/api.ts
var events = {
  addProperty: "junit:addProperty"
};
var tellReporter = (event, msg = {}) => {
  process.emit(event, msg);
};
function addProperty(name, value) {
  tellReporter(events.addProperty, { name, value });
}

// src/index.ts
var ansiRegex = new RegExp([
  "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
  "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))"
].join("|"), "g");
var JunitReporter = class extends WDIOReporter {
  constructor(options) {
    super(options);
    this.options = options;
    this._isWindows = os.platform() === "win32";
    this._addWorkerLogs = options.addWorkerLogs ?? false;
    this._testToAdditionalInformation = {};
    this._originalStdoutWrite = process.stdout.write.bind(process.stdout);
    this._suiteNameRegEx = this.options.suiteNameFormat instanceof RegExp ? this.options.suiteNameFormat : /[^a-zA-Z0-9@]+/;
    const processObj = process;
    if (this._addWorkerLogs) {
      processObj.stdout.write = this._appendConsoleLog.bind(this);
    }
    processObj.on(events.addProperty, this._addPropertyToCurrentTest.bind(this));
  }
  _suiteNameRegEx;
  _packageName;
  _suiteTitleLabel;
  _fileNameLabel;
  _activeFeature;
  _activeFeatureName;
  _testToAdditionalInformation;
  _currentTest;
  _originalStdoutWrite;
  _addWorkerLogs;
  _isWindows;
  onTestRetry(testStats) {
    testStats.skip("Retry");
  }
  onTestStart(test) {
    this._currentTest = test;
    this._testToAdditionalInformation[test.uid] = {
      workerConsoleLog: "",
      properties: {},
      uid: test.uid
    };
  }
  onRunnerEnd(runner) {
    const xml = this._buildJunitXml(runner);
    this.write(xml);
  }
  _addPropertyToCurrentTest(dataObj) {
    if (this._currentTest?.uid) {
      this._testToAdditionalInformation[this._currentTest.uid].properties[dataObj.name] = dataObj.value;
    }
  }
  _appendConsoleLog(chunk, encoding, callback) {
    if (this._currentTest?.uid) {
      if (typeof chunk === "string" && !chunk.includes("mwebdriver")) {
        this._testToAdditionalInformation[this._currentTest.uid].workerConsoleLog = (this._testToAdditionalInformation[this._currentTest.uid].workerConsoleLog ?? "") + chunk;
      }
    }
    return this._originalStdoutWrite(chunk, encoding, callback);
  }
  _prepareName(name = "Skipped test") {
    return name.split(this._suiteNameRegEx).filter(
      (item) => item && item.length
    ).join(" ");
  }
  _addFailedHooks(suite) {
    const failedHooks = suite.hooks.filter((hook) => hook.error && hook.title.match(/^"(before|after)( all| each)?" hook/));
    failedHooks.forEach((hook) => {
      const { title, _duration, error, state } = hook;
      suite.tests.push({
        _duration,
        title,
        error,
        state,
        output: []
      });
    });
    return suite;
  }
  _addCucumberFeatureToBuilder(builder, runner, specFileName, suite) {
    const featureName = !this.options.suiteNameFormat || this.options.suiteNameFormat instanceof RegExp ? this._prepareName(suite.title) : this.options.suiteNameFormat({ name: this.options.suiteNameFormat.name, suite });
    const filePath = specFileName.replace(process.cwd(), ".");
    if (suite.type === "feature") {
      const feature = builder.testSuite().name(featureName).timestamp(suite.start).time(suite._duration / 1e3).property("specId", 0).property(this._suiteTitleLabel, suite.title).property("capabilities", runner.sanitizedCapabilities).property(this._fileNameLabel, filePath);
      this._activeFeature = feature;
      this._activeFeatureName = featureName;
    } else if (this._activeFeature) {
      let scenario = suite;
      const testName = this._prepareName(suite.title);
      const classNameFormat = this.options.classNameFormat ? this.options.classNameFormat({ packageName: this._packageName, activeFeatureName: this._activeFeatureName }) : `${this._packageName}.${this._activeFeatureName}`;
      const testCase = this._activeFeature.testCase().className(classNameFormat).name(`${testName}`).time(scenario._duration / 1e3);
      if (this.options.addFileAttribute) {
        testCase.file(filePath);
      }
      scenario = this._addFailedHooks(scenario);
      let stepsOutput = "";
      let isFailing = false;
      for (const stepKey of Object.keys(scenario.tests)) {
        if (stepKey === "undefined") {
          continue;
        }
        let stepEmoji = "\u2705";
        const step = scenario.tests[stepKey];
        if (step.state === "pending" || step.state === "skipped") {
          if (!isFailing) {
            testCase.skipped();
          }
          stepEmoji = "\u26A0\uFE0F";
        } else if (step.state === "failed") {
          if (step.error) {
            if (this.options.errorOptions) {
              const errorOptions = this.options.errorOptions;
              for (const key of Object.keys(errorOptions)) {
                testCase[key](
                  step.error ? step.error[errorOptions[key]] : null
                );
              }
            } else {
              testCase.failure(step.error.message);
            }
            testCase.standardError(`
${step.error.stack}
`);
          } else {
            testCase.failure();
          }
          isFailing = true;
          stepEmoji = "\u2757";
        }
        const output = this._getStandardOutput(step);
        stepsOutput += output ? stepEmoji + " " + step.title : stepEmoji + " " + step.title + "\n" + output;
      }
      testCase.standardOutput(`
${stepsOutput}
`);
    }
    return builder;
  }
  _addSuiteToBuilder(builder, runner, specFileName, suite) {
    const filePath = specFileName.replace(process.cwd(), ".");
    const suiteName = !this.options.suiteNameFormat || this.options.suiteNameFormat instanceof RegExp ? this._prepareName(suite.title) : this.options.suiteNameFormat({ name: this.options.suiteNameFormat.name, suite });
    const testSuite = builder.testSuite().name(suiteName).timestamp(suite.start).time(suite._duration / 1e3).property("specId", 0).property(this._suiteTitleLabel, suite.title).property("capabilities", runner.sanitizedCapabilities).property(this._fileNameLabel, filePath);
    suite = this._addFailedHooks(suite);
    for (const testKey of Object.keys(suite.tests)) {
      if (testKey === "undefined") {
        continue;
      }
      const test = suite.tests[testKey];
      const testName = this._prepareName(test.title);
      const classNameFormat = this.options.classNameFormat ? this.options.classNameFormat({ packageName: this._packageName, suite }) : `${this._packageName}.${(suite.fullTitle || suite.title).replace(/\s/g, "_")}`;
      const testCase = testSuite.testCase().className(classNameFormat).name(testName).time(test._duration / 1e3);
      if (this.options.addFileAttribute) {
        testCase.file(filePath);
      }
      if (test.state === "pending" || test.state === "skipped") {
        testCase.skipped();
        if (test.error) {
          testCase.standardError(`
${test.error.stack?.replace(ansiRegex, "")}
`);
        }
      } else if (test.state === "failed") {
        if (test.error) {
          if (test.error.message) {
            test.error.message = test.error.message.replace(ansiRegex, "");
          }
          if (this.options.errorOptions) {
            const errorOptions = this.options.errorOptions;
            for (const key of Object.keys(errorOptions)) {
              testCase[key](test.error[errorOptions[key]]);
            }
          } else {
            testCase.failure(test.error.message);
          }
          testCase.standardError(`
${test.error.stack?.replace(ansiRegex, "")}
`);
        } else {
          testCase.failure();
        }
      }
      for (const propName of Object.keys(this._testToAdditionalInformation[test.uid]?.properties ?? {})) {
        testCase.property(propName, this._testToAdditionalInformation[test.uid].properties[propName]);
      }
      const output = this._getStandardOutput(test);
      if (output) {
        testCase.standardOutput(`
${output}
`);
      }
    }
    return builder;
  }
  _buildJunitXml(runner) {
    const builder = junitReportBuilder.newBuilder();
    if (runner.config.hostname !== void 0 && runner.config.hostname.indexOf("browserstack") > -1) {
      const browserstackSanitizedCapabilities = [
        // @ts-expect-error capability only exists when running on BrowserStack
        runner.capabilities.device,
        // @ts-expect-error capability only exists when running on BrowserStack
        runner.capabilities.os,
        // @ts-expect-error capability only exists when running on BrowserStack
        (runner.capabilities.os_version || "").replace(/\./g, "_")
      ].filter(Boolean).map((capability) => capability.toLowerCase()).join(".").replace(/ /g, "") || runner.sanitizedCapabilities;
      this._packageName = this.options.packageName || browserstackSanitizedCapabilities;
    } else {
      this._packageName = this.options.packageName || runner.sanitizedCapabilities;
    }
    const isCucumberFrameworkRunner = runner.config.framework === "cucumber";
    if (isCucumberFrameworkRunner) {
      this._packageName = `CucumberJUnitReport-${this._packageName}`;
      this._suiteTitleLabel = "featureName";
      this._fileNameLabel = "featureFile";
    } else {
      this._suiteTitleLabel = "suiteName";
      this._fileNameLabel = "file";
    }
    runner.specs.forEach((specFileName) => {
      if (isCucumberFrameworkRunner) {
        this._buildOrderedReport(builder, runner, specFileName, "feature", isCucumberFrameworkRunner);
        this._buildOrderedReport(builder, runner, specFileName, "scenario", isCucumberFrameworkRunner);
      } else {
        this._buildOrderedReport(builder, runner, specFileName, "", isCucumberFrameworkRunner);
      }
    });
    return builder.build();
  }
  _buildOrderedReport(builder, runner, specFileName, type, isCucumberFrameworkRunner) {
    const suiteKeys = Object.keys(this.suites);
    if (suiteKeys.length === 0) {
      const error = this.runnerStat?.error ?? "No tests found";
      if (!isCucumberFrameworkRunner || isCucumberFrameworkRunner && type === "feature") {
        return builder.testSuite().testCase().className("").name("").failure(error);
      }
    }
    for (const suiteKey of Object.keys(this.suites)) {
      if (suiteKey.match(/^"before all"/)) {
        continue;
      }
      const suite = this.suites[suiteKey];
      const sameSpecFileName = this._sameFileName(specFileName, suite.file);
      if (isCucumberFrameworkRunner && suite.type === type && sameSpecFileName) {
        builder = this._addCucumberFeatureToBuilder(builder, runner, specFileName, suite);
      } else if (!isCucumberFrameworkRunner && sameSpecFileName) {
        builder = this._addSuiteToBuilder(builder, runner, specFileName, suite);
      }
    }
    return builder;
  }
  _getStandardOutput(test) {
    let consoleOutput = "";
    if (this._addWorkerLogs) {
      consoleOutput = this._testToAdditionalInformation[test.uid]?.workerConsoleLog ?? "";
    }
    const commandText = this._getCommandStandardOutput(test);
    let result = "";
    if (consoleOutput !== "") {
      result += consoleOutput;
    }
    if (commandText !== "" && consoleOutput !== "") {
      result += "\n...command output...\n\n";
    }
    result += commandText;
    return result;
  }
  _getCommandStandardOutput(test) {
    const standardOutput = [];
    test.output.forEach((data) => {
      switch (data.type) {
        case "command":
          standardOutput.push(
            data.method ? `COMMAND: ${data.method.toUpperCase()} ${data.endpoint.replace(":sessionId", data.sessionId)} - ${this._format(data.body)}` : `COMMAND: ${data.command} - ${this._format(data.params)}`
          );
          break;
        case "result":
          standardOutput.push(`RESULT: ${this._format(data.body)}`);
          break;
      }
    });
    return standardOutput.length ? standardOutput.join("\n") : "";
  }
  _format(val) {
    return JSON.stringify(limit(val));
  }
  _sameFileName(file1, file2) {
    if (!file1 && !file2) {
      return true;
    }
    if (!file1 || !file2) {
      return false;
    }
    file1 = file1.startsWith("file://") ? url.fileURLToPath(file1) : file1;
    file2 = file2.startsWith("file://") ? url.fileURLToPath(file2) : file2;
    return file1.localeCompare(file2, void 0, { sensitivity: this._isWindows ? "accent" : "variant" }) === 0;
  }
};
var index_default = JunitReporter;
export {
  addProperty,
  index_default as default,
  events
};
