/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { assert } from './util';
/**
 * Serializable defines the serialization contract.
 *
 * TFJS requires serializable classes to return their className when asked
 * to avoid issues with minification.
 */
export class Serializable {
    /**
     * Return the class name for this class to use in serialization contexts.
     *
     * Generally speaking this will be the same thing that constructor.name
     * would have returned.  However, the class name needs to be robust
     * against minification for serialization/deserialization to work properly.
     *
     * There's also places such as initializers.VarianceScaling, where
     * implementation details between different languages led to different
     * class hierarchies and a non-leaf node is used for serialization purposes.
     */
    getClassName() {
        return this.constructor
            .className;
    }
    /**
     * Creates an instance of T from a ConfigDict.
     *
     * This works for most descendants of serializable.  A few need to
     * provide special handling.
     * @param cls A Constructor for the class to instantiate.
     * @param config The Configuration for the object.
     */
    /** @nocollapse */
    static fromConfig(cls, config) {
        return new cls(config);
    }
}
/**
 * Maps string keys to class constructors.
 *
 * Used during (de)serialization from the cross-language JSON format, which
 * requires the class name in the serialization format matches the class
 * names as used in Python, should it exist.
 */
export class SerializationMap {
    constructor() {
        this.classNameMap = {};
    }
    /**
     * Returns the singleton instance of the map.
     */
    static getMap() {
        if (SerializationMap.instance == null) {
            SerializationMap.instance = new SerializationMap();
        }
        return SerializationMap.instance;
    }
    /**
     * Registers the class as serializable.
     */
    static register(cls) {
        SerializationMap.getMap().classNameMap[cls.className] =
            [cls, cls.fromConfig];
    }
}
/**
 * Register a class with the serialization map of TensorFlow.js.
 *
 * This is often used for registering custom Layers, so they can be
 * serialized and deserialized.
 *
 * Example:
 *
 * ```js
 * class MyCustomLayer extends tf.layers.Layer {
 *   static className = 'MyCustomLayer';
 *
 *   constructor(config) {
 *     super(config);
 *   }
 * }
 * tf.serialization.registerClass(MyCustomLayer);
 * ```
 *
 * @param cls The class to be registered. It must have a public static member
 *   called `className` defined and the value must be a non-empty string.
 *
 * @doc {heading: 'Models', subheading: 'Serialization', ignoreCI: true}
 */
export function registerClass(cls) {
    assert(cls.className != null, () => `Class being registered does not have the static className ` +
        `property defined.`);
    assert(typeof cls.className === 'string', () => `className is required to be a string, but got type ` +
        typeof cls.className);
    assert(cls.className.length > 0, () => `Class being registered has an empty-string as its className, ` +
        `which is disallowed.`);
    SerializationMap.register(cls);
}
//# sourceMappingURL=data:application/json;base64,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