/**
 * @license
 * Copyright 2019 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { describeWithFlags, NODE_ENVS } from '../jasmine_util';
import * as platform_node from './platform_node';
import { PlatformNode } from './platform_node';
describeWithFlags('PlatformNode', NODE_ENVS, () => {
    it('fetch should use global.fetch if defined', async () => {
        const globalFetch = tf.env().global.fetch;
        spyOn(tf.env().global, 'fetch').and.returnValue(() => { });
        const platform = new PlatformNode();
        await platform.fetch('test/url', { method: 'GET' });
        expect(tf.env().global.fetch).toHaveBeenCalledWith('test/url', {
            method: 'GET'
        });
        tf.env().global.fetch = globalFetch;
    });
    it('fetch should use node-fetch with tf.env().global.fetch is null', async () => {
        const globalFetch = tf.env().global.fetch;
        tf.env().global.fetch = null;
        const platform = new PlatformNode();
        const savedFetch = platform_node.getSystemFetch();
        // Null out the system fetch so we force it to require node-fetch.
        platform_node.resetSystemFetch();
        const testFetch = { fetch: (url, init) => { } };
        // Mock the actual fetch call.
        spyOn(testFetch, 'fetch').and.returnValue(() => { });
        // Mock the import to override the real require of node-fetch.
        spyOn(platform_node.getNodeFetch, 'importFetch')
            .and.callFake(() => (url, init) => testFetch.fetch(url, init));
        await platform.fetch('test/url', { method: 'GET' });
        expect(platform_node.getNodeFetch.importFetch).toHaveBeenCalled();
        expect(testFetch.fetch).toHaveBeenCalledWith('test/url', {
            method: 'GET'
        });
        platform_node.setSystemFetch(savedFetch);
        tf.env().global.fetch = globalFetch;
    });
    it('now should use process.hrtime', async () => {
        const time = [100, 200];
        spyOn(process, 'hrtime').and.returnValue(time);
        expect(tf.env().platform.now()).toEqual(time[0] * 1000 + time[1] / 1000000);
    });
    it('encodeUTF8 single string', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('hello', 'utf-8');
        expect(bytes.length).toBe(5);
        expect(bytes).toEqual(new Uint8Array([104, 101, 108, 108, 111]));
    });
    it('encodeUTF8 two strings delimited', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('hello\x00world', 'utf-8');
        expect(bytes.length).toBe(11);
        expect(bytes).toEqual(new Uint8Array([104, 101, 108, 108, 111, 0, 119, 111, 114, 108, 100]));
    });
    it('encodeUTF8 cyrillic', () => {
        const platform = new PlatformNode();
        const bytes = platform.encode('Здраво', 'utf-8');
        expect(bytes.length).toBe(12);
        expect(bytes).toEqual(new Uint8Array([208, 151, 208, 180, 209, 128, 208, 176, 208, 178, 208, 190]));
    });
    it('decode single string', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([104, 101, 108, 108, 111]), 'utf8');
        expect(s.length).toBe(5);
        expect(s).toEqual('hello');
    });
    it('decode two strings delimited', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([104, 101, 108, 108, 111, 0, 119, 111, 114, 108, 100]), 'utf8');
        expect(s.length).toBe(11);
        expect(s).toEqual('hello\x00world');
    });
    it('decode cyrillic', () => {
        const platform = new PlatformNode();
        const s = platform.decode(new Uint8Array([208, 151, 208, 180, 209, 128, 208, 176, 208, 178, 208, 190]), 'utf8');
        expect(s.length).toBe(6);
        expect(s).toEqual('Здраво');
    });
});
//# sourceMappingURL=data:application/json;base64,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