/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import * as tf from '../index';
import { ALL_ENVS, describeWithFlags } from '../jasmine_util';
import { expectArrayInMeanStdRange, jarqueBeraNormalityTest } from './rand_util';
describeWithFlags('randomNormal', ALL_ENVS, () => {
    const SEED = 2002;
    const EPSILON = 0.05;
    it('should return a float32 1D of random normal values', async () => {
        const SAMPLES = 10000;
        // Ensure defaults to float32.
        let result = tf.randomNormal([SAMPLES], 0, 0.5, null, SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual([SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 0.5, EPSILON);
        result = tf.randomNormal([SAMPLES], 0, 1.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual([SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 1.5, EPSILON);
    });
    it('should return a int32 1D of random normal values', async () => {
        const SAMPLES = 10000;
        const result = tf.randomNormal([SAMPLES], 0, 2, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        expect(result.shape).toEqual([SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2, EPSILON);
    });
    it('should return a float32 2D of random normal values', async () => {
        const SAMPLES = 100;
        // Ensure defaults to float32.
        let result = tf.randomNormal([SAMPLES, SAMPLES], 0, 2.5, null, SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual([SAMPLES, SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2.5, EPSILON);
        result = tf.randomNormal([SAMPLES, SAMPLES], 0, 3.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual([SAMPLES, SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 3.5, EPSILON);
    });
    it('should return a int32 2D of random normal values', async () => {
        const SAMPLES = 100;
        const result = tf.randomNormal([SAMPLES, SAMPLES], 0, 2, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        expect(result.shape).toEqual([SAMPLES, SAMPLES]);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2, EPSILON);
    });
    it('should return a float32 3D of random normal values', async () => {
        const SAMPLES_SHAPE = [20, 20, 20];
        // Ensure defaults to float32.
        let result = tf.randomNormal(SAMPLES_SHAPE, 0, 0.5, null, SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 0.5, EPSILON);
        result = tf.randomNormal(SAMPLES_SHAPE, 0, 1.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 1.5, EPSILON);
    });
    it('should return a int32 3D of random normal values', async () => {
        const SAMPLES_SHAPE = [20, 20, 20];
        const result = tf.randomNormal(SAMPLES_SHAPE, 0, 2, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2, EPSILON);
    });
    it('should return a float32 4D of random normal values', async () => {
        const SAMPLES_SHAPE = [10, 10, 10, 10];
        // Ensure defaults to float32.
        let result = tf.randomNormal(SAMPLES_SHAPE, 0, 0.5, null, SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 0.5, EPSILON);
        result = tf.randomNormal(SAMPLES_SHAPE, 0, 1.5, 'float32', SEED);
        expect(result.dtype).toBe('float32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 1.5, EPSILON);
    });
    it('should return a int32 4D of random normal values', async () => {
        const SAMPLES_SHAPE = [10, 10, 10, 10];
        const result = tf.randomNormal(SAMPLES_SHAPE, 0, 2, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2, EPSILON);
    });
    it('should return a int32 5D of random normal values', async () => {
        const SAMPLES_SHAPE = [10, 10, 10, 10, 10];
        const result = tf.randomNormal(SAMPLES_SHAPE, 0, 2, 'int32', SEED);
        expect(result.dtype).toBe('int32');
        expect(result.shape).toEqual(SAMPLES_SHAPE);
        jarqueBeraNormalityTest(await result.data());
        expectArrayInMeanStdRange(await result.data(), 0, 2, EPSILON);
    });
});
//# sourceMappingURL=data:application/json;base64,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