/**
 * @license
 * Copyright 2020 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { convertToTensor } from '../tensor_util_env';
import { add } from './add';
import { concat } from './concat';
import { matMul } from './mat_mul';
import { mul } from './mul';
import { op } from './operation';
import { sigmoid } from './sigmoid';
import { slice } from './slice';
import { tanh } from './tanh';
/**
 * Computes the next state and output of a BasicLSTMCell.
 *
 * Returns `[newC, newH]`.
 *
 * Derived from tf.contrib.rnn.BasicLSTMCell.
 *
 * @param forgetBias Forget bias for the cell.
 * @param lstmKernel The weights for the cell.
 * @param lstmBias The bias for the cell.
 * @param data The input to the cell.
 * @param c Previous cell state.
 * @param h Previous cell output.
 *
 * @doc {heading: 'Operations', subheading: 'RNN'}
 */
function basicLSTMCell_(forgetBias, lstmKernel, lstmBias, data, c, h) {
    const $forgetBias = convertToTensor(forgetBias, 'forgetBias', 'basicLSTMCell');
    const $lstmKernel = convertToTensor(lstmKernel, 'lstmKernel', 'basicLSTMCell');
    const $lstmBias = convertToTensor(lstmBias, 'lstmBias', 'basicLSTMCell');
    const $data = convertToTensor(data, 'data', 'basicLSTMCell');
    const $c = convertToTensor(c, 'c', 'basicLSTMCell');
    const $h = convertToTensor(h, 'h', 'basicLSTMCell');
    const combined = concat([$data, $h], 1);
    const weighted = matMul(combined, $lstmKernel);
    const res = add(weighted, $lstmBias);
    // i = input_gate, j = new_input, f = forget_gate, o = output_gate
    const batchSize = res.shape[0];
    const sliceCols = res.shape[1] / 4;
    const sliceSize = [batchSize, sliceCols];
    const i = slice(res, [0, 0], sliceSize);
    const j = slice(res, [0, sliceCols], sliceSize);
    const f = slice(res, [0, sliceCols * 2], sliceSize);
    const o = slice(res, [0, sliceCols * 3], sliceSize);
    const newC = add(mul(sigmoid(i), tanh(j)), mul($c, sigmoid(add($forgetBias, f))));
    const newH = mul(tanh(newC), sigmoid(o));
    return [newC, newH];
}
export const basicLSTMCell = op({ basicLSTMCell_ });
//# sourceMappingURL=data:application/json;base64,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