/**
 * @license
 * Copyright 2021 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { backend_util, buffer, DataStorage, engine, env, kernel_impls, KernelBackend, util } from '@tensorflow/tfjs-core';
const whereImpl = kernel_impls.whereImpl;
import { assertNotComplex } from './cpu_util';
export class MathBackendCPU extends KernelBackend {
    constructor() {
        super();
        this.blockSize = 48;
        this.firstUse = true;
        this.data = new DataStorage(this, engine());
    }
    nextDataId() {
        return MathBackendCPU.nextDataId++;
    }
    write(values, shape, dtype) {
        if (this.firstUse) {
            this.firstUse = false;
            if (env().get('IS_NODE')) {
                backend_util.warn('\n============================\n' +
                    'Hi there 👋. Looks like you are running TensorFlow.js in ' +
                    'Node.js. To speed things up dramatically, install our node ' +
                    'backend, which binds to TensorFlow C++, by running ' +
                    'npm i @tensorflow/tfjs-node, ' +
                    'or npm i @tensorflow/tfjs-node-gpu if you have CUDA. ' +
                    'Then call require(\'@tensorflow/tfjs-node\'); (-gpu ' +
                    'suffix for CUDA) at the start of your program. ' +
                    'Visit https://github.com/tensorflow/tfjs-node for more details.' +
                    '\n============================');
            }
        }
        const dataId = { id: this.nextDataId() };
        this.data.set(dataId, { values, dtype, refCount: 1 });
        return dataId;
    }
    /**
     * Create a data bucket in cpu backend.
     * @param shape Shape of the `TensorInfo`.
     * @param dtype DType of the `TensorInfo`.
     * @param values The value of the `TensorInfo` stored as a flattened array.
     */
    makeTensorInfo(shape, dtype, values) {
        let outId;
        if (dtype === 'string' && values != null && values.length > 0 &&
            util.isString(values[0])) {
            const encodedValues = values.map(d => util.encodeString(d));
            outId = this.write(encodedValues, shape, dtype);
        }
        else {
            outId = this.write(values, shape, dtype);
        }
        return { dataId: outId, shape, dtype };
    }
    /** Return refCount of a `TensorData`. */
    refCount(dataId) {
        if (this.data.has(dataId)) {
            const tensorData = this.data.get(dataId);
            return tensorData.refCount;
        }
        return 0;
    }
    /** Increase refCount of a `TensorData`. */
    incRef(dataId) {
        const tensorData = this.data.get(dataId);
        tensorData.refCount++;
    }
    /** Decrease refCount of a `TensorData`. */
    decRef(dataId) {
        if (this.data.has(dataId)) {
            const tensorData = this.data.get(dataId);
            tensorData.refCount--;
        }
    }
    move(dataId, values, shape, dtype, refCount) {
        this.data.set(dataId, { values, dtype, refCount });
    }
    numDataIds() {
        return this.data.numDataIds();
    }
    async read(dataId) {
        return this.readSync(dataId);
    }
    readSync(dataId) {
        const { dtype, complexTensorInfos } = this.data.get(dataId);
        if (dtype === 'complex64') {
            const realValues = this.readSync(complexTensorInfos.real.dataId);
            const imagValues = this.readSync(complexTensorInfos.imag.dataId);
            return backend_util.mergeRealAndImagArrays(realValues, imagValues);
        }
        return this.data.get(dataId).values;
    }
    bufferSync(t) {
        const data = this.readSync(t.dataId);
        let decodedData = data;
        if (t.dtype === 'string') {
            try {
                // Decode the bytes into string.
                decodedData = data.map(d => util.decodeString(d));
            }
            catch (_a) {
                throw new Error('Failed to decode encoded string bytes into utf-8');
            }
        }
        return buffer(t.shape, t.dtype, decodedData);
    }
    makeOutput(values, shape, dtype) {
        const dataId = this.write(values, shape, dtype);
        return engine().makeTensorFromDataId(dataId, shape, dtype, this);
    }
    /**
     * Dispose the memory if the dataId has 0 refCount. Return true if the memory
     * is released or memory is not managed in this backend, false if memory is
     * not cleared.
     * @param dataId
     * @oaram force Optional, remove the data regardless of refCount
     */
    disposeData(dataId, force = false) {
        if (this.data.has(dataId)) {
            this.data.get(dataId).refCount--;
            if (!force && this.data.get(dataId).refCount > 0) {
                return false;
            }
            const { complexTensorInfos } = this.data.get(dataId);
            if (complexTensorInfos != null) {
                this.disposeData(complexTensorInfos.real.dataId, true);
                this.disposeData(complexTensorInfos.imag.dataId, true);
            }
            this.data.delete(dataId);
        }
        return true;
    }
    disposeIntermediateTensorInfo(tensorInfo) {
        this.disposeData(tensorInfo.dataId);
    }
    async time(f) {
        const start = util.now();
        f();
        const kernelMs = util.now() - start;
        return { kernelMs };
    }
    memory() {
        return {
            // Unreliable due to automatic gc. The numbers above are cumulative.
            unreliable: true,
            reasons: ['The reported memory is an upper bound. Due to automatic garbage ' +
                    'collection, the true allocated memory may be less.']
        };
    }
    where(condition) {
        assertNotComplex([condition], 'where');
        const condVals = this.readSync(condition.dataId);
        return whereImpl(condition.shape, condVals);
    }
    dispose() { }
    floatPrecision() {
        return 32;
    }
    /** Returns the smallest representable number.  */
    epsilon() {
        return super.epsilon();
    }
}
MathBackendCPU.nextDataId = 0;
//# sourceMappingURL=data:application/json;base64,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