import * as React from 'react';
export var CompoundComponentContext = /*#__PURE__*/React.createContext(null);
CompoundComponentContext.displayName = 'CompoundComponentContext';
/**
 * Provides a way for a component to know about its children.
 *
 * Child components register themselves with the `useCompoundItem` hook, passing in arbitrary metadata to the parent.
 *
 * This is a more powerful altervantive to `children` traversal, as child components don't have to be placed
 * directly inside the parent component. They can be anywhere in the tree (and even rendered by other components).
 *
 * The downside is that this doesn't work with SSR as it relies on the useEffect hook.
 *
 * @ignore - internal hook.
 */
export function useCompoundParent() {
  var _React$useState = React.useState(new Map()),
    subitems = _React$useState[0],
    setSubitems = _React$useState[1];
  var subitemKeys = React.useRef(new Set());
  var deregisterItem = React.useCallback(function deregisterItem(id) {
    subitemKeys.current.delete(id);
    setSubitems(function (previousState) {
      var newState = new Map(previousState);
      newState.delete(id);
      return newState;
    });
  }, []);
  var registerItem = React.useCallback(function registerItem(id, item, missingKeyGenerator) {
    var providedOrGeneratedId;
    if (id === undefined) {
      if (missingKeyGenerator === undefined) {
        throw new Error("The compound component's child doesn't have a key. You need to provide a missingKeyGenerator to generate it.");
      }
      providedOrGeneratedId = missingKeyGenerator(subitemKeys.current);
    } else {
      providedOrGeneratedId = id;
    }
    subitemKeys.current.add(providedOrGeneratedId);
    setSubitems(function (previousState) {
      var newState = new Map(previousState);
      newState.set(providedOrGeneratedId, item);
      return newState;
    });
    return {
      id: providedOrGeneratedId,
      deregister: function deregister() {
        return deregisterItem(providedOrGeneratedId);
      }
    };
  }, [deregisterItem]);
  var getItemIndex = React.useCallback(function getItemIndex(id) {
    return Array.from(subitems.keys()).indexOf(id);
  }, [subitems]);
  return {
    contextValue: {
      getItemIndex: getItemIndex,
      registerItem: registerItem,
      totalSubitemCount: subitems.size
    },
    subitems: subitems
  };
}