import { __assign, __awaiter, __generator, __read } from "tslib";
import { getGlobalScope } from './global-scope';
import { UUID } from './utils/uuid';
import { NetworkRequestEvent, RequestWrapperFetch, ResponseWrapperFetch, RequestWrapperXhr, ResponseWrapperXhr, } from './network-request-event';
/**
 * Typeguard function checks if an input is a Request object.
 */
function isRequest(requestInfo) {
    return typeof requestInfo === 'object' && requestInfo !== null && 'url' in requestInfo && 'method' in requestInfo;
}
var NetworkEventCallback = /** @class */ (function () {
    function NetworkEventCallback(callback, id) {
        if (id === void 0) { id = UUID(); }
        this.callback = callback;
        this.id = id;
    }
    return NetworkEventCallback;
}());
export { NetworkEventCallback };
var NetworkObserver = /** @class */ (function () {
    function NetworkObserver(logger) {
        this.eventCallbacks = new Map();
        this.isObserving = false;
        this.logger = logger;
        var globalScope = getGlobalScope();
        if (!NetworkObserver.isSupported()) {
            /* istanbul ignore next */
            return;
        }
        this.globalScope = globalScope;
    }
    NetworkObserver.isSupported = function () {
        var globalScope = getGlobalScope();
        return !!globalScope && !!globalScope.fetch;
    };
    NetworkObserver.prototype.subscribe = function (eventCallback, logger) {
        var _a, _b, _c, _d, _e, _f, _g;
        if (!this.logger) {
            this.logger = logger;
        }
        this.eventCallbacks.set(eventCallback.id, eventCallback);
        if (!this.isObserving) {
            /* istanbul ignore next */
            // eslint-disable-next-line @typescript-eslint/unbound-method
            var originalXhrOpen = (_c = (_b = (_a = this.globalScope) === null || _a === void 0 ? void 0 : _a.XMLHttpRequest) === null || _b === void 0 ? void 0 : _b.prototype) === null || _c === void 0 ? void 0 : _c.open;
            /* istanbul ignore next */
            // eslint-disable-next-line @typescript-eslint/unbound-method
            var originalXhrSend = (_f = (_e = (_d = this.globalScope) === null || _d === void 0 ? void 0 : _d.XMLHttpRequest) === null || _e === void 0 ? void 0 : _e.prototype) === null || _f === void 0 ? void 0 : _f.send;
            if (originalXhrOpen && originalXhrSend) {
                this.observeXhr(originalXhrOpen, originalXhrSend);
            }
            /* istanbul ignore next */
            var originalFetch = (_g = this.globalScope) === null || _g === void 0 ? void 0 : _g.fetch;
            /* istanbul ignore next */
            if (originalFetch) {
                this.observeFetch(originalFetch);
            }
            /* istanbul ignore next */
            this.isObserving = true;
        }
    };
    NetworkObserver.prototype.unsubscribe = function (eventCallback) {
        this.eventCallbacks.delete(eventCallback.id);
    };
    NetworkObserver.prototype.triggerEventCallbacks = function (event) {
        var _this = this;
        this.eventCallbacks.forEach(function (callback) {
            var _a;
            try {
                callback.callback(event);
            }
            catch (err) {
                // if the callback throws an error, we should catch it
                // to avoid breaking the fetch promise chain
                /* istanbul ignore next */
                (_a = _this.logger) === null || _a === void 0 ? void 0 : _a.debug('an unexpected error occurred while triggering event callbacks', err);
            }
        });
    };
    NetworkObserver.prototype.handleNetworkRequestEvent = function (requestType, requestInfo, requestWrapper, responseWrapper, typedError, startTime, durationStart) {
        var _a;
        /* istanbul ignore next */
        if (startTime === undefined || durationStart === undefined) {
            // if we reach this point, it means that the performance API is not supported
            // so we can't construct a NetworkRequestEvent
            return;
        }
        // parse the URL and Method
        var url;
        var method = 'GET';
        if (isRequest(requestInfo)) {
            url = requestInfo['url'];
            method = requestInfo['method'];
        }
        else {
            url = (_a = requestInfo === null || requestInfo === void 0 ? void 0 : requestInfo.toString) === null || _a === void 0 ? void 0 : _a.call(requestInfo);
        }
        method = (requestWrapper === null || requestWrapper === void 0 ? void 0 : requestWrapper.method) || method;
        var status, error;
        if (responseWrapper) {
            status = responseWrapper.status;
        }
        if (typedError) {
            error = {
                name: typedError.name || 'UnknownError',
                message: typedError.message || 'An unknown error occurred',
            };
            status = 0;
        }
        var duration = Math.floor(performance.now() - durationStart);
        var endTime = Math.floor(startTime + duration);
        var requestEvent = new NetworkRequestEvent(requestType, method, startTime, // timestamp and startTime are aliases
        startTime, url, requestWrapper, status, duration, responseWrapper, error, endTime);
        this.triggerEventCallbacks(requestEvent);
    };
    NetworkObserver.prototype.getTimestamps = function () {
        var _a, _b;
        /* istanbul ignore next */
        return {
            startTime: (_a = Date.now) === null || _a === void 0 ? void 0 : _a.call(Date),
            durationStart: (_b = performance === null || performance === void 0 ? void 0 : performance.now) === null || _b === void 0 ? void 0 : _b.call(performance),
        };
    };
    NetworkObserver.prototype.observeFetch = function (originalFetch) {
        var _this = this;
        /* istanbul ignore next */
        if (!this.globalScope || !originalFetch) {
            return;
        }
        /**
         * IMPORTANT: This overrides window.fetch in browsers.
         * You probably never need to make changes to this function.
         * If you do, please be careful to preserve the original functionality of fetch
         * and make sure another developer who is an expert reviews this change throughly
         */
        this.globalScope.fetch = function (requestInfo, requestInit) { return __awaiter(_this, void 0, void 0, function () {
            var timestamps, originalResponse, originalError, err_1;
            var _a, _b;
            return __generator(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        try {
                            timestamps = this.getTimestamps();
                        }
                        catch (error) {
                            /* istanbul ignore next */
                            (_a = this.logger) === null || _a === void 0 ? void 0 : _a.debug('an unexpected error occurred while retrieving timestamps', error);
                        }
                        _c.label = 1;
                    case 1:
                        _c.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, originalFetch(requestInfo, requestInit)];
                    case 2:
                        originalResponse = _c.sent();
                        return [3 /*break*/, 4];
                    case 3:
                        err_1 = _c.sent();
                        // Capture error information
                        originalError = err_1;
                        return [3 /*break*/, 4];
                    case 4:
                        // 3. call the handler after the fetch call is done
                        try {
                            this.handleNetworkRequestEvent('fetch', requestInfo, requestInit ? new RequestWrapperFetch(requestInit) : undefined, originalResponse ? new ResponseWrapperFetch(originalResponse) : undefined, originalError, 
                            /* istanbul ignore next */
                            timestamps === null || timestamps === void 0 ? void 0 : timestamps.startTime, 
                            /* istanbul ignore next */
                            timestamps === null || timestamps === void 0 ? void 0 : timestamps.durationStart);
                        }
                        catch (err) {
                            // this catch shouldn't be reachable, but keep it here for safety
                            // because we're overriding the fetch function and better to be safe than sorry
                            /* istanbul ignore next */
                            (_b = this.logger) === null || _b === void 0 ? void 0 : _b.debug('an unexpected error occurred while handling fetch', err);
                        }
                        // 4. return the original response or throw the original error
                        if (originalResponse) {
                            // if the response is not undefined, return it
                            return [2 /*return*/, originalResponse];
                        }
                        else {
                            throw originalError;
                        }
                        return [2 /*return*/];
                }
            });
        }); };
    };
    NetworkObserver.prototype.observeXhr = function (originalXhrOpen, originalXhrSend) {
        /* istanbul ignore next */
        if (!this.globalScope || !originalXhrOpen || !originalXhrSend) {
            return;
        }
        var xhrProto = this.globalScope.XMLHttpRequest.prototype;
        var networkObserverContext = this;
        /**
         * IMPORTANT: This overrides window.XMLHttpRequest.prototype.open
         * You probably never need to make changes to this function.
         * If you do, please be careful to preserve the original functionality of xhr.open
         * and make sure another developer who is an expert reviews this change throughly
         */
        xhrProto.open = function () {
            var _a, _b;
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i] = arguments[_i];
            }
            var xhrSafe = this;
            var _c = __read(args, 2), method = _c[0], url = _c[1];
            try {
                /* istanbul ignore next */
                xhrSafe.$$AmplitudeAnalyticsEvent = __assign({ method: method, url: (_a = url === null || url === void 0 ? void 0 : url.toString) === null || _a === void 0 ? void 0 : _a.call(url) }, networkObserverContext.getTimestamps());
            }
            catch (err) {
                /* istanbul ignore next */
                (_b = networkObserverContext.logger) === null || _b === void 0 ? void 0 : _b.debug('an unexpected error occurred while calling xhr open', err);
            }
            // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
            return originalXhrOpen.apply(xhrSafe, args);
        };
        /**
         * IMPORTANT: This overrides window.XMLHttpRequest.prototype.send
         * You probably never need to make changes to this function.
         * If you do, please be careful to preserve the original functionality of xhr.send
         * and make sure another developer who is an expert reviews this change throughly
         */
        // allow "any" type for args to reflect how it's used in the browser
        /* eslint-disable-next-line @typescript-eslint/no-unsafe-argument */
        xhrProto.send = function () {
            var args = [];
            for (var _i = 0; _i < arguments.length; _i++) {
                args[_i] = arguments[_i];
            }
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            var xhrSafe = this;
            var body = args[0];
            var requestEvent = xhrSafe.$$AmplitudeAnalyticsEvent;
            xhrSafe.addEventListener('loadend', function () {
                var _a;
                try {
                    var responseHeaders = xhrSafe.getAllResponseHeaders();
                    var responseBodySize = xhrSafe.getResponseHeader('content-length');
                    var responseWrapper = new ResponseWrapperXhr(xhrSafe.status, responseHeaders, 
                    /* istanbul ignore next */
                    responseBodySize ? parseInt(responseBodySize, 10) : undefined);
                    var requestWrapper = new RequestWrapperXhr(body);
                    requestEvent.status = xhrSafe.status;
                    networkObserverContext.handleNetworkRequestEvent('xhr', { url: requestEvent.url, method: requestEvent.method }, requestWrapper, responseWrapper, undefined, requestEvent.startTime, requestEvent.durationStart);
                }
                catch (err) {
                    /* istanbul ignore next */
                    (_a = networkObserverContext.logger) === null || _a === void 0 ? void 0 : _a.debug('an unexpected error occurred while handling xhr send', err);
                }
            });
            /* eslint-disable-next-line @typescript-eslint/no-unsafe-argument */
            return originalXhrSend.apply(xhrSafe, args);
        };
    };
    return NetworkObserver;
}());
export { NetworkObserver };
// singleton instance of NetworkObserver
export var networkObserver = new NetworkObserver();
//# sourceMappingURL=network-observer.js.map