import { ActionType, Messenger } from './element-interactions';
/**
 * Configuration options for dead clicks tracking
 */
export interface DeadClickOptions {
    /**
     * CSS selectors to define which elements on the page to track for dead clicks.
     * A dead click is a click that doesn't result in any visible change or navigation.
     */
    cssSelectorAllowlist?: string[];
}
/**
 * Configuration options for rage clicks tracking
 */
export interface RageClickOptions {
    /**
     * CSS selectors to define which elements on the page to track for rage clicks.
     * A rage click is multiple rapid clicks on the same element within a 3s time window.
     */
    cssSelectorAllowlist?: string[];
}
/**
 * Configuration options for frustration interactions tracking.
 * This includes dead clicks and rage clicks tracking.
 */
export interface FrustrationInteractionsOptions {
    /**
     * List of page URLs to allow auto tracking on.
     * When provided, only allow tracking on these URLs.
     * Both full URLs and regex are supported.
     */
    pageUrlAllowlist?: (string | RegExp)[];
    /**
     * Function to determine whether an event should be tracked.
     * When provided, this function overwrites all other allowlists and configurations.
     * If the function returns true, the event will be tracked.
     * If the function returns false, the event will not be tracked.
     * @param actionType - The type of action that triggered the event.
     * @param element - The [Element](https://developer.mozilla.org/en-US/docs/Web/API/Element) that triggered the event.
     */
    shouldTrackEventResolver?: (actionType: ActionType, element: DomElement) => boolean;
    /**
     * Prefix for data attributes to allow auto collecting.
     * Default is 'data-amp-track-'.
     */
    dataAttributePrefix?: string;
    /**
     * Options for integrating visual tagging selector.
     */
    visualTaggingOptions?: {
        enabled?: boolean;
        messenger?: Messenger;
    };
    /**
     * Configuration for dead clicks tracking
     */
    deadClicks?: DeadClickOptions;
    /**
     * Configuration for rage clicks tracking
     */
    rageClicks?: RageClickOptions;
}
/**
 * Default CSS selectors for dead clicks tracking
 */
export declare const DEFAULT_DEAD_CLICK_ALLOWLIST: string[];
/**
 * Default CSS selectors for rage clicks tracking
 */
export declare const DEFAULT_RAGE_CLICK_ALLOWLIST: string[];
/**
 * Default time window for dead clicks (3 seconds)
 */
export declare const DEFAULT_DEAD_CLICK_WINDOW_MS = 3000;
/**
 * Default time window for rage clicks (3 seconds)
 */
export declare const DEFAULT_RAGE_CLICK_WINDOW_MS = 3000;
/**
 * Default threshold for rage clicks (5 clicks)
 */
export declare const DEFAULT_RAGE_CLICK_THRESHOLD = 5;
type DomElement = typeof globalThis extends {
    Element: new (...args: any) => infer T;
} ? T : never;
export {};
//# sourceMappingURL=frustration-interactions.d.ts.map