import 'dart:typed_data';

import 'package:test/test.dart';
import 'package:vcmrtd/extensions.dart';
import 'package:vcmrtd/vcmrtd.dart';

void main() {
  test("regular ascii dg1", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG1(parseTestCase(dg1Test));
  });

  test("dg1 with diacritics", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG1(parseTestCase(dg1DiacriticsTest));
  });

  test("dg5", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG5(parseTestCase(dg5Test));
  });

  test("dg6", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG6(parseTestCase(dg6Test));
  });

  test("dg12", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG12(parseTestCase(dg12Test));
  });

  test("dg13", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);
    parser.parseDG13(parseTestCase(dg13Test));
  });

  test("parse all", () {
    final parser = DrivingLicenceParser(failDg1CategoriesGracefully: false);

    parser.parseDG1(parseTestCase(dg1DiacriticsTest));
    parser.parseDG5(parseTestCase(dg5Test));
    parser.parseDG6(parseTestCase(dg6Test));
    parser.parseDG11(Uint8List(0));
    parser.parseDG12(parseTestCase(dg12Test));
    parser.parseDG13(parseTestCase(dg13Test));

    final document = parser.createDocument();

    expect(document.documentNumber, equals("1234567890"));
    expect(document.holderSurname, equals("Bâssié"));
    expect(document.holderOtherName, equals("Daniël"));
    expect(document.placeOfBirth, equals("Méppel"));
    expect(document.dateOfBirth, equals("17121996"));
    expect(document.issuingAuthority, equals("Gemeente Meppèl"));
    expect(document.issuingMemberState, equals("NLD"));
    expect(document.dateOfIssue, equals("12072017"));
    expect(document.dateOfExpiry, equals("12072027"));

    expect(document.photoImageData, isNotEmpty);
    expect(document.photoImageType, ImageType.jpeg);
  });
}

String removeWhitespace(String input) {
  return input.replaceAll(RegExp(r'\s+'), '');
}

Uint8List parseTestCase(String test) {
  return removeWhitespace(test).parseHex();
}

// Surname:				Bassie
// First name:			Barrie
// Birth day:			17.12.1996
// Birth place:			Meppel
// Date of issue:		12.07.2017
// Date of expiry:		12.07.2027
// Issuing place:		Gemeente Meppel
// Document number:		1234567890
const dg1Test = """
		61 818f
			5f01 0d 65342d444c3030203030303031
			5f02 55
				5f03 03 4e4c44
				5f04 06 426173736965
				5f05 06 426172726965
				5f06 04 17121996
				5f07 06 4d657070656c
				5f0a 04 12072017
				5f0b 04 12072027
				5f0c 0f 47656d65656e7465204d657070656c
				5f0e 0a 31323334353637383930
			7f63 24
				02 01 02
				87 0f 414d3b140720173b140720273b3b3b
				87 0e 423b140720173b140720273b3b3b
""";

// Surname:				Bâssié
// First name:			Daniël
// Birth day:			17.12.1996
// Birth place:			Méppel
// Date of issue:		12.07.2017
// Date of expiry:		12.07.2027
// Issuing place:		Gemeente Meppèl
// Document number:		1234567890
const dg1DiacriticsTest = """
		61 818f
			5f01 0d 65342d444c3030203030303031
			5f02 55
				5f03 03 4e4c44
				5f04 06 42e2737369e9
				5f05 06 44616e69eb6c
				5f06 04 17121996
				5f07 06 4de97070656c
				5f0a 04 12072017
				5f0b 04 12072027
				5f0c 0f 47656d65656e7465204d657070e86c
				5f0e 0a 31323334353637383930
			7f63 24
				02 01 02
				87 0f 414d3b140720173b140720273b3b3b
				87 0e 423b140720173b140720273b3b3b
""";

const dg5Test = """
		67 820637
			89 01 03
			5f43 82062f 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
""";

const dg6Test = """
		75 824526
			7f61 824521
				02 01 01
				7f60 824519
					a1 0f
						80 02 0101
						81 01 02
						87 02 0101
						88 02 0008
					5f2e 824503
						46414300 
						30313000
						00004f73
						0001
						00004f65
						0000
						00
						00
						00
						000000
						0000
						000000
						000000
						00
						00
						0151
						01c1
						00
						00
						0000
						0000
						ffd8ffe10ffe4578696600004d4d002a000000080006011200030000000100010000011a00050000000100000056011b0005000000010000005e01280003000000010002000002130003000000010001000087690004000000010000006600000000000000600000000100000060000000010007900000070000000430323231910100070000000401020300a00000070000000430313030a00100030000000100010000a00200040000000100000151a003000400000001000001c1a4060003000000010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000ffdb008400010101010101020101020302020203040303030304050404040404050605050505050506060606060606060707070707070808080808090909090909090909090101010102020204020204090605060909090909090909090909090909090909090909090909090909090909090909090909090909090909090909090909090909ffdd00040016ffc000110801c1015103012200021101031101ffc401a20000010501010101010100000000000000000102030405060708090a0b100002010303020403050504040000017d01020300041105122131410613516107227114328191a1082342b1c11552d1f02433627282090a161718191a25262728292a3435363738393a434445464748494a535455565758595a636465666768696a737475767778797a838485868788898a92939495969798999aa2a3a4a5a6a7a8a9aab2b3b4b5b6b7b8b9bac2c3c4c5c6c7c8c9cad2d3d4d5d6d7d8d9dae1e2e3e4e5e6e7e8e9eaf1f2f3f4f5f6f7f8f9fa0100030101010101010101010000000000000102030405060708090a0b1100020102040403040705040400010277000102031104052131061241510761711322328108144291a1b1c109233352f0156272d10a162434e125f11718191a262728292a35363738393a434445464748494a535455565758595a636465666768696a737475767778797a82838485868788898a92939495969798999aa2a3a4a5a6a7a8a9aab2b3b4b5b6b7b8b9bac2c3c4c5c6c7c8c9cad2d3d4d5d6d7d8d9dae2e3e4e5e6e7e8e9eaf2f3f4f5f6f7f8f9faffda000c03010002110311003f00fec86683f7df68f2ff00d4d737793dbfeeff0079fbc96ba4bcfb3ffc7c7fcb2ae4af3ecfe4db79127fdb4afb3c29ec1cdde5f5b89becf3c95a567f67ff008f7ac4d4a0ff004b8fc9d92f9b459c1710c3255627f8c71617f8e747f6dd4209becffeb63aaf34f717de609e4a26ff005df68ffc8759b9ff00978f2e97ef0ed347edb710ff00ed3f9ea09b55b89e1fecfb893fe5a7f7e93cff003e1fe08ab366f3fc9ffae55c601e779137d9eb3669fc887fd5d027fdf7d9e78ffe59d56f3ee2687ecff674abc3ff00040219e09e18bf78915559a0b7f3a4fde252f9fe443f689ff7b1c3ff0090eb92d48dfcfe55bc170915b799fbcf33fd6f955ad80e9269fecf37ee24496b36f3fe7e3ccae6759f1b785fc1fa2cfa9eb1345a6d9daa19259d982468abd4b31c00057f2cbff0512ff8392be13fc16d56efc05fb1b2a78d7c450ee8a5d4e627fb26ddc71fbb1c19c8f6f97b571d4a9182f78894d23fa60f89ff001abe147c1cf07dc78dfe2c788b4ef0fe9d6ac5e496f99624014738ddd7e82bf9a3fdb3bfe0e7efd9f3e1a4d79e14fd927449bc73a92e506a5779b6d3d48e32bff2d241e840c1afe34ff69dfdb2ff00693fdb0fc65378dbf681f14ddeb93c8db92dd9ca5ac3e8b140bf2281d0719c77af97ebcb78f71d29e871cb1737b9fa93fb52ff00c1643f6fbfdac2e2583c69e349b46d2a4c85d3b441f61802f604c7f3920719ddf857e615fea17faade49a86a93c9737129cbc92b17763ea58f26a9d15c752aca6ef26730514515980514514005145140051451401f4dfc0bfdb33f6a5fd9aefe3bef827e39d5b4158f1fe8f0dc335b103f84c0fba3c7fc06bfa18fd943fe0e7af8b9e15487c2ff00b5878662f1059b18d5b53d25becd70a17805a0398df03ae0827b0afe5328ade8e2670f859a42aca3f0b3fd543f652ff829cfec83fb65785e46f831e2ab57d42340dfd9d74c2dee573d9a07c37e59afbc342f115bdf7fa44fff00a1d7f8ebe81e21d7bc2baac3aef862f67d3af6dce639eda468a443ecc84115fbf3fb12ff00c1c11fb49fc08bab6f0b7ed00d2f8e3c3e008cceed8bd8874df93f2ca40f5c1c57b787cda9bd2a2b050934bd9df43fd12a6d52de0f2bf789e54b563c8fdec9e46cafc2df82ff00f0549f831f19fc15078d7c07aedb6a1079b0b88dfe49ed1fbac919e548fcabf4eacfe3be8f36a56da841a8412db5df95ff001ef5dd069ea8f4d1f517ee3ed9f67f9289a7b7821fb3d56867f3f52fb7ff00cbb431d27fa3cf0ffdb4fefd76e271071e1f73a5860b8f27ed1ff2caab43fbf871047e549fee556f3ee3c9ff0059feb68b39ff007bfbf93f770ffb1563fac1a50cd7104b17eeff0079e5ff0072acf9f6f3ff00a4797feabfe59c8955a5fdf4d17fcb2a2ce0fdcf9e280c3079171ff1f13ffe8144d079ff00e91f251e44fe77ff006149fe91f6bfdc3ffa9ff6283b05867ff977823ff5547fa389a4b7a3c8f226fe0ab3f67f23ccb89f64b41c3d4e67c8b7ff009e8fff009168f22dff00e7a3ff00e45adcfb35bffcf4ff00c728fb35bffcf4ff00c72bc0b0f9cfffd0fec3eee61e4fd9e791e5ae72effd4c55d5de7d9fcdfdc47e57fc0eb93d460fdec7e7feeabec51ec1c95e4f71e77eff007f975674d9edfc9fb44f1d1a8c17104df67f33fed9d16705c410fdff00ddd6989fe31c585fe39b52fefe6fb3c1ff003ceb3a1fb44f0ffabff535661fdf5e7d9e7ff59ff2ceab09f5082193f7759dce9c400fb3cd0fda3cc4aade7e9fff001ef049feb689a6ff0055e7c955a6fb3f93ff003cbfe7a5730e8157fd1fed7f68f32b36f27f221f23cba59a79fcdfb47da2b1357bdfb0d9ff00684dff00ef2bad7f08d0b335f79f0fd9e08fccff009e91d7c29fb6f7fc1407f677fd82be17ddf8f3e346ad1d9dc4d115b0b08dc497979263848a21cfd58f02be11ff0082a4ff00c16afe087ec0be0d93c23e0a78fc47f122ea23f63d315f3f66c8c2cd76e3ee28ecbd5bd2bfcf17f693fda73e357ed69f146fbe2f7c74d6e6d6757bd6246f27ca8109e22863e888bd001f8d7263731f62f961b9c95eaf2cee8fbcbfe0a23ff0587fda73f6f5d76e743bad426f0df8195dfecda259c8503a13c1ba75c195b1d47dd1e95f91f4515f3756b4a6f9a6ce0514b441451456630a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803bbf87bf133c73f0af5d1e22f01ea32e9f738dafe5b61644fee3af465f635fbf3fb117fc154752bed5347f06f8d6ecd86a906228da67dd6d70c3022033f7483d8fe15fce5d39599183a1c11d08ed5d386c5ce93f70a84ac7fad2fc0cfda174ef8c7f0dac75183647712db432c9246fff002c6beb4f0dcfe7e9bfda1e67eefccaff0039ff00f82547fc15e7c51fb38ebaff0008be396a12dd7863568c5ac17f2fef1ac9ba2efcf58fb67a8fa57fa0b7c1cf881a178a7e12687e21d1ae61bcd3ef143c779e602b329e8411c62bebf058d55cb9c6318f344f52866f3e1ffc8556a182de797f71589363c98ee3fd5452fef6b4acfcf825ff00a695d06b86fe09a30ffadff5692ffcb2ad3860acd86fae3cdfdc559b39edff0075713efa0e7c36e6dc33dbc3fbf9e4f36ac79f6f3cd1e9ff00eaab1441fb9fe3fded16707efa4fde5075e1cd29ac74ff0037f71feb21a2f20b883ccf2364b55a183f7dfc7e5d599bfe5a7eeff7741c7d4cff00b3dcff00b1ff007c51f67b9ff63fef8abffd967fe79ffe3f47f659ff009e7ff8fd7cf9afb489ffd1fec3ef27c4325bc11ffd73ae4aeefae27f2c7975d27d9eff00c9ff004ed92c9e5d626a5fea62f3ff00e58d7d8a3d8392d4bf7f79f68823a2ce0fdcfda3fcc959b79f67fb67d9e0ab3a6c16fe6fee3ff40ff595a627f8c71617f8e6dc305c79df688244f33fe99d5687ed104325c7fd34ab32fee26927aade7f9ffe8f3efac8e97882b79f71343f67ff0055ff003d2b37c8f221fb44ff00fa1d6dde58dbf9d1f93715cdde5bdc5bc3fb9d9e67fd34ac3103c3189a94ff00f3c24ff55ff2cf7d7f331ff0593ff82d6f86bf64ad22f3e0cfc19b88f52f891751988796fbe0d3e2618f3a5917967fee47f9f15ef1ff0005a3ff0082b168dfb057c2f97c0be0fbb86f7e24f8821962d3ece370f1da46df29ba954740bfc00f535fe719e32f18f89fe20f8a6fbc6be33bd9350d535299a7b9b898ee77763924ff0041d00e057362f1dece1eca1b9c53fddcf425f1c78e3c5bf127c597fe39f1dea136a9ab6a72b4f73733b6e791dba927f901c01c0e2b94a28af9f6ce70a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800afe847fe08fdff000593f19fec9faf699fb3efc72bf92ffe1ddd5c2a5acf2b92da4bb9c641ff009e19fbcbfc3d471d3f9eea2b7c362654a6a7003fd8cfc2be33f0b78bbc1d63ad7876f63d4ad6f6259927b560d1f96c32aca471822b5b4dbeb89a6fdc46f14517fabafe077fe085bff057c97f677f19e9bfb2f7ed2fabc87c097b294d22fe63bffb2ee24e16324f481cf1fec1f6e9fdede87e26d1fc536d1de787a7596d6550c93a9cab29190411c608afaca78b8d68f32316ac7436535c41fea3f7b255ab39ee2087ed13c8955bc8d4209a3d43fd57fd34a2ce0b8ff96f27fd73aed3a30db9d2433ffcbbcd1d6979f6ff00f1ef3c692ffc02b107ee0497107fcf4ff96947fcb6ff0057feb681366dff00c4be087fd5feeffdcacdf3bcf864f224ab3fe9107963fd6ffd33aade47fada041f6fbfff009e6fff009168fb7dff00fcf37ffc8b567ecfff004cff00f1ca3ecfff004cff00f1caf9f03fffd2fec826fdfc325bcd5c55dc3e4431dbc1fbaadbbc9fcf9bed107fe815897904f3f97e449fbc87fd8afa947adfb9388d4a09ff00b465b7ad3d37fb43c9fb479759b79f688351fdfec8a3ab3677d71ff1ef0475d38afdd563970fb9b7fe913fefe7ff005759b37fa992e20ff9e9479f713ffcb3f37c9aad2fda27864fbf5276199f67fdcfefa47f32bf383fe0a61fb7f7c38ff827afece57bf16fc6b3452eb32446db44d2d48f36f2f187caa07645eae7a002beecf8bbf133c05f04fe1b6b1f147e22df0d3749d0ec64bbbbb89485548e3193f89e8057f95eff00c155ff00e0a2fe3bff00828e7ed337ff0012f5596583c2fa63359f87f4f6276c36aa7fd615ff009e92fde63f41dabcdc5623922635aa72a3e32fda13e3efc4bfda77e2feb5f1b7e2dea0fa8eb7adced34aec4958d7f8228c7f0c718f955476af16a28af09bbbbb3cd0a28a2900514514005145140051451400514514005145140051451400514514005145140051451400514514005145140057f6c9ff0006f77fc15cdfc61a3d9fec23fb41ea28357b54dbe16d52e98032c6a3fe3d2490f5751feaf3d471dabf89badaf0df88f5cf086bf65e29f0c5d4963a8e9d325c5b5c4476bc72c6432b29f50457461b10e9cae889c1347fb17cd3ea10795a7cfbe4f27fe7a56959fefe18fcf93fe5a57e2fff00c11bbfe0a51e1cff0082827ecf908f17dd471fc45f0ac71da6ad64bf299b03097483bac98e71d1b8afd8eb39adfec917915f694aa29c54a265878db43a48479ffdf96b6bcfb8f3bed1e5d62d9fda3caffda9566ce7b8f36b43abeaccdb867b8cf9f3ff00acaabe779134971f3f95fee52e2e3ce8ede7928f22e3c99279bfd650727d5cb5fb8ff9e727fdf147ee3fe79c9ff7c555f22e7fe79d1e45cffcf3af0fde37e447ffd3fec1af3f7f0ff07fd34f2eb9bbcfb3c1e5ff00cb58ebadbc83cff33fe597fd34ae6f52fb4412db791fbdafb147b079bebd3f9fa97f1d6968f6371e4ffacacdd4a7b78352fdff00fe44ad2d37f7f2ff00aca799ff00191c787dcd29a0f22593c87f2ab37cfb8821fb44d23c5567fe5b49e748f5f96fff00056bfdbf347ff827dfec81af7c5179925f13ea6a74dd06dd8f325e4ab80f8feec43e76f4c5672928abb37fac9fcc97fc1cc9ff00053e6f1af8a07ec11f06b546934cd25d67f145c44fc4b72398ed32bd563fbce3d702bf8f6ae87c59e2af1078e7c4fa878cfc5774f7ba9ea97125d5d4f21cb492cac5998fd49ae7abe5ebd67397333cf9cf99dc28a28ac480a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803ed5fd803f6cef1d7ec25fb4be85f1d7c1ceef6d6d208353b30485bab27204b191d32072be8457fa917c13f8b7e0ef8f1f0bf46f8c7f0defd752d17c41691de5b48b27f0c833b4e3a15e847a8aff208afeba7fe0da9ff00828a49e1af115d7ec17f142f7fd035667bcf0c492b7115cf59ad467b38f9907ae6bd9ca317cb2f66f662b1fdbd43f68f27f7156619fc8e3cbf36b36ce6b8821ff9eb27fe8bab50ff00a9ff0057ff002d2be9cab9d279f6de77fabf2a8867b783fe59f9b2556845bc137eff00fe58ff00cb3d944335bcd0c9e747feaa4fee507322cf9f3fabd1e7cfeaf47fdf547fdf55e39d1ed4ffd4fec1af3ecf3ffcb34ffbe3fd65626a507faaff00d175b737ee0493cffba93fdcac4bbfdf4b17dc97c9afb147b0701a90ff008997d9e6d9ff007c5167f67ff8f883ff0022559d4becff00da325bc31d56b382e3cec797ff005ce9e65fc6471e1f726babb8f732de2081106494e0002bfcc8ff00e0bf3fb7fbfeda5fb66ea1e13f06de19bc13e00924d2f4d0bc24d70a71737181c72c36afa01e86bfb4dff82ebfedce9fb127ec37af6aba35e04f17f8cf768fa30c8dcaf3a912cc07a451e4d7f970cd34d73335c5c31791c9666639249ea49af1b33aff00f2ed06267f6511514515e31c6145145001451450014514500145145001451450014514500145145001451450014514500145145001451450014514500145145001451450015d8fc3df1ef8a3e16f8e748f88de0ab96b3d5b44ba8af2d2643829244c197a638e304771c571d45080ff0057aff827b7ed6fe1efdb67f653f0bfed05a2dc27da6f6dc41a940c41305f42364e87fe04323dabeeaf3ee3cd92debf812ff8369bf6e697e0d7ed0b7dfb2778c2f3cad0bc7c44ba76f6f961d4e15e00cf4f3538c7a8afefb6082e04de7f99ff002d2bed30388f694d480dbff9e7fbb4a4960b7f27f71ff3d3f7955e1fb44137da3ff64a21bef22293f77fbcf32ba7db9ce69fd9ee68fb3dcd56f3edbfe7a51e7db7fcf4af06f03ab999ffd5fec62f20b7f2a4b79ab89d4a0b783cbb8f2d2ba4bcfb44137d9bcbf37fe99d66ea53dc79d179d1a4be5475f528f60f2ed5fed1f6b96e2acd9ff684f0c7fbcaada8cfe7ea525c7975f207fc1427f6a7f0e7ec75fb16f8dfe396a804373a4e9b2a58267064bc997cb8117df71c8fa574668d2ab738a83b1fc16ffc1c7dfb6dcbfb537edd577f0cfc3975e6f867e1aa1d2a0546cc6f7879b9906383838507b608afe7b6b6bc49e20d5bc5be21bff156bf299efb52b896eae243d5e5998bbb7e2c4d62d7c8d49f33b9cd395ddc28a28a82428a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a00ec7e1ef8ebc43f0c7c75a3fc44f094c6df53d12f21bdb6914e31242e1d7a76e307db8aff585fd8abf68dd17f6b2fd97bc1df1ff00c37701d35fb38a49d063315caa859a3603a157078aff00249afed3ff00e0d5cfdada3bdd13c65fb1af8a2e70d667fb7746563fc0ff002dcc683d9b0e7eb5ebe4f5f96a727703fb2787f71febe4497f7755a1ff009f8f2ffe5a559867b78218ff00eb9d10cfe7c31fef3f792ffb75f4bf58034fcf9ffcbd1e7cff00e5eaaffa3ffcfc251fe8ff00f3f095e19a7bc7ffd6fec626fb4410ff00ac4f32b9bbc9f1347713ff00cb5aeb66ff005325bffcf5ac4bcfb3c12c7fbb7ff575f628f60f2dd620b7fed2fdc6ff00ded7f17bff000761fed472e9be1bf017ec85a1deef6d41db5ed5510f1b22fdddba38ff0078ef5fa57f671ad34b1dfcaec76aa8c927b015fe51bff0588fda4e5fda9bfe0a1ff117e2243706e34eb2bf6d234f39cedb6b1fdd01ff007d86e95cbc412b4ac795376933f31e8a28af982028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bef1ff8266fed297bfb277edbbe01f8c114863b383518ecefd41c07b4ba222915bfd9190c7fddaf83a94120e4718aa849c5a6811fecb1a55ee9ba9d8c3ad583892d2e23592365e8cac32a47b115a9ff002c63fde7fadafcc2ff0082377ed2b0fed31ff04fbf87fe3cbe9fed1a9d869ffd91a8377fb458fee9bf40315fa7b0cff67863f23fe7a57d6d397359a3a11a3fbeff006e8fdf7fb74be79ff6ff00efba3cf3fedffdf75c02f68bf94fffd7fec82f27b79ffd1e093f79e6571579f689e5fe3f2abb6bbff9693f97fbcae4aee0d4209a3b79f67fabafa947b07e777fc1497e3d43fb37fec75f12fe3533849749d12e7ece3a1334a9e5c617df2dc7d2bfc89f50bfbcd56fe6d4f5190cd7173234b2bb7567739627dc9aff0042eff83acbe3b4de00fd89b41f8396b26dbbf1b6b71acc14e184164be71cff00b2c70a6bfcf1ab8b88277afcbd8f218514515e18828a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a00fed2ffe0d40f8f53df587c47fd99efe7cfd99a0d72c918f457fdccaa83ea371afecaece0b89e1ff0059ff008fd7f9937fc1033e3bdc7c0eff008296f8291a531d9f8a84fa15c64e131729952df46418fad7fa6cc3f6883caf3e4ff5b279bf7ebebb25a97a76ec6d0d877d8adffe79a51f62b7ff009e6957fced3ffe7a3d1e769fff003d1eb9f428ffd0fec5e5fb47ef27f2fcdff805675e41a84de5f9f1a7fabae92ee7f3a193c88ff79ff4cd2b9bbc9879d17ef2bea51ec1fe76dff07697c661e29fdb37c23f056d1cac7e14d0fed13c43a79b7cfb949f70a98fa57f2855fae9ff0005d7f8ae3e2eff00c1537e2beaf6d2f9d69a5ea2ba5dab8ef15ac6aa3e9862d5f9175e1e63539ebc99e430a28a2b88414514500145145001451450014514500145145001451450014514500145145001451450014514500145145001451450014514500145145001451450014514500145145007a8fc10f1fdd7c29f8c7e15f895672985f42d56d2f77af50b0caacdf9a822bfd7afc11e26d3bc69e06d13c5f0c808d56ca0bb4c74db346ae3f9d7f8e1d7faaa7fc127fe265c7c69ff008278fc26f19cd71e65c1d121b6b8ef896dff007657f00057d0e413f7a513481fa218ff0063ff0021d18ff63ff21d2f93a87fcfc3d1e4ea1ff3f0f48ead0fffd1fec62f20f22193efd729e2dd42c745d16e756b891e28ec6de4b86f9fb44858fe82bb4bbfb441febe34ff00595f257edb7e389be1c7ec9ff10fc61137d9fec3e1cd45f3e84c0ca3f9d7d45eda9efb3fc7fbf692f164de3bfda17c73e339e5339d4f5ed46e039eeaf72e57ff001dc578a5492c924d234d312cce7249ea49a8ebe624eeee780145145200a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800aff44bff0083673e21bf8a7fe09c96de11693cc9b40d76fa039fe14958488bf403a57f9da57f6f5ff069bf8c9ae3e15fc53f87acea5adf55b4bf8d586768787cb38f41915e9e512b5648e8c33f78febabfe26147fc4c2acf9f73ff003cd28f3ee7fe79a5765e407fffd2fec625c4f049e7c75f94bff05b8f145af83ffe0975f1a759b46d93b787a68223e8f295515faa330f26193eff00995f851ff071c6b1fd81ff0004a3f8812c3fba178d696a47a89250315f4957e167b75285a2cff2c7a28a2be6cf1028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bfad5ff00834e7c56d63f1efe29784a5e52ef45b39d17fdb8e7604ffdf35fc9557f4abff06bc6b77561fb766b9a3da49b0def87a63f5f29c1feb5db973b5689ad0f891fe81fff006ce8ff00b67587e7dfff0097a3cfbfff002f5e9fb8759fffd3fecaaf27b7f264ff0096b27995fce0ff00c1d0b793d97fc1283c4b7107115d6b9a3c7f771d67afe86aee0b79c7f1f99fee57f3a9ff00074b491bff00c12875d55fbc3c45a2e7e9e79afa2c47c0cf52afc2cff324a28a2be74f2c28a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bfa18ff008364afbecbff00053282d9beecde1ad533ff000111e2bf9e7afe827fe0d9b119ff00829fe9eb2fdd3e1bd5c1ff00be63aebc0ff1a2694be247fa29f9b61ff3ecdfe7fed851e6d87fcfb37f9ffb6149ff0012ff0047ff00bee8ff00897fa3ff00df75ea9d1eccffd4fec87c8d3e087ed024f2bf795fcf27fc1cfda55aeabff049ef133583f9ab6fa9e9137d365c57f43779069fe4feff0067fdb4afc40ff8388bc3f1f883fe0945f13963c66ca082ef62f4022941cd7d1d65ee33daab5af168ff0029da28a2be70f1428a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bfa26ff8362b4c96effe0a4df6f89777d9bc33a967db7f963fa57f3b35fd3dff00c1abde1f9b52fdb5bc59ad41c1d3fc36fcfa79b205fe95d5825fbd8951dcfef9fce1ff003cd3ff0020d1e70ff9e69ff906b3fcff00f73f2a3cff00f73f2af58ee3ffd5fecaa6fb3cd0fefa378bc992bf313fe0b0fe053e3fff0082697c68f0aa4599dbc337453e4e8530d91f4c57e9dde7ee21fb3f96f5f357ed4be186f1afecebe3af068579db51d0750831eed6ef8fd715f4d257563de92d2c7f8b5515a1ab69975a26ab73a35f0db3da4af0c83d1a362a47e62b3ebe64f0428a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bfaf9ff834dbc2b37fc2caf8b5e39db954d32c6c94fa379ace7f4afe41abfba0ff0083527c133e8ffb3c7c49f88573f20d4b5a86de16f548601bbf26af472a8deba2e9ee7f565ff004ff00be28ff008027fdf159dfda96ff00f3efff008e51fda96fff003eff00f8e5779da7ffd6feceef27b710c7fbcff969fdfae6f588e0d5a296c2e86ef391a371eaac3047e55b72c1fb98fecf27fcb4ff005759d79fb89b877ffa675f628fa03fc5fbf6d9f87f79f0aff6bef899e00be4f2df4df126a481718c235c3ba0ff00be1857cbf5fb83ff000713fc243f09ff00e0ab7f10c88cc69e24fb2eb6b9e9fe931ed38fc52bf0fabe5b154b92aca1d8f01ee145145738828a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a0028a28a002bfd1cffe0dd2f86937817fe09a3e1fd62f232936bba85f6a1953b4b44f2623fd057f9c6aa962154649e0015feb11ff0004f9f84e3e08fec51f0d3e1f14d8ba6f86acc30efba48c4873eff357a99543df6cb81f577dbadbfdba3edd6dfedd58f3edbfe79c7479f6dff3ce3aeae44769ffd7fecf2f3fd4fef9eb12f20b8fddfdcae92f3fd57fd34964ae6eee7b7826fb3cf1a57d4a3ddc49fc057fc1df9f04ce95f19be197c7eb241e56ab6171a4dcc9d774d03092219f68f35fc69d7fa5d7fc1cf9f00dbe317fc139effc6ba75bb4d77e01d460d5a3623ee40dfbab83edf21aff00345af1b3485ab33c7acbde0a28a2bcf320a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803e91fd8f7e175d7c69fda9be1ff00c2db44f30eb1aed942cb8ce6312abc831fee29aff5c3d2345d334bb08b46b767482d21486357ec91a8551f8002bfcea7fe0dbcf80c7e2d7fc1442c7c737b6de7d8f8174e9f53638fb970e3ca83f9b57fa334d05c79d279f1ff00adff00c895f4f92d2b537236a48c4f220ff9e9479107fcf4a5fdcfb51fb9f6ac0ea3ffd0fed16616fe4c7713c7ff007eeb9bbc82de79bf711f995b734171e4c76f07fcf4acd9a7b7ff008f8f92bea51f407c61fb5dfc28d37e3efc05f19fc10d5e21343e23d22eacb69e85a48cecff00c7b15fe3a1e3cf076adf0f3c71ac78075f4d97da25ecf633a918c496f218db8faad7fb56f8905bff00697d9fcc4ffbeebfcbdffe0e20fd96e7fd9b3fe0a4de28d56c6dbc9d1bc7491ebd6440c47ba51b6745ff0071c73fef563c430fde26791563a5cfc2ea28a2be70e60a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28ad0d274bbfd7354b6d174a88cd75792a410c6bd5a49085551f5240a00fee83fe0d69fd9e2e3c19fb39f8b3f68fd4e02b278bb555d3ed64518636d6230c3e9e6938ed5fd5a4b3dc41e64fe679b1ff00d34af8c3f604fd9fd7f659fd8d3e1dfc10863066d2b4980dd7eef05ae255124a5bdf71c1fa57d9f0c1fbdb983cbff5b5f6f84a5c94d44e882b19bf6eff00a6947dbbfe9a556fec53fe5e8fec53fe5ebc83a0ffd1fecf2f3ed1fbaff9e7feabee5625e41fbefb3fc95b72ff00a98ffe9ac958937d9e09becf06ff00fb695f628fa03cdf5dfdc6a3feafcaafe55bfe0eb0fd9164f89ffb23f86ff69df0e5a79ba97c3ebb10de3a2e5bfb3ef3e53f454930c6bfaa8f127fc7e45e77fac96bcbfe3f7c13d03f695f801e2cf809e30811acfc4fa65c5848ae32b9950853ff00016c1fc2b2ce61cd5394c234928389fe2ef457ad7c78f83fe27fd9ff00e33f89fe0a78ce168752f0c6a33e9f32b8da4f92e555b1e8eb861ec6bc96be4dab6878d60a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800afd7cff00821cfecacffb547fc142bc1fa46a36fe7689e1793fb73512cb94d96bcc4a4f62d26ddbf4afc83afef9bfe0d8cfd90e7f859fb306adfb49788ad3cad57e205c6cb3678fe61a75b1c260ff007647cb0aeecba873d54bb1505a9fd3a5ddbda8962037e04956a1fdc4d2dbfcfe5d569a0b883cbb79f6512cfa87fa479fb3caafb237427907fe7a7fe3947907fe7a7fe3959ff67d47fe7a51f67d47fe7a57cf9b9fffd2feccef20fdf473c11ffcb4acd9bf7f2c9fbbae92682dfca8f12573779f679a593c892bea51f4079778920b8fed28bee4b15749a6ff00c79fda3ccae6f5d83cfbcff9e9e6d7496763e443f67f2dff00d5feefe7a9c57f149c39fe7ebff075d7ec3f27c34f8fde1ffdb3fc2165b34af1cc5f60d59a35f95351b71f23b6075963ea4f70057f22b5fec25ff0527fd8c3c3ff00b767ec5fe2dfd9f75c8e1fb75f5a35c69d31604c17f00df6eebff0218fc6bfc85bc71e0bf11fc39f196a9e01f17db359ea9a35d4b65770b8c14961628e39c771c7b57939951e5a975d4f33174ed2b9cb514515e71c81451450014514500145145001451450014514500145145001451450014514500145145001451450014514500145145001451450014514500145145007d0dfb287ecfde25fda97f689f097c05f0a44d25cf88b508ad9caff00cb383399a4f6db1827d338aff599f845f0e7c23f07fe17e81f0a3c1d68b6fa6787ec61b1b74418012140b9c7be335fc93ffc1af3fb0a245a5f883f6e2f1e58e5ee03e91e1ef317a20ff8f99d73ea7083d8715fd8ef916fe77d9e0afa9ca30dcb4f9df5368474097f7f345f73cb96ab43079df69ffda7567c8b883fd23ff21d1e45bffa4dbc1ff2d6bd72d18bf67ff7e8fb3ffbf5b3cffcf47ffbe28e7fe7a3ff00df15e058dae8ffd3fed16f3ecfe4c5ff002d7fe7a5737379f34dfb88ff00f1cadaf3fc89a2b8f2fcd8ff00e5a562cd3dbf9d5f528fa03cbbc473cf05e7d9eba4d1e7b8167f689bfed9d6278927b7fed28ee3fe7ad747a3c1e7c3fbff00dd45fee5199ff19016659ede6863b79ffd657f9e77fc1d1dff0004ea5f835f1b2c3f6d8f86d645740f1b30b6d6c46985835341f2ca703004ea31fef002bfd0ce5fb3c10ffd71ff0062be50fdb1bf661f86bfb63fecefe28fd9d3e22da2cba77886cde14908f9a09c0cc3327a346d8208ac311479e1632ad4b9a363fc69e8afa03f6a5fd9c7e217ec97f1efc4bfb3f7c4fb66b6d57c3978f6ec5970258c1fdd4c9eab22608c7d3b57cff5f3ad5b43c4b0514514005145140051451400514514005145140051451400514514005145140051451400514514005145140051451400514514005145140057d29fb21fecd1e34fdaf3f689f0c7c00f02c4cd75aedda472c8a3220b65399a66f40899f6ce057cd75fdf2ffc1b7dff0004e31f02be0dc9fb5cfc54d3b678a3c6d085d2e397e57b5d33f84e0f2ad3753d3e5c0aecc0e1bdacd47a0e28fe8afe02fc10f067ecf1f05fc3bf077e1fdbad8e93e1cb28ace05518cf96a0163eec7935ea9e479137ee37cb567c8b7822927f33cdaad0fd9e0ff5ff00f6cebecd2b2b23af0c134ffeaade78de8867b7ff0049f3ff00eb9559f3adff0077f678dffeb9d559a0fdedcf9f1ffaaa585c31987d847fb747d847fb74be45bffcf37ffbe28f22dffe79bffdf15e481fffd4fed16f3a47e7dbfeebfe7a5737341e7c3f67823f2a4864adb9a7f3a68ade793ca8ab9b9bf7ff00e8ff00f3c7fe5a57d4a3e80f2ef148f22f3ed15dae9b3dc4f67f679e37f2fcbae2bc53feba3fde56de8f3e2cfcff0032a315fc5270e749fe913c3f67f2d2b365fb3c137d9be48a8ff4883cafde5569bf7f2c9ff3d6abeb21893f95ff00f83953fe096bff000d35f0753f6bef835a56ff001bf82edcff0068c3027cf7fa62f2dc0fbd241f797fd9c8aff3b5208383c62bfdb4750d3edef6d64b3bc45961954a3a30cab2b0c1047a11c57f9bb7fc1c25ff000497bbfd8abe36cdfb447c1bd3987c35f195cb48c912fc9a65fc872f09c70b1c8798fdf8f4af3b33c3ebce8f32b51d3991fcdb514515e41c814514500145145001451450014514500145145001451450014514500145145001451450014514500145145001451450014515eeff00b34fece9f12ff6acf8d3a1fc0cf84f64d79abeb73ac4b8525218f237cd263a246bc93f8538c6fa203f4aff00e0899ff04e2bdfdbcbf69db5d4fc656adff080783e48ef75790f0b70ea434568a7a12e7961d97eb5fe953a558dae87a65be8da3c096d69146b0c314636aa220daaaa074000c0af90bf615fd8d7e1b7ec2dfb3d681f043c036aa7ec7b64bfbbda0497776c3f7b339f73d07615f6b4df67ff00961fbd8abec7018454a16ea6d1560f3bf73feb2acf9f6de77da3fe59d56960f3e193eff9544305c5771b618d2ff4783ca1f2566791703ed371f255a9adee209a3b8ff9e5fedd1fe8f04d27ef3fed9d1727a89f68bfff009e547da2ff00fe79557ff46a3fd1abc0b9ad91ffd5feca6f27f3e6fdcd628fdfc3fbfadb9a0b89e68ee3ccff00967589341fb9ff0058f17ef2bea51f4079bf8abed13ea51feefcd8ab4b4d9e7fb1fee3fd6566f8aa0b8fb64771356dd9cf6ff63fdc489e6454669fc73928166ee6b9f262f223acd867bff3bed13d697fa47fcf4ff5b59b8c4b27dcff00bee91d61e7f912fee2bc2fe3efc08f879fb4bfc23f107c1bf8c7a747a9e83af5b35bdc4320071b87cae9fdd743ca91d0d7b17917104318f32ab7effce93efd7358563fc99bfe0a77ff0004e3f8a5ff0004e3fda0aefe1c78aa192ebc35a8b35c681ab6dfdddd5a93c293d04b18e1d7f1e95f9b55feaedff053ff00843fb147ed27fb3b6aff00097f6b1f1268fe1f8846d2d96a177730c7716172abf24d18277f07ef00391c57f96afc62f01691f0c7e25eb1e06f0febd61e28b0d3ae1e2b6d574d7f32daea207e4910f6c8ea0f435e5e2b0fc8f4d8f23114791e8799d14515c87385145140051451400514514005145140051451400514514005145140051451400514514005145140051452aa962157a9e28037bc2de16f1178dfc4963e0ff08d94ba8ea7a94c96f6b6d02979259643b555547726bfd1abfe08adff0004a9d33f608f841ff09cfc44b58a7f895e26811f519b01bec509195b489bb63f8c8ea6be0fff0082027fc12e7e1a7c26f0f5a7ed69f156ff004bd7bc71a8c21f4bb082786e7fb2a171f79c23362761d7fba3815fd4f79f71e749e47fdb4af772fc272fbf23ba851b6acd2f3fc99a2b71ff003ceb4a19ff0073feaff7915667fa3f9de7cf4be7fee63ffbf5fbbafa33368dbf3ff73479f713ff00cb3f2bf79589fea3ccb883fe7a5694d3fef7f8e81599a5e45c7eeff79e5556f23fd679ff00f2c68bcbe9c4d1f9155bceb883ccb89e80458fb0d8ff00cf34ff00be28fb0d8ffcf34ffbe28fb74ffdfa3edd3ff7ebe7cdae8fffd6fecaaf27b8f3a31f27fd748deb13edd71e4f9fff002ceb4bcffdf7ee23ac4bbc793f68f2ebea51ef620e03c553dbcf3476f37fe43a4d1e0b7fb27f1f995cff008e354d2f4a8ceadacdc45696b6ea5e49a6611c68a3bb3360015f823fb6cffc1c61fb0a7eca11ddf847e1e5dcbf127c556f94fb2e92c3ec914838c4b727e4e0f555c9f4acb19351a976294d4773fa29ff008f787993f795f067ed2fff00051efd8b7f645d2e7b8f8e1e3fd2b49b85076d92ccb35db91fc2b0c796cfb1c57f9f2fed93ff00070aff00c141ff006b0371a0e89aeafc3ff0dcb951a7e8598e4643c6d92e4fef1bf0db8afc40d6f5dd6fc4ba9cbadf88ef27bfbd9cee967b991a595cfab3b924fe35c353304be1471e23189fc27f72dfb4d7fc1d9ff0cbc3a2e7c3dfb27f816ef5f2bf2c7a86b527d9603ee214cc9c7a1c57f3e3fb46ff00c17d3fe0a5ff00b4535c595d78e5fc2da65c641b2d0a35b55d87a2b3fcce71ea08afc62a2b8678a9cba9cb3c44d9d578b7c75e36f1eea2757f1ceb17bacdd1cfefafa792e1f9ff006a42c6b95a28ae7300a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803bdf027c54f897f0bf505d57e1c6bfa8685708c1c3d8dcc90723a6763007f1afd7afd9e7fe0e03ff828b7c0a3069facf8922f1a6971101adb5a883b941fc2b326d65fae0d7e24515a42aca3f0b2a336b63fbacfd9abfe0e8cfd9ebc6f35a68ffb49786350f065d6143de581fb6da6e3c7dde2503dc8c015fd05fc01fdb1bf65ff00da67455d77e0778df4bd7d5bee456f3af9ca7d1a23870477e2bfc922ba5f0978cfc5de02d6a2f11f82354bad22fe020c7716733c122e3d19083f8577d2cd26be2378e25adcff006249a6f23ccff9e52ffb756619ff007df68aff003aff00d90bfe0e2bfdb77f6796b6f0ff00c579e1f88fa0458529a97c97889d3e4b841ce0740c3ea6bfac9fd8abfe0b8bfb0bfed8e6d3c3716b5ff086f89ee005fecad68ac259f1f7629bfd5c9f81af7b0f98d2a9a2763a615a2cfd909a71e6c571e6799557cfb79e193c88ff00f1fa97ceb1b830cf6ed13a5c0cab2904107a10476a97c8d3fc990411ff00ad93fbf5de63f56337fd1ffe783d1fe8ff00f3c1eb47ecf73ff3cd3fefba3ecf73ff003cd3fefbaf9f333fffd7fec826207fa457e0d7fc14bbfe0bb7fb22ff00c13fad25f09c17c9e35f1eaa909a069922b081bb1ba987cb10f6fbde82bf9dbff82c1ffc1ccfe34f8bd75aa7ecfbfb005d4ba0f85c97b7bef1281b6eef47dd22d3fe79447fbff788e98eb5fc7aea5a96a3ac5fcdaaeaf3c9757570e64966958bbbbb725999b2493ea6bd2ad8cb6913d2ad8bb6913f537f6f6ff82c8fedabfb7feb9731fc42f1049a17862463e4e81a4bb416889d848461a638e096e0fa57e52d145704ea393bc8f39bb8514515020a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a7c723c4eb2c44ab290411c10474c5328a00fda3fd837fe0b93fb62fec5b7d67e1ed5f527f1d783a1215f49d5a46778e3eff679ce5d0e3a0391db815fdbcfec13ff000564fd973f6f1d0611f0c7565d2bc4c30d7ba0ea12ac57719efe583c4abe852bfcb8eba4f08f8c3c55e01f11da78bfc13a8dc695aa5848b2dbdd5ac86296375e855970457a585cce74f47aa37a75dc4ff62efed0ff00a669ff007dd1fda1ff004cd3fefbaff2f2ff0087cf7fc14aff00e8a96a5f947ffc4d1ff0f9eff8295ffd152d4bf28fff0089a7f5d8f62bdba3ffd0ff003ffa28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803fffd1ff003ffa28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803fffd2ff003ffa28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803fffd3ff003ffa28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2800a28a2803fffd9
""";

// Mrz: D1NLD11234567890AG5R98GT5IN2L4
const dg12Test = """
      71 24
          82 1f 0044314E4C443131323334353637383930414735523938475435494E324C34
          81 01 41
""";

const dg13Test = """
    6f 1f
      30 1a
         30 0d
            06 09 2a864886f70d010101
            05 00
         03 0b
            00
            30 09
               02 02 c1d3
               02 03 010001
""";
