import browserslist from 'browserslist';
import RegexpTree from 'regexp-tree';
import { regexes } from 'ua-regexes-lite';

/**
 * Array of numbers to array of first and last elements.
 * @param numbers - Array of numbers.
 * @returns Number or two numbers.
 */ function numbersToRanges(numbers) {
    if (typeof numbers === "number") {
        return numbers;
    }
    if (numbers.length === 1) {
        return numbers[0];
    }
    return [
        numbers[0],
        numbers[numbers.length - 1]
    ];
}

var SemverPart;
(function(SemverPart) {
    SemverPart[SemverPart["Major"] = 0] = "Major";
    SemverPart[SemverPart["Minor"] = 1] = "Minor";
    SemverPart[SemverPart["Patch"] = 2] = "Patch";
})(SemverPart || (SemverPart = {}));

/**
 * Get semver from string or array.
 * @param version - Target to convert.
 * @returns Array with semver parts.
 */ function semverify(version) {
    const versionParts = Array.isArray(version) ? version : version.toString().split(".");
    if (versionParts[0] === "all") {
        return [
            Infinity,
            0,
            0
        ];
    }
    let versionPart = null;
    let semverPart = null;
    const semver = [
        0,
        0,
        0
    ];
    for(let i = 0; i < 3; i++){
        versionPart = versionParts[i];
        if (typeof versionPart === "undefined") {
            continue;
        }
        semverPart = typeof versionPart === "number" ? versionPart : parseInt(versionPart, 10);
        if (isNaN(semverPart)) {
            return null;
        }
        semver[i] = semverPart;
    }
    return semver;
}
/**
 * Get semver range.
 * @param from
 * @param to
 * @returns Semver range.
 */ function rangeSemver(from, to) {
    let partIndex = 0;
    const range = [];
    for(let i = 2; i >= 0; i--){
        if (from[i] !== to[i]) {
            partIndex = i;
            break;
        }
    }
    for(let i1 = from[partIndex], max = to[partIndex]; i1 <= max; i1++){
        range.push(from.map((v, j)=>j === partIndex ? i1 : v));
    }
    return range;
}
/**
 * Compare semvers.
 * @param a - Semver to compare.
 * @param b - Semver to compare with.
 * @param options - Compare options.
 * @returns Equals or not.
 */ function compareSemvers(a, b, options) {
    const [major, minor, patch] = a;
    const [majorBase, minorBase, patchBase] = b;
    const { ignoreMinor , ignorePatch , allowHigherVersions  } = options;
    if (majorBase === Infinity) {
        return true;
    }
    const compareMinor = !ignoreMinor;
    const comparePatch = compareMinor && !ignorePatch;
    if (allowHigherVersions) {
        if (comparePatch && patch < patchBase || compareMinor && minor < minorBase) {
            return false;
        }
        return major >= majorBase;
    }
    if (comparePatch && patch !== patchBase || compareMinor && minor !== minorBase) {
        return false;
    }
    return major === majorBase;
}
/**
 * Get required semver parts count.
 * @param version - Semver parts or ranges.
 * @param options - Semver compare options.
 * @returns Required semver parts count.
 */ function getRequiredSemverPartsCount(version, options) {
    const { ignoreMinor , ignorePatch , allowZeroSubversions  } = options;
    let shouldRepeatCount = ignoreMinor ? 1 : ignorePatch ? 2 : 3;
    if (allowZeroSubversions) {
        for(let i = shouldRepeatCount - 1; i > 0; i--){
            if (version[i] !== 0 || shouldRepeatCount === 1) {
                break;
            }
            shouldRepeatCount--;
        }
    }
    return shouldRepeatCount;
}

/**
 * Browsers strings to info objects.
 * @param browsersList - Browsers strings with family and version.
 * @returns Browser info objects.
 */ function parseBrowsersList(browsersList) {
    return browsersList.reduce((browsers, browser)=>{
        const [family, versionString, versionStringTo] = browser.split(/ |-/);
        const version = semverify(versionString);
        const versions = !version ? [] : versionStringTo ? rangeSemver(version, semverify(versionStringTo)) : [
            version
        ];
        return versions.reduce((browsers, semver)=>{
            if (semver) {
                browsers.push({
                    family,
                    version: semver
                });
            }
            return browsers;
        }, browsers);
    }, []);
}
/**
 * Request browsers list.
 * @param options - Options to get browsers list.
 * @returns Browser info objects.
 */ function getBrowsersList(options = {}) {
    const { browsers , ...browserslistOptions } = options;
    const browsersList = browserslist(browsers, browserslistOptions);
    const parsedBrowsers = parseBrowsersList(browsersList);
    return parsedBrowsers;
}

/**
 * Compare two arrays.
 * @param a - Array to compare.
 * @param b - Array to compare.
 * @param from - Index to start compare from.
 * @returns Equals or not.
 */ function compareArrays(a, b, from = 0) {
    const len = a.length;
    for(let i = from; i < len; i++){
        if (a[i] !== b[i]) {
            return false;
        }
    }
    return true;
}
/**
 * Clone simple object.
 * @param value
 * @returns Object clone.
 */ function clone(value) {
    if (value === null || typeof value !== "object") {
        return value;
    }
    /* eslint-disable */ const copy = Array.isArray(value) ? [] : {};
    let i;
    for(i in value){
        copy[i] = clone(value[i]);
    }
    /* eslint-enable */ return copy;
}
/**
 * Concat arrays.
 * @param items
 * @returns Concatinated arrays.
 */ function concat(items) {
    return [].concat(...items);
}

/**
 * Merge browser info object to map with versions.
 * @param browsers - Browser info object to merge.
 * @returns Merged browsers map.
 */ function mergeBrowserVersions(browsers) {
    const merge = new Map();
    browsers.forEach(({ family , version  })=>{
        const versions = merge.get(family);
        if (versions) {
            const strVersion = version.join(".");
            if (versions.every((_)=>_.join(".") !== strVersion)) {
                versions.push(version);
            }
            return;
        }
        merge.set(family, [
            version
        ]);
    });
    merge.forEach((versions)=>{
        versions.sort((a, b)=>{
            for(const i in a){
                if (a[i] !== b[i]) {
                    return a[i] - b[i];
                }
            }
            return 0;
        });
    });
    return merge;
}
/**
 * Versions to ranged versions.
 * @param versions - Semver versions list.
 * @returns Ranged versions list.
 */ function versionsListToRanges(versions) {
    if (versions.length < 2) {
        return versions;
    }
    const max = versions.length + 1;
    const ranges = [];
    let prev = null;
    let current = versions[0];
    let major = [
        current[SemverPart.Major]
    ];
    let minor = [
        current[SemverPart.Minor]
    ];
    let patch = [
        current[SemverPart.Patch]
    ];
    let part = null;
    for(let i = 1; i < max; i++){
        prev = versions[i - 1];
        current = versions[i] || [];
        for(let p = SemverPart.Major; p <= SemverPart.Patch; p++){
            if ((p === part || part === null) && prev[p] + 1 === current[p] && compareArrays(prev, current, p + 1)) {
                part = p;
                if (p === SemverPart.Major) {
                    major.push(current[SemverPart.Major]);
                } else {
                    major = current[SemverPart.Major];
                }
                if (p === SemverPart.Minor) {
                    minor.push(current[SemverPart.Minor]);
                } else {
                    minor = current[SemverPart.Minor];
                }
                if (p === SemverPart.Patch) {
                    patch.push(current[SemverPart.Patch]);
                } else {
                    patch = current[SemverPart.Patch];
                }
                break;
            }
            if (part === p || prev[p] !== current[p]) {
                ranges.push([
                    numbersToRanges(major),
                    numbersToRanges(minor),
                    numbersToRanges(patch)
                ]);
                major = [
                    current[SemverPart.Major]
                ];
                minor = [
                    current[SemverPart.Minor]
                ];
                patch = [
                    current[SemverPart.Patch]
                ];
                part = null;
                break;
            }
        }
    }
    return ranges;
}

function AstRegExpNode(body) {
    return {
        type: "RegExp",
        body,
        flags: ""
    };
}
function AlternativeNode(...expressions) {
    const exps = concat(expressions).filter(Boolean);
    if (exps.length === 1) {
        return exps[0];
    }
    return {
        type: "Alternative",
        expressions: exps
    };
}
function SimpleCharNode(value) {
    return {
        type: "Char",
        kind: "simple",
        value: String(value),
        codePoint: NaN
    };
}
function MetaCharNode(value) {
    return {
        type: "Char",
        kind: "meta",
        value,
        codePoint: NaN
    };
}
function ClassRangeNode(from, to) {
    return {
        type: "ClassRange",
        from,
        to
    };
}
function CharacterClassNode(...expressions) {
    return {
        type: "CharacterClass",
        expressions: concat(expressions).filter(Boolean)
    };
}
function SimpleQuantifierNode(kind) {
    return {
        type: "Quantifier",
        kind,
        greedy: true
    };
}
function RangeQuantifierNode(from, to) {
    return {
        type: "Quantifier",
        kind: "Range",
        from,
        to,
        greedy: true
    };
}
function CapturingGroupNode(expression) {
    return {
        type: "Group",
        capturing: true,
        expression,
        number: null
    };
}
function RepetitionNode(expression, quantifier) {
    return {
        type: "Repetition",
        expression,
        quantifier
    };
}
function DisjunctionNode(...expressions) {
    const exprs = concat(expressions).filter(Boolean);
    if (exprs.length === 1) {
        return exprs[0];
    }
    const disjunction = {
        type: "Disjunction",
        left: null,
        right: exprs.pop()
    };
    exprs.reduceRight((disjunction, expr, i)=>{
        if (i === 0) {
            disjunction.left = expr;
            return disjunction;
        }
        disjunction.left = {
            type: "Disjunction",
            left: null,
            right: expr
        };
        return disjunction.left;
    }, disjunction);
    return disjunction;
}
function DisjunctionCapturingGroupNode(...expressions) {
    const expr = DisjunctionNode(...expressions);
    if (expr.type === "Disjunction") {
        return CapturingGroupNode(expr);
    }
    return expr;
}
function DigitPatternNode() {
    return MetaCharNode("\\d");
}
function NumberPatternNode(quantifier = SimpleQuantifierNode("+")) {
    const numberPattern = RepetitionNode(DigitPatternNode(), quantifier);
    return numberPattern;
}
function NumberCharsNode(value) {
    return AlternativeNode(Array.from(String(value), SimpleCharNode));
}

/**
 * Check node whether is number pattern.
 * @param node - AST node to check.
 * @returns Is number pattern or not.
 */ function isNumberPatternNode(node) {
    if (node.type === "Group" && node.expression.type === "Repetition") {
        const { expression , quantifier  } = node.expression;
        return expression.type === "Char" && expression.value === "\\d" && quantifier.kind === "+" && quantifier.greedy;
    }
    return false;
}
/**
 * Check node whether is char node.
 * @param node - AST node to check.
 * @param value - Value to compare.
 * @returns Is char node or not.
 */ function isCharNode(node, value) {
    if (node && node.type === "Char") {
        return typeof value === "undefined" || value instanceof RegExp && value.test(node.value) || String(value) === node.value;
    }
    return false;
}
/**
 * Check node whether is digit range.
 * @param node - AST node to check.
 * @returns Is digit range or not.
 */ function isDigitRangeNode(node) {
    if (node.type === "CharacterClass" && node.expressions.length === 1) {
        const [expression] = node.expressions;
        return expression.type === "ClassRange" && isCharNode(expression.from, /\d/) && isCharNode(expression.to, /\d/);
    }
    return false;
}
/**
 * Check node whether is expression.
 * @param node - AST node to check.
 * @returns Is expression node or not.
 */ function isExpressionNode(node) {
    return node.type !== "RegExp" && node.type !== "ClassRange" && node.type !== "Quantifier";
}
function parseRegex(regex) {
    return typeof regex === "string" ? RegexpTree.parse(regex.replace(/^([^/])/, "/$1").replace(/([^/])$/, "$1/")) : regex instanceof RegExp ? RegexpTree.parse(regex) : regex;
}
/**
 * Get regex from string or AST.
 * @param src - String or AST.
 * @returns RegExp.
 */ function toRegex(src) {
    return typeof src === "string" ? new RegExp(src) : new RegExp(RegexpTree.generate(src.body), src.flags);
}
/**
 * Get string from regex or AST.
 * @param src - RegExp or AST.
 * @returns String.
 */ function toString(src) {
    return typeof src === "string" ? src : src instanceof RegExp ? src.toString() : RegexpTree.generate(src);
}

const classes = [
    "RegExp",
    "Disjunction",
    "Alternative",
    "Assertion",
    "Char",
    "CharacterClass",
    "ClassRange",
    "Backreference",
    "Group",
    "Repetition",
    "Quantifier"
];
/**
 * Create traversal visitors.
 * @param visitors
 * @returns Traversal handlers.
 */ function visitors(visitors) {
    const { every  } = visitors;
    if (!every) {
        return visitors;
    }
    if (typeof every === "function") {
        return {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            "*": every,
            ...visitors
        };
    }
    return classes.reduce((newVisitors, className)=>{
        const visitor = visitors[className];
        const visitorPre = visitor ? "pre" in visitor ? visitor.pre : visitor : null;
        const visitorPost = visitor ? "post" in visitor ? visitor.post : null : null;
        newVisitors[className] = {
            pre (nodePath) {
                if (every.pre(nodePath) !== false && visitorPre) {
                    return visitorPre(nodePath);
                }
                return true;
            },
            post (nodePath) {
                if (every.post(nodePath) !== false && visitorPost) {
                    return visitorPost(nodePath);
                }
                return true;
            }
        };
        return newVisitors;
    }, {});
}

function optimizeRegex(regex) {
    // Optimization requires filled codePoints
    const regexAst = RegexpTree.optimize(parseRegex(toString(regex))).getAST();
    RegexpTree.traverse(regexAst, {
        Group (nodePath) {
            const { parent , node  } = nodePath;
            const { expression  } = node;
            node.capturing = true;
            if (parent.type === "RegExp" || expression.type !== "Disjunction" && parent.type !== "Repetition" || expression.type === "Disjunction" && parent.type === "Disjunction") {
                nodePath.replace(nodePath.node.expression);
            }
        }
    });
    return regexAst;
}
/**
 * Merge digits patterns if possible.
 * @param a
 * @param b
 * @returns Merged node.
 */ function mergeDigits(a, b) {
    if (isCharNode(a) && isCharNode(b) && a.value === b.value) {
        return b;
    }
    if (isCharNode(a, /\d/) && isDigitRangeNode(b) && Number(b.expressions[0].from.value) - Number(a.value) === 1) {
        return {
            ...b,
            expressions: [
                {
                    ...b.expressions[0],
                    from: a
                }
            ]
        };
    }
    if (isDigitRangeNode(a) && isCharNode(b, /\d/) && Number(b.value) - Number(a.expressions[0].to.value) === 1) {
        return {
            ...a,
            expressions: [
                {
                    ...a.expressions[0],
                    to: b
                }
            ]
        };
    }
    return null;
}
/**
 * Optimize segment number patterns.
 * @param patterns
 * @returns Optimized segment number patterns.
 */ function optimizeSegmentNumberPatterns(patterns) {
    return patterns.reduce((patterns, node)=>{
        const prevNode = patterns[patterns.length - 1];
        if (prevNode && node.type === "Alternative" && prevNode.type === "Alternative" && node.expressions.length === prevNode.expressions.length) {
            const merged = prevNode.expressions.reduceRight((exps, exp, i)=>{
                if (!exps) {
                    return exps;
                }
                const merged = mergeDigits(exp, node.expressions[i]);
                if (merged) {
                    exps.unshift(merged);
                } else {
                    return null;
                }
                return exps;
            }, []);
            if (merged) {
                node.expressions = merged;
                patterns.pop();
            }
        }
        patterns.push(node);
        return patterns;
    }, []);
}

/**
 * Transform number to digits array.
 * @param num - Target number.
 * @returns Digits array.
 */ function numberToDigits(num) {
    return Array.from(num.toString(), Number);
}

/**
 * Get digit pattern.
 * @param digit - Ray start.
 * @param includes - Include start digit or use next.
 * @returns Digit pattern.
 */ function rayRangeDigitPattern(digit, includes) {
    const rangeStart = digit + Number(!includes);
    if (rangeStart === 0) {
        return DigitPatternNode();
    }
    if (rangeStart === 9) {
        return SimpleCharNode("9");
    }
    if (rangeStart > 9) {
        return null;
    }
    return CharacterClassNode(ClassRangeNode(SimpleCharNode(rangeStart), SimpleCharNode("9")));
}
/**
 * Create numeric ray pattern.
 * @param from - Start from this number.
 * @returns Numeric ray pattern parts.
 */ function rayToNumberPatterns(from) {
    if (from === 0) {
        return [
            NumberPatternNode()
        ];
    }
    const digits = numberToDigits(from);
    const digitsCount = digits.length;
    const other = NumberPatternNode(RangeQuantifierNode(digitsCount + 1));
    const zeros = digitsCount - 1;
    if (from / Math.pow(10, zeros) === digits[0]) {
        return [
            AlternativeNode(rayRangeDigitPattern(digits[0], true), Array.from({
                length: zeros
            }, DigitPatternNode)),
            other
        ];
    }
    const raysNumberPatterns = digits.reduce((topNodes, _, i)=>{
        const ri = digitsCount - i - 1;
        const d = i === 0;
        let prev = SimpleCharNode("");
        const nodes = digits.reduce((nodes, digit, j)=>{
            if (j < ri) {
                nodes.push(SimpleCharNode(digit));
            } else if (prev) {
                if (j > ri) {
                    nodes.push(DigitPatternNode());
                } else {
                    prev = rayRangeDigitPattern(digit, d);
                    if (prev) {
                        nodes.push(prev);
                    } else {
                        return [];
                    }
                }
            }
            return nodes;
        }, []);
        if (nodes.length) {
            topNodes.push(nodes);
        }
        return topNodes;
    }, []);
    const numberPatterns = raysNumberPatterns.map((_)=>AlternativeNode(_));
    numberPatterns.push(other);
    return numberPatterns;
}

/**
 * Get digit pattern.
 * @param from - Segment start.
 * @param to - Segment end.
 * @param zeros - Zeros to add as prefix.
 * @returns Digit pattern.
 */ function segmentRangeNumberPattern(from, to, zeros) {
    if (to < from) {
        return null;
    }
    const fromNode = SimpleCharNode(from);
    const toNode = SimpleCharNode(to);
    const zerosPrefix = typeof zeros === "number" && zeros > 0 ? Array.from({
        length: zeros
    }, ()=>SimpleCharNode(0)) : [];
    const addPrefix = zerosPrefix.length ? (node)=>AlternativeNode(zerosPrefix, node) : (node)=>node;
    if (from === to) {
        return addPrefix(fromNode);
    }
    if (from === 0 && to === 9) {
        return addPrefix(DigitPatternNode());
    }
    if (to - from === 1) {
        return addPrefix(CharacterClassNode(fromNode, toNode));
    }
    return addPrefix(CharacterClassNode(ClassRangeNode(fromNode, toNode)));
}
/**
 * Split segment range to decade ranges.
 * @param from - Segment start.
 * @param to - Segment end.
 * @returns Ranges.
 */ function splitToDecadeRanges(from, to) {
    const ranges = [];
    let num = from;
    let decade = 1;
    do {
        decade *= 10;
        if (num < decade) {
            ranges.push([
                num,
                Math.min(decade - 1, to)
            ]);
            num = decade;
        }
    }while (decade <= to);
    return ranges;
}
/**
 * Get common and diffs of two numbers (arrays of digits).
 * @param a - Digits.
 * @param b - Other digits.
 * @returns Common part and diffs.
 */ function splitCommonDiff(a, b) {
    const len = a.length;
    if (len !== b.length || a[0] !== b[0]) {
        return null;
    }
    let common = a[0].toString();
    let currA = 0;
    let currB = 0;
    let diffA = "";
    let diffB = "";
    for(let i = 1; i < len; i++){
        currA = a[i];
        currB = b[i];
        if (currA === currB && diffA === "" && diffB === "") {
            common += currA;
        } else {
            diffA += currA;
            diffB += currB;
        }
    }
    return [
        common,
        parseInt(diffA, 10),
        parseInt(diffB, 10)
    ];
}
/**
 * Get segment patterns.
 * @param from - Segment start.
 * @param to - Segment end.
 * @param digitsInNumber - How many digits should be en number. Will be filled by zeros.
 * @returns Segment patterns.
 */ function segmentToNumberPatterns(from, to, digitsInNumber = 0) {
    const fromDigits = numberToDigits(from);
    const digitsCount = fromDigits.length;
    if (from < 10 && to < 10 || from === to) {
        const zeros = digitsInNumber - digitsCount;
        return [
            segmentRangeNumberPattern(from, to, zeros)
        ];
    }
    const toDigits = numberToDigits(to);
    if (digitsCount !== toDigits.length) {
        const decadeRanges = splitToDecadeRanges(from, to);
        const parts = concat(decadeRanges.map(([from, to])=>segmentToNumberPatterns(from, to, digitsInNumber)));
        return parts;
    }
    const commonStart = splitCommonDiff(fromDigits, toDigits);
    if (Array.isArray(commonStart)) {
        const [common, from1, to1] = commonStart;
        const digitsInNumber1 = digitsCount - common.length;
        const diffParts = segmentToNumberPatterns(from1, to1, digitsInNumber1);
        return [
            AlternativeNode(Array.from(common, SimpleCharNode), DisjunctionCapturingGroupNode(diffParts))
        ];
    }
    const range = Array.from({
        length: digitsCount - 1
    });
    const middleSegment = segmentRangeNumberPattern(fromDigits[0] + 1, toDigits[0] - 1);
    const parts1 = [
        ...range.map((_, i)=>{
            const ri = digitsCount - i - 1;
            const d = Number(i > 0);
            return AlternativeNode(fromDigits.map((digit, j)=>{
                if (j < ri) {
                    return SimpleCharNode(digit);
                }
                if (j > ri) {
                    return segmentRangeNumberPattern(0, 9);
                }
                return segmentRangeNumberPattern(digit + d, 9);
            }));
        }),
        // but output more readable
        ...middleSegment ? [
            AlternativeNode(middleSegment, Array.from({
                length: digitsCount - 1
            }, ()=>DigitPatternNode()))
        ] : [],
        ...range.map((_, i)=>{
            const ri = digitsCount - i - 1;
            const d = Number(i > 0);
            return AlternativeNode(toDigits.map((digit, j)=>{
                if (j < ri) {
                    return SimpleCharNode(digit);
                }
                if (j > ri) {
                    return segmentRangeNumberPattern(0, 9);
                }
                return segmentRangeNumberPattern(0, digit - d);
            }));
        })
    ];
    return optimizeSegmentNumberPatterns(parts1);
}

/**
 * Get regex for given numeric range.
 * @param from - Range start.
 * @param to - Range end.
 * @returns Range pattern.
 */ function rangeToRegex(from, to = Infinity) {
    if (from === Infinity) {
        return NumberPatternNode();
    }
    const numberPatterns = to === Infinity ? rayToNumberPatterns(from) : segmentToNumberPatterns(from, to);
    const regex = DisjunctionCapturingGroupNode(numberPatterns);
    return regex;
}

/**
 * Find matched versions.
 * @param minVersion - Semver version.
 * @param maxVersion - Semver version.
 * @param bases - Base semver versions.
 * @param options - Semver compare options.
 * @returns Matched versions.
 */ function findMatchedVersions(minVersion, maxVersion, bases, options) {
    const compareOptions = {
        ...options,
        allowHigherVersions: true
    };
    const minComparator = (ver)=>compareSemvers(ver, minVersion, compareOptions);
    const maxComparator = (ver)=>compareSemvers(maxVersion, ver, compareOptions);
    const comparator = minVersion && maxVersion ? (ver)=>minComparator(ver) && maxComparator(ver) : minVersion ? minComparator : maxVersion ? maxComparator : ()=>true;
    return bases.filter(comparator);
}

/**
 * Get useragent regexes for given browsers.
 * @param browsers - Browsers.
 * @param options - Semver compare options.
 * @param targetRegexes - Override default regexes.
 * @returns User agent regexes.
 */ function getRegexesForBrowsers(browsers, options, targetRegexes = regexes) {
    const result = [];
    let prevFamily = "";
    let prevRegexIsGlobal = false;
    targetRegexes.forEach((regex)=>{
        const requestVersions = browsers.get(regex.family);
        if (!requestVersions) {
            return;
        }
        let { version , minVersion , maxVersion  } = regex;
        if (version) {
            minVersion = version;
            maxVersion = version;
        }
        let matchedVersions = findMatchedVersions(minVersion, maxVersion, requestVersions, options);
        if (matchedVersions.length) {
            // regex contains global patch
            if (prevFamily === regex.family && prevRegexIsGlobal) {
                version = undefined;
                minVersion = undefined;
                maxVersion = undefined;
                matchedVersions = requestVersions;
                result.pop();
            }
            result.push({
                ...regex,
                version,
                minVersion,
                maxVersion,
                requestVersions,
                matchedVersions
            });
        }
        prevRegexIsGlobal = !version && !minVersion && !maxVersion;
        prevFamily = regex.family;
    });
    return result;
}

/**
 * Compile regexes.
 * @param regexes - Objects with info about compiled regexes.
 * @returns Objects with info about compiled regexes.
 */ function compileRegexes(regexes) {
    return regexes.map(({ regexAst , ...regex })=>{
        const optimizedRegexAst = optimizeRegex(regexAst);
        return {
            ...regex,
            regexAst: optimizedRegexAst,
            regex: toRegex(optimizedRegexAst)
        };
    });
}
/**
 * Compile regex.
 * @param regexes - Objects with info about compiled regexes.
 * @returns Compiled common regex.
 */ function compileRegex(regexes) {
    const partsRegexes = regexes.map(({ regexAst  })=>CapturingGroupNode(regexAst.body));
    const regexAst = optimizeRegex(AstRegExpNode(DisjunctionCapturingGroupNode(partsRegexes)));
    return toRegex(regexAst);
}

/**
 * Get number patterns count from the regex.
 * @param regex - Target regex.
 * @returns Number patterns count.
 */ function getNumberPatternsCount(regex) {
    const regexAst = parseRegex(regex);
    let count = 0;
    RegexpTree.traverse(regexAst, {
        Group (nodePath) {
            if (isNumberPatternNode(nodePath.node)) {
                count++;
            }
        }
    });
    return count;
}
function replaceNumberPatterns(regex, numbers, numberPatternsCount) {
    let regexAst = parseRegex(regex);
    const numbersToReplace = typeof numberPatternsCount === "number" && numberPatternsCount < numbers.length ? numbers.slice(0, numberPatternsCount) : numbers.slice();
    RegexpTree.traverse(regexAst, visitors({
        every () {
            return Boolean(numbersToReplace.length);
        },
        Group (nodePath) {
            if (isNumberPatternNode(nodePath.node) && numbersToReplace.length) {
                if (regexAst === nodePath.node) {
                    regexAst = numbersToReplace.shift();
                } else {
                    nodePath.replace(numbersToReplace.shift());
                }
                return false;
            }
            return true;
        }
    }));
    return regexAst;
}
/**
 * Get from regex part with number patterns.
 * @param regex - Target regex.
 * @param numberPatternsCount - Number patterns to extract.
 * @returns Regex part with number patterns.
 */ function getNumberPatternsPart(regex, numberPatternsCount) {
    const regexAst = parseRegex(regex);
    const maxNumbersCount = Math.min(getNumberPatternsCount(regexAst), numberPatternsCount || Infinity);
    const expressions = [];
    let numbersCounter = 0;
    let containsNumberPattern = false;
    RegexpTree.traverse(regexAst, visitors({
        every: {
            pre ({ node , parent  }) {
                if (node === regexAst) {
                    return true;
                }
                if (!isExpressionNode(node)) {
                    return false;
                }
                if (parent === regexAst) {
                    containsNumberPattern = false;
                }
                return numbersCounter < maxNumbersCount;
            },
            post ({ node , parent  }) {
                if (node !== regexAst && parent === regexAst && isExpressionNode(node) && (containsNumberPattern || numbersCounter > 0 && numbersCounter < maxNumbersCount)) {
                    expressions.push(node);
                }
            }
        },
        Group (nodePath) {
            if (isNumberPatternNode(nodePath.node) && numbersCounter < maxNumbersCount) {
                containsNumberPattern = true;
                numbersCounter++;
                return false;
            }
            return true;
        }
    }));
    if (expressions.length === 1 && !isNumberPatternNode(expressions[0])) {
        return getNumberPatternsPart(expressions[0], maxNumbersCount);
    }
    return expressions;
}
/**
 * Ranged semver to regex patterns.
 * @param rangedVersion - Ranged semver.
 * @param options - Semver compare options.
 * @returns Array of regex pattern.
 */ function rangedSemverToRegex(rangedVersion, options) {
    const { ignoreMinor , ignorePatch , allowHigherVersions  } = options;
    const ignoreIndex = rangedVersion[0] === Infinity ? 0 : ignoreMinor ? 1 : ignorePatch ? 2 : 3;
    if (allowHigherVersions) {
        const numberPatterns = [];
        let prevWasZero = true;
        let d = 0;
        let start = 0;
        const createMapper = (i)=>(range, j)=>{
                if (j >= ignoreIndex) {
                    return NumberPatternNode();
                }
                start = Array.isArray(range) ? range[0] : range;
                if (j < i) {
                    return NumberCharsNode(start);
                }
                if (j > i) {
                    return NumberPatternNode();
                }
                return rangeToRegex(start + d);
            };
        for(let i = ignoreIndex - 1; i >= 0; i--){
            if (prevWasZero && !rangedVersion[i]) {
                continue;
            }
            prevWasZero = false;
            numberPatterns.push(rangedVersion.map(createMapper(i)));
            d = 1;
        }
        return numberPatterns;
    }
    const numberPatterns1 = rangedVersion.map((range, i)=>{
        if (i >= ignoreIndex) {
            return NumberPatternNode();
        }
        if (Array.isArray(range)) {
            return rangeToRegex(range[0], range[1]);
        }
        return NumberCharsNode(range);
    });
    return [
        numberPatterns1
    ];
}

function applyVersionsToRegex(regex, versions, options) {
    const { allowHigherVersions  } = options;
    const regexAst = parseRegex(regex);
    const finalVersions = allowHigherVersions && versions.length ? [
        versions[0]
    ] : versions;
    const maxRequiredPartsCount = finalVersions.reduce((maxRequiredPartsCount, version)=>Math.max(maxRequiredPartsCount, getRequiredSemverPartsCount(version, options)), 1);
    const numberPatternsPart = getNumberPatternsPart(regexAst, maxRequiredPartsCount);
    const versionsPart = DisjunctionCapturingGroupNode(...finalVersions.map((version)=>rangedSemverToRegex(version, options).map((parts)=>replaceNumberPatterns(AlternativeNode(clone(numberPatternsPart)), parts, maxRequiredPartsCount))));
    RegexpTree.traverse(regexAst, visitors({
        every (nodePath) {
            if (!numberPatternsPart.length) {
                return false;
            }
            if (nodePath.node === numberPatternsPart[0]) {
                if (numberPatternsPart.length === 1) {
                    nodePath.replace(versionsPart);
                } else {
                    nodePath.remove();
                }
                numberPatternsPart.shift();
            }
            return true;
        }
    }));
    return regexAst;
}
/**
 * Apply browser versions to info objects.
 * @param browserRegexes - Objects with requested browser version and regex.
 * @param options - Semver compare options.
 * @returns Objects with requested browser version and regex special for this version.
 */ function applyVersionsToRegexes(browserRegexes, options) {
    return browserRegexes.map(({ regex: sourceRegex , version , maxVersion , matchedVersions , ...other })=>{
        let regexAst = parseRegex(sourceRegex);
        if (!version) {
            regexAst = applyVersionsToRegex(regexAst, versionsListToRanges(matchedVersions), {
                ...options,
                allowHigherVersions: !maxVersion && options.allowHigherVersions
            });
        }
        return {
            regex: null,
            sourceRegex,
            regexAst,
            version,
            maxVersion,
            matchedVersions,
            ...other
        };
    });
}

const defaultOptions = {
    ignoreMinor: false,
    ignorePatch: true,
    allowZeroSubversions: false,
    allowHigherVersions: false
};
/**
 * Get source regexes objects from browserslist query.
 * @param options - Browserslist and semver compare options.
 * @returns Source regexes objects.
 */ function getPreUserAgentRegexes(options = {}) {
    const finalOptions = {
        ...defaultOptions,
        ...options
    };
    const browsersList = getBrowsersList(finalOptions);
    const mergedBrowsers = mergeBrowserVersions(browsersList);
    const sourceRegexes = getRegexesForBrowsers(mergedBrowsers, finalOptions);
    const versionedRegexes = applyVersionsToRegexes(sourceRegexes, finalOptions);
    return versionedRegexes;
}
/**
 * Compile browserslist query to regexes.
 * @param options - Browserslist and semver compare options.
 * @returns Objects with info about compiled regexes.
 */ function getUserAgentRegexes(options = {}) {
    return compileRegexes(getPreUserAgentRegexes(options));
}
/**
 * Compile browserslist query to regex.
 * @param options - Browserslist and semver compare options.
 * @returns Compiled regex.
 */ function getUserAgentRegex(options = {}) {
    return compileRegex(getPreUserAgentRegexes(options));
}

export { AlternativeNode, AstRegExpNode, CapturingGroupNode, CharacterClassNode, ClassRangeNode, DigitPatternNode, DisjunctionCapturingGroupNode, DisjunctionNode, MetaCharNode, NumberCharsNode, NumberPatternNode, RangeQuantifierNode, RepetitionNode, SemverPart, SimpleCharNode, SimpleQuantifierNode, applyVersionsToRegex, applyVersionsToRegexes, clone, compareArrays, compareSemvers, compileRegex, compileRegexes, concat, defaultOptions, findMatchedVersions, getBrowsersList, getNumberPatternsCount, getNumberPatternsPart, getPreUserAgentRegexes, getRegexesForBrowsers, getRequiredSemverPartsCount, getUserAgentRegex, getUserAgentRegexes, isCharNode, isDigitRangeNode, isExpressionNode, isNumberPatternNode, mergeBrowserVersions, mergeDigits, numberToDigits, numbersToRanges, optimizeRegex, optimizeSegmentNumberPatterns, parseBrowsersList, parseRegex, rangeSemver, rangeToRegex, rangedSemverToRegex, rayRangeDigitPattern, rayToNumberPatterns, replaceNumberPatterns, segmentRangeNumberPattern, segmentToNumberPatterns, semverify, splitCommonDiff, splitToDecadeRanges, toRegex, toString, versionsListToRanges, visitors };
//# sourceMappingURL=index.js.map
