const ARGV_START_INDEX = 2;
/**
 * Internal state of arguments.
 */ const argv = process.argv.slice(ARGV_START_INDEX);
/**
 * Reset arguments state.
 */ function resetArgs() {
    argv.splice(0, argv.length);
    argv.push(...process.argv.slice(ARGV_START_INDEX));
}
/**
 * Set arguments state.
 * @param args
 */ function setArgs(...args) {
    argv.splice(0, argv.length);
    argv.push(...args);
}

/**
 * Match reference with argument.
 * @param argRef
 * @param arg
 * @returns Full name or null.
 */ function matchArgName(argRef, arg) {
    if (typeof argRef === "string") {
        return argRef === arg ? argRef : null;
    }
    if (argRef.name === arg) {
        return argRef.name;
    }
    return argRef.aliases.includes(arg) ? argRef.name : null;
}
/**
 * Find full argument name in references.
 * @param argRefs
 * @param arg
 * @returns Found argument's full name or null.
 */ function findArgName(argRefs, arg) {
    let argRef;
    let argName;
    for (argRef of argRefs){
        argName = matchArgName(argRef, arg);
        if (argName) {
            return argName;
        }
    }
    return null;
}

/**
 * Read next argument.
 * Throws error if no next argument.
 * @returns Next argument.
 */ function read() {
    const value = argv.shift();
    if (!value) {
        throw new Error("Unexpected end of arguments");
    }
    return value;
}
/**
 * Expectation of the end.
 * Throws an error if there are more arguments left.
 */ function end() {
    if (argv.length) {
        throw new Error(`Unexpected argument "${argv[0]}"`);
    }
}
/**
 * Expect one of the given arguments.
 * @param argRefs
 * @returns Expected full argument name.
 */ function expect(...argRefs) {
    const arg = read();
    const argName = findArgName(argRefs, arg);
    if (!argName) {
        throw new Error(`Unexpected argument "${arg}"`);
    }
    return argName;
}

/**
 * Describe argument with aliases.
 * @param name - Main name.
 * @param alias - Alias name.
 * @param restAliases - Rest aliases.
 * @returns Description of argument with aliases..
 */ function alias(name, alias, ...restAliases) {
    return {
        name,
        aliases: [
            alias,
            ...restAliases
        ]
    };
}
/**
 * Describe option with value.
 * @param argRef - Option name.
 * @param type - Value type.
 * @returns Option reader.
 */ function option(argRef, type) {
    if (type === String) {
        return (option, read)=>{
            const argName = matchArgName(argRef, option);
            if (argName) {
                return {
                    [argName]: read()
                };
            }
            return null;
        };
    }
    if (type === Number) {
        return (option, read)=>{
            const argName = matchArgName(argRef, option);
            if (argName) {
                return {
                    [argName]: parseFloat(read())
                };
            }
            return null;
        };
    }
    if (type === Array) {
        return (option, read, options)=>{
            const argName = matchArgName(argRef, option);
            if (argName) {
                const prevValues = options[argName];
                const values = read().split(",");
                return {
                    [argName]: Array.isArray(prevValues) ? prevValues.concat(values) : values
                };
            }
            return null;
        };
    }
    return (option)=>{
        const argName = matchArgName(argRef, option);
        if (argName) {
            return {
                [argName]: true
            };
        }
        return null;
    };
}

function isOption(arg) {
    return /^--?[^-].*/.test(arg);
}
function removePrefix(arg) {
    return arg.replace(/^--?/, "");
}
function createOptionReader(optionReaders) {
    const optionReader = optionReaders.reduceRight((readNextOption, readOption)=>{
        if (!readNextOption) {
            return readOption;
        }
        return (option, read, options)=>readOption(option, read, options) ?? readNextOption(option, read, options);
    }, null);
    return optionReader;
}
/**
 * Read options from arguments.
 * @param optionReaders
 * @returns Options with values.
 */ function readOptions(...optionReaders) {
    if (!argv.length) {
        return {};
    }
    const readOption = createOptionReader(optionReaders);
    if (!readOption) {
        return {};
    }
    const options = {};
    let i = 0;
    let arg = argv[i];
    let optionResult;
    const next = ()=>{
        arg = argv[++i];
    };
    const remove = ()=>{
        argv.splice(i--, 1);
    };
    const read = ()=>{
        next();
        remove();
        if (!arg) {
            throw new Error("Unexpected end of arguments");
        }
        return arg;
    };
    // eslint-disable-next-line no-unmodified-loop-condition
    while(arg){
        if (isOption(arg)) {
            optionResult = readOption(removePrefix(arg), read, options);
            if (optionResult) {
                remove();
                Object.assign(options, optionResult);
            }
        }
        next();
    }
    return options;
}

export { alias, argv, end, expect, option, read, readOptions, resetArgs, setArgs };
//# sourceMappingURL=index.js.map
