"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cuint_1 = require("cuint");
const to_buffer_1 = require("./to-buffer");
function isClonable(obj) {
    return obj.hasOwnProperty("clone");
}
class XXHash extends cuint_1.UINT64 {
    /**
     * @param seed unsigned 32-bit integer
     */
    constructor(uintConstructor) {
        super(NaN);
        this.uintConstructor = uintConstructor;
    }
    get vn() {
        return [this.v1, this.v2, this.v3, this.v4];
    }
    getIncrement() {
        return this.size / 4;
    }
    reseed(seed) {
        this.seed = isClonable(seed)
            ? seed.clone()
            : this.uintConstructor(seed);
        this.v1 = this.seed
            .clone()
            .add(this.primes.P1)
            .add(this.primes.P2);
        this.v2 = this.seed.clone().add(this.primes.P2);
        this.v3 = this.seed.clone();
        this.v4 = this.seed.clone().subtract(this.primes.P1);
        this.totalLen = 0;
        this.memsize = 0;
        this.memory = undefined;
    }
    /**
     * Finalize the hash computation. The hash instance is ready for reuse for the given seed
     */
    digest() {
        const m = this.memory;
        if (m === undefined)
            throw new ReferenceError("Hash Memory not set, .update() has to be called before digest()");
        const { P5 } = this.primes;
        const h = this.totalLen >= this.size
            ? this.v1
                .rotl(1)
                .add(this.v2.clone().rotl(7))
                .add(this.v3.clone().rotl(12))
                .add(this.v4.clone().rotl(18))
            : this.seed.clone().add(P5);
        const hash = this.digestCore(m, h);
        // Reset the state
        this.reseed(this.seed);
        return hash;
    }
    /**
     * Add data to be computed for the hash
     */
    update(v) {
        const input = to_buffer_1.default(v);
        const len = input.length;
        if (len === 0)
            return this;
        this.totalLen += len;
        const memory = this.memsize === 0
            ? input instanceof Buffer
                ? new Buffer(this.size)
                : new Uint8Array(this.size)
            : this.memory;
        if (this.memsize + len < this.size) {
            // fill in tmp buffer
            // XXH64_memcpy(memory + this.memsize, input, len)
            if (input instanceof Buffer) {
                input.copy(memory, this.memsize, 0, len);
            }
            else {
                memory.set(input.subarray(0, len), this.memsize);
            }
            this.memsize += len;
            this.memory = memory;
            return this;
        }
        let p = 0;
        const bEnd = p + len;
        const inc = this.getIncrement();
        if (this.memsize > 0) {
            // some data left from previous update
            // XXH64_memcpy(memory + this.memsize, input, 16-this.memsize);
            if (input instanceof Buffer) {
                input.copy(memory, this.memsize, 0, this.size - this.memsize);
            }
            else {
                memory.set(input.subarray(0, this.size - this.memsize), this.memsize);
            }
            let i = 0;
            for (const v of this.vn) {
                this.shiftUpdate(v, memory, i);
                i += inc;
            }
            p += this.size - this.memsize;
            this.memsize = 0;
        }
        if (p <= bEnd - this.size) {
            const limit = bEnd - this.size;
            do {
                for (const v of this.vn) {
                    this.shiftUpdate(v, input, p);
                    p += inc;
                }
            } while (p <= limit);
        }
        if (p < bEnd) {
            // XXH64_memcpy(memory, p, bEnd-p);
            if (input instanceof Buffer) {
                input.copy(memory, this.memsize, p, bEnd);
            }
            else {
                memory.set(input.subarray(p, bEnd), this.memsize);
            }
            this.memsize = bEnd - p;
        }
        this.memory = memory;
        return this;
    }
}
exports.default = XXHash;
//# sourceMappingURL=xxhash.js.map