package org.gonbei774.pocketcheck.ui.screen.settings

import android.app.Activity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.automirrored.filled.KeyboardArrowRight
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TimePicker
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import org.gonbei774.pocketcheck.R
import org.gonbei774.pocketcheck.data.AppLanguage
import org.gonbei774.pocketcheck.data.AppTheme
import org.gonbei774.pocketcheck.data.LanguagePreferences
import org.gonbei774.pocketcheck.data.ThemePreferences
import java.util.Locale

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SettingsScreen(
    viewModel: SettingsViewModel,
    onNavigateBack: () -> Unit,
    onNavigateToArchivedItems: () -> Unit = {},
    onNavigateToLicenses: () -> Unit = {},
    onRequestNotificationPermission: ((Boolean) -> Unit) -> Unit = {},
    onScheduleReminder: (Int) -> Unit = {},
    onCancelReminder: () -> Unit = {},
    onThemeChange: (AppTheme) -> Unit = {},
    modifier: Modifier = Modifier
) {
    val settings by viewModel.settings.collectAsState()
    var showIntervalDialog by remember { mutableStateOf(false) }
    var showQuietStartPicker by remember { mutableStateOf(false) }
    var showQuietEndPicker by remember { mutableStateOf(false) }
    var showLanguageDialog by remember { mutableStateOf(false) }
    var showThemeDialog by remember { mutableStateOf(false) }

    val context = LocalContext.current
    val languagePrefs = remember { LanguagePreferences(context) }
    val themePrefs = remember { ThemePreferences(context) }
    var selectedLanguage by remember { mutableStateOf(languagePrefs.getLanguage()) }
    var selectedTheme by remember { mutableStateOf(themePrefs.getTheme()) }
    val currentLocale = Locale.getDefault().language

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text(stringResource(R.string.settings_title)) },
                navigationIcon = {
                    IconButton(onClick = onNavigateBack) {
                        Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = stringResource(R.string.cd_back))
                    }
                }
            )
        },
        modifier = modifier
    ) { innerPadding ->
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(innerPadding)
                .verticalScroll(rememberScrollState())
        ) {
            // リマインド通知セクション
            Text(
                text = stringResource(R.string.settings_reminder_section),
                style = MaterialTheme.typography.titleSmall,
                color = MaterialTheme.colorScheme.primary,
                modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)
            )

            // リマインド有効/無効
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable {
                        if (!settings.reminderEnabled) {
                            // ONにする場合は権限をチェック
                            onRequestNotificationPermission { granted ->
                                if (granted) {
                                    viewModel.updateReminderEnabled(true)
                                    onScheduleReminder(settings.reminderIntervalMinutes)
                                }
                            }
                        } else {
                            viewModel.updateReminderEnabled(false)
                            onCancelReminder()
                        }
                    }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text(stringResource(R.string.settings_reminder_section))
                    Text(
                        text = stringResource(R.string.settings_reminder_description),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                Switch(
                    checked = settings.reminderEnabled,
                    onCheckedChange = { enabled ->
                        if (enabled) {
                            onRequestNotificationPermission { granted ->
                                if (granted) {
                                    viewModel.updateReminderEnabled(true)
                                    onScheduleReminder(settings.reminderIntervalMinutes)
                                }
                            }
                        } else {
                            viewModel.updateReminderEnabled(false)
                            onCancelReminder()
                        }
                    }
                )
            }

            HorizontalDivider()

            // 通知間隔
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable(enabled = settings.reminderEnabled) { showIntervalDialog = true }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text(
                        text = stringResource(R.string.settings_interval),
                        color = if (settings.reminderEnabled) {
                            MaterialTheme.colorScheme.onSurface
                        } else {
                            MaterialTheme.colorScheme.onSurfaceVariant
                        }
                    )
                    Text(
                        text = formatInterval(settings.reminderIntervalMinutes),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
            }

            HorizontalDivider()

            // 常駐通知
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text(stringResource(R.string.settings_persistent_notification_enabled))
                    Text(
                        text = stringResource(R.string.settings_persistent_notification_description),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                Switch(
                    checked = settings.persistentNotificationEnabled,
                    onCheckedChange = { viewModel.updatePersistentNotificationEnabled(it) }
                )
            }

            HorizontalDivider()

            // 静かな時間帯
            Text(
                text = stringResource(R.string.settings_quiet_time_section),
                style = MaterialTheme.typography.titleSmall,
                color = MaterialTheme.colorScheme.primary,
                modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)
            )

            val quietTimeAvailable = settings.reminderEnabled
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = stringResource(R.string.settings_quiet_time_description),
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.onSurfaceVariant,
                    modifier = Modifier.weight(1f)
                )
                Switch(
                    checked = settings.quietTimeEnabled,
                    onCheckedChange = { viewModel.updateQuietTimeEnabled(it) },
                    enabled = quietTimeAvailable
                )
            }

            HorizontalDivider()

            // 開始時刻
            val quietTimeSettingsEnabled = settings.reminderEnabled && settings.quietTimeEnabled
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable(enabled = quietTimeSettingsEnabled) { showQuietStartPicker = true }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = stringResource(R.string.settings_start),
                    modifier = Modifier.weight(1f),
                    color = if (quietTimeSettingsEnabled) {
                        MaterialTheme.colorScheme.onSurface
                    } else {
                        MaterialTheme.colorScheme.onSurfaceVariant
                    }
                )
                Text(
                    text = formatTime(settings.quietTimeStartHour, settings.quietTimeStartMinute),
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }

            HorizontalDivider()

            // 終了時刻
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable(enabled = quietTimeSettingsEnabled) { showQuietEndPicker = true }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = stringResource(R.string.settings_end),
                    modifier = Modifier.weight(1f),
                    color = if (quietTimeSettingsEnabled) {
                        MaterialTheme.colorScheme.onSurface
                    } else {
                        MaterialTheme.colorScheme.onSurfaceVariant
                    }
                )
                Text(
                    text = formatTime(settings.quietTimeEndHour, settings.quietTimeEndMinute),
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }

            HorizontalDivider()

            // アーカイブセクション
            Text(
                text = stringResource(R.string.settings_data_section),
                style = MaterialTheme.typography.titleSmall,
                color = MaterialTheme.colorScheme.primary,
                modifier = Modifier.padding(horizontal = 16.dp, vertical = 8.dp)
            )

            // アーカイブ済みアイテム
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable { onNavigateToArchivedItems() }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = stringResource(R.string.settings_archived_items),
                    modifier = Modifier.weight(1f)
                )
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowRight,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }

            HorizontalDivider()

            // 言語設定
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable { showLanguageDialog = true }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text(text = stringResource(R.string.settings_language))
                    Text(
                        text = selectedLanguage.getDisplayName(currentLocale),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowRight,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }

            HorizontalDivider()

            // テーマ設定
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable { showThemeDialog = true }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column(modifier = Modifier.weight(1f)) {
                    Text(text = stringResource(R.string.settings_theme))
                    Text(
                        text = selectedTheme.getDisplayName(currentLocale),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowRight,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }

            HorizontalDivider()

            // オープンソースライセンス
            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .clickable { onNavigateToLicenses() }
                    .padding(horizontal = 16.dp, vertical = 12.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = stringResource(R.string.settings_licenses),
                    modifier = Modifier.weight(1f)
                )
                Icon(
                    imageVector = Icons.AutoMirrored.Filled.KeyboardArrowRight,
                    contentDescription = null,
                    tint = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }
        }
    }

    // 言語選択ダイアログ
    if (showLanguageDialog) {
        AlertDialog(
            onDismissRequest = { showLanguageDialog = false },
            title = { Text(text = stringResource(R.string.settings_language)) },
            text = {
                Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
                    AppLanguage.entries.forEach { language ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .clickable {
                                    selectedLanguage = language
                                    languagePrefs.setLanguage(language)
                                    showLanguageDialog = false
                                    (context as? Activity)?.recreate()
                                },
                            colors = CardDefaults.cardColors(
                                containerColor = if (selectedLanguage == language) {
                                    MaterialTheme.colorScheme.primaryContainer
                                } else {
                                    MaterialTheme.colorScheme.surface
                                }
                            )
                        ) {
                            Text(
                                text = language.getDisplayName(currentLocale),
                                modifier = Modifier.padding(16.dp)
                            )
                        }
                    }
                }
            },
            confirmButton = {
                TextButton(onClick = { showLanguageDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }

    // テーマ選択ダイアログ
    if (showThemeDialog) {
        AlertDialog(
            onDismissRequest = { showThemeDialog = false },
            title = { Text(text = stringResource(R.string.settings_theme)) },
            text = {
                Column(verticalArrangement = Arrangement.spacedBy(8.dp)) {
                    AppTheme.entries.forEach { theme ->
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .clickable {
                                    selectedTheme = theme
                                    onThemeChange(theme)
                                    showThemeDialog = false
                                },
                            colors = CardDefaults.cardColors(
                                containerColor = if (selectedTheme == theme) {
                                    MaterialTheme.colorScheme.primaryContainer
                                } else {
                                    MaterialTheme.colorScheme.surface
                                }
                            )
                        ) {
                            Text(
                                text = theme.getDisplayName(currentLocale),
                                modifier = Modifier.padding(16.dp)
                            )
                        }
                    }
                }
            },
            confirmButton = {
                TextButton(onClick = { showThemeDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }

    // 通知間隔入力ダイアログ
    if (showIntervalDialog) {
        IntervalInputDialog(
            currentMinutes = settings.reminderIntervalMinutes,
            onDismiss = { showIntervalDialog = false },
            onConfirm = { newMinutes ->
                viewModel.updateReminderIntervalMinutes(newMinutes)
                onScheduleReminder(newMinutes)
                showIntervalDialog = false
            }
        )
    }

    // 静かな時間帯 開始時刻ピッカー
    if (showQuietStartPicker) {
        TimePickerDialog(
            titleResId = R.string.settings_start_time,
            initialHour = settings.quietTimeStartHour,
            initialMinute = settings.quietTimeStartMinute,
            onDismiss = { showQuietStartPicker = false },
            onConfirm = { hour, minute ->
                viewModel.updateQuietTimeStart(hour, minute)
                showQuietStartPicker = false
            }
        )
    }

    // 静かな時間帯 終了時刻ピッカー
    if (showQuietEndPicker) {
        TimePickerDialog(
            titleResId = R.string.settings_end_time,
            initialHour = settings.quietTimeEndHour,
            initialMinute = settings.quietTimeEndMinute,
            onDismiss = { showQuietEndPicker = false },
            onConfirm = { hour, minute ->
                viewModel.updateQuietTimeEnd(hour, minute)
                showQuietEndPicker = false
            }
        )
    }
}

@Composable
private fun formatInterval(minutes: Int): String {
    val hours = minutes / 60
    val mins = minutes % 60
    return when {
        hours > 0 && mins > 0 -> stringResource(R.string.interval_hours_minutes, hours, mins)
        hours > 0 -> stringResource(R.string.interval_hours, hours)
        else -> stringResource(R.string.interval_minutes, mins)
    }
}

private fun formatTime(hour: Int, minute: Int): String {
    return "%d:%02d".format(hour, minute)
}

@Composable
private fun IntervalInputDialog(
    currentMinutes: Int,
    onDismiss: () -> Unit,
    onConfirm: (Int) -> Unit
) {
    val currentHours = currentMinutes / 60
    val currentMins = currentMinutes % 60

    var hoursText by remember { mutableStateOf(if (currentHours > 0) currentHours.toString() else "") }
    var minutesText by remember { mutableStateOf(if (currentMins > 0 || currentHours == 0) currentMins.toString() else "") }

    val totalMinutes = (hoursText.toIntOrNull() ?: 0) * 60 + (minutesText.toIntOrNull() ?: 0)
    val isValid = totalMinutes >= 15

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.settings_interval)) },
        text = {
            Column {
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.spacedBy(8.dp),
                    modifier = Modifier.fillMaxWidth()
                ) {
                    OutlinedTextField(
                        value = hoursText,
                        onValueChange = { hoursText = it.filter { c -> c.isDigit() }.take(2) },
                        label = { Text(stringResource(R.string.settings_hours)) },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                        singleLine = true,
                        modifier = Modifier.weight(1f)
                    )
                    OutlinedTextField(
                        value = minutesText,
                        onValueChange = { minutesText = it.filter { c -> c.isDigit() }.take(2) },
                        label = { Text(stringResource(R.string.settings_minutes)) },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                        singleLine = true,
                        modifier = Modifier.weight(1f)
                    )
                }
                if (!isValid) {
                    Spacer(modifier = Modifier.height(8.dp))
                    Text(
                        text = stringResource(R.string.settings_min_interval),
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.error
                    )
                }
            }
        },
        confirmButton = {
            TextButton(
                onClick = { onConfirm(totalMinutes) },
                enabled = isValid
            ) {
                Text(stringResource(R.string.ok))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun TimePickerDialog(
    titleResId: Int,
    initialHour: Int,
    initialMinute: Int,
    onDismiss: () -> Unit,
    onConfirm: (Int, Int) -> Unit
) {
    val timePickerState = rememberTimePickerState(
        initialHour = initialHour,
        initialMinute = initialMinute,
        is24Hour = true
    )

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(titleResId)) },
        text = {
            TimePicker(state = timePickerState)
        },
        confirmButton = {
            TextButton(onClick = { onConfirm(timePickerState.hour, timePickerState.minute) }) {
                Text(stringResource(R.string.ok))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}