package org.gonbei774.pocketcheck.ui.screen.itemlist

import android.content.Context
import android.net.Uri
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.data.local.SettingsDataStore
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.data.model.Profile
import org.gonbei774.pocketcheck.data.model.ProfileItemCount
import org.gonbei774.pocketcheck.data.repository.ItemRepository
import org.gonbei774.pocketcheck.data.repository.ProfileRepository
import org.gonbei774.pocketcheck.service.ProfileAlarmScheduler
import org.gonbei774.pocketcheck.util.ImageUtils

@OptIn(ExperimentalCoroutinesApi::class)
class ItemListViewModel(
    private val itemRepository: ItemRepository,
    private val profileRepository: ProfileRepository,
    private val settingsDataStore: SettingsDataStore
) : ViewModel() {

    private val _isLoading = MutableStateFlow(true)
    val isLoading: StateFlow<Boolean> = _isLoading.asStateFlow()

    private val _currentProfileId = MutableStateFlow(1L)
    val currentProfileId: StateFlow<Long> = _currentProfileId.asStateFlow()

    val allProfiles: StateFlow<List<Profile>> = profileRepository.allProfiles
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5000),
            initialValue = emptyList()
        )

    val inUseCountByProfile: StateFlow<List<ProfileItemCount>> = itemRepository.inUseCountByProfile
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5000),
            initialValue = emptyList()
        )

    val items: StateFlow<List<Item>> = _currentProfileId
        .flatMapLatest { profileId ->
            itemRepository.getItemsByProfile(profileId)
        }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5000),
            initialValue = emptyList()
        )

    init {
        viewModelScope.launch {
            // 保存されたプロファイルIDを読み込み
            val savedProfileId = settingsDataStore.lastSelectedProfileId.first()
            // プロファイルが存在するか確認、なければデフォルトにフォールバック
            val profileExists = profileRepository.getProfileById(savedProfileId) != null
            val initialProfileId = if (profileExists) savedProfileId else 1L
            _currentProfileId.value = initialProfileId

            // 初回データ読み込み完了
            itemRepository.getItemsByProfile(initialProfileId).first()
            _isLoading.value = false
        }
    }

    fun switchProfile(profileId: Long) {
        _currentProfileId.value = profileId
        viewModelScope.launch {
            settingsDataStore.updateLastSelectedProfileId(profileId)
        }
    }

    fun createProfile(name: String) {
        viewModelScope.launch {
            val newProfile = profileRepository.insert(name)
            _currentProfileId.value = newProfile.id
            settingsDataStore.updateLastSelectedProfileId(newProfile.id)
        }
    }

    fun renameProfile(profile: Profile, newName: String) {
        viewModelScope.launch {
            profileRepository.update(profile.copy(name = newName))
        }
    }

    fun deleteProfile(profile: Profile) {
        viewModelScope.launch {
            // Switch to default profile if deleting current
            if (_currentProfileId.value == profile.id) {
                _currentProfileId.value = 1L
                settingsDataStore.updateLastSelectedProfileId(1L)
            }
            profileRepository.delete(profile)
        }
    }

    fun updateProfileAlarm(
        context: Context,
        profile: Profile,
        enabled: Boolean,
        hour: Int,
        minute: Int,
        days: Int
    ) {
        viewModelScope.launch {
            profileRepository.updateAlarm(profile.id, enabled, hour, minute, days)
            val updatedProfile = profile.copy(
                alarmEnabled = enabled,
                alarmHour = hour,
                alarmMinute = minute,
                alarmDays = days
            )
            ProfileAlarmScheduler(context).scheduleAlarm(updatedProfile)
        }
    }

    val currentProfile: StateFlow<Profile?> = _currentProfileId
        .flatMapLatest { profileId ->
            profileRepository.getProfileByIdFlow(profileId)
        }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5000),
            initialValue = null
        )

    private val _showAddDialog = MutableStateFlow(false)
    val showAddDialog: StateFlow<Boolean> = _showAddDialog.asStateFlow()

    private val _editingItem = MutableStateFlow<Item?>(null)
    val editingItem: StateFlow<Item?> = _editingItem.asStateFlow()

    // カメラで撮影した画像のURI（プレビュー用）
    private val _capturedImageUri = MutableStateFlow<Uri?>(null)
    val capturedImageUri: StateFlow<Uri?> = _capturedImageUri.asStateFlow()

    // 処理済み画像のパス（保存用）
    private val _pendingImagePath = MutableStateFlow<String?>(null)
    val pendingImagePath: StateFlow<String?> = _pendingImagePath.asStateFlow()

    // 画像を削除するか（編集時、既存画像を消す場合）
    private val _isImageCleared = MutableStateFlow(false)
    val isImageCleared: StateFlow<Boolean> = _isImageCleared.asStateFlow()

    // 削除確認ダイアログ用
    private val _itemToDelete = MutableStateFlow<Item?>(null)
    val itemToDelete: StateFlow<Item?> = _itemToDelete.asStateFlow()

    fun showAddDialog() {
        _editingItem.value = null
        _capturedImageUri.value = null
        _pendingImagePath.value = null
        _isImageCleared.value = false
        _showAddDialog.value = true
    }

    fun showEditDialog(item: Item) {
        _editingItem.value = item
        _capturedImageUri.value = null
        _pendingImagePath.value = null
        _isImageCleared.value = false
        _showAddDialog.value = true
    }

    fun dismissDialog() {
        // ダイアログを閉じるとき、保存されなかった画像を削除
        _pendingImagePath.value?.let { ImageUtils.deleteImage(it) }

        _showAddDialog.value = false
        _editingItem.value = null
        _capturedImageUri.value = null
        _pendingImagePath.value = null
        _isImageCleared.value = false
    }

    fun setCapturedImageUri(uri: Uri?) {
        _capturedImageUri.value = uri
    }

    fun setPendingImagePath(path: String?) {
        // 以前の保留画像があれば削除
        _pendingImagePath.value?.let { ImageUtils.deleteImage(it) }
        _pendingImagePath.value = path
        // 新しい画像を設定したらクリアフラグをリセット
        if (path != null) {
            _isImageCleared.value = false
        }
    }

    fun clearImage() {
        // 新しく撮影した画像をクリア
        _pendingImagePath.value?.let { ImageUtils.deleteImage(it) }
        _pendingImagePath.value = null
        _capturedImageUri.value = null
        // 画像クリアフラグを立てる
        _isImageCleared.value = true
    }

    fun addItem(name: String, location: String, imagePath: String?, profileId: Long = _currentProfileId.value) {
        viewModelScope.launch {
            itemRepository.insert(
                Item(
                    name = name,
                    location = location,
                    imagePath = imagePath,
                    isInUse = true,
                    profileId = profileId
                )
            )
            // 画像は保存されたので、pendingを消す（削除しない）
            _pendingImagePath.value = null
            dismissDialogWithoutCleanup()
        }
    }

    fun updateItem(item: Item, name: String, location: String, newImagePath: String?, profileId: Long = item.profileId) {
        viewModelScope.launch {
            val finalImagePath = if (_isImageCleared.value && newImagePath == null) {
                // 既存画像を削除
                ImageUtils.deleteImage(item.imagePath)
                null
            } else if (newImagePath != null && newImagePath != item.imagePath) {
                // 新しい画像に置き換え、古い画像を削除
                ImageUtils.deleteImage(item.imagePath)
                newImagePath
            } else if (_isImageCleared.value) {
                // 既存画像を削除して新しい画像を使用
                ImageUtils.deleteImage(item.imagePath)
                newImagePath
            } else {
                // 変更なし
                newImagePath ?: item.imagePath
            }

            itemRepository.update(item.copy(name = name, location = location, imagePath = finalImagePath, profileId = profileId))
            // 画像は保存されたので、pendingを消す（削除しない）
            _pendingImagePath.value = null
            dismissDialogWithoutCleanup()
        }
    }

    private fun dismissDialogWithoutCleanup() {
        _showAddDialog.value = false
        _editingItem.value = null
        _capturedImageUri.value = null
        _pendingImagePath.value = null
        _isImageCleared.value = false
    }

    fun deleteItem(item: Item) {
        viewModelScope.launch {
            // アイテム削除時に画像も削除
            ImageUtils.deleteImage(item.imagePath)
            itemRepository.delete(item)
        }
    }

    fun toggleUsageStatus(item: Item) {
        viewModelScope.launch {
            itemRepository.toggleUsageStatus(item)
        }
    }

    fun reorderItems(fromIndex: Int, toIndex: Int) {
        viewModelScope.launch {
            val currentItems = items.value.toMutableList()

            if (fromIndex < 0 || toIndex < 0 ||
                fromIndex >= currentItems.size ||
                toIndex >= currentItems.size
            ) {
                return@launch
            }

            val item = currentItems.removeAt(fromIndex)
            currentItems.add(toIndex, item)

            itemRepository.reorderItems(currentItems.map { it.id })
        }
    }

    fun archiveItem(item: Item) {
        viewModelScope.launch {
            itemRepository.archiveItem(item)
        }
    }

    fun showDeleteConfirmation(item: Item) {
        _itemToDelete.value = item
    }

    fun dismissDeleteConfirmation() {
        _itemToDelete.value = null
    }

    fun confirmDelete() {
        _itemToDelete.value?.let { item ->
            deleteItem(item)
            _itemToDelete.value = null
        }
    }

    companion object {
        fun factory(
            itemRepository: ItemRepository,
            profileRepository: ProfileRepository,
            settingsDataStore: SettingsDataStore
        ): ViewModelProvider.Factory {
            return object : ViewModelProvider.Factory {
                @Suppress("UNCHECKED_CAST")
                override fun <T : ViewModel> create(modelClass: Class<T>): T {
                    return ItemListViewModel(itemRepository, profileRepository, settingsDataStore) as T
                }
            }
        }
    }
}