package org.gonbei774.pocketcheck.ui.component

import android.net.Uri
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material.icons.filled.PhotoCamera
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.ExposedDropdownMenuDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import coil.compose.AsyncImage
import org.gonbei774.pocketcheck.R
import coil.request.CachePolicy
import coil.request.ImageRequest
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.data.model.Profile
import java.io.File

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AddItemDialog(
    editingItem: Item?,
    onDismiss: () -> Unit,
    onConfirm: (name: String, location: String, imagePath: String?, profileId: Long) -> Unit,
    onDelete: ((Item) -> Unit)? = null,
    onCameraClick: () -> Unit = {},
    capturedImageUri: Uri? = null,
    pendingImagePath: String? = null,
    onClearImage: () -> Unit = {},
    isImageCleared: Boolean = false,
    profiles: List<Profile> = emptyList(),
    currentProfileId: Long = 1L
) {
    var name by remember(editingItem) { mutableStateOf(editingItem?.name ?: "") }
    var location by remember(editingItem) { mutableStateOf(editingItem?.location ?: "") }
    var showFullScreenPreview by remember { mutableStateOf(false) }
    var showDeleteConfirmation by remember { mutableStateOf(false) }

    // Profile selection state
    val initialProfileId = editingItem?.profileId ?: currentProfileId
    var selectedProfileId by remember(editingItem, currentProfileId) { mutableStateOf(initialProfileId) }
    var profileDropdownExpanded by remember { mutableStateOf(false) }
    val selectedProfile = profiles.find { it.id == selectedProfileId } ?: profiles.firstOrNull()

    val isEditing = editingItem != null
    val title = stringResource(if (isEditing) R.string.dialog_title_edit else R.string.dialog_title_add)
    val confirmText = stringResource(if (isEditing) R.string.update else R.string.add)

    // 画像の表示用パス（新しい画像 > 既存画像、ただしクリア済みなら既存画像は無視）
    val existingImagePath = if (isImageCleared) null else editingItem?.imagePath
    val displayImagePath = pendingImagePath ?: existingImagePath
    val hasImage = displayImagePath != null || capturedImageUri != null

    // 写真があれば名前は任意、なければ必須
    val canConfirm = name.isNotBlank() || hasImage

    val context = LocalContext.current

    // フルスクリーンプレビューダイアログ
    if (showFullScreenPreview && hasImage) {
        Dialog(
            onDismissRequest = { showFullScreenPreview = false },
            properties = DialogProperties(usePlatformDefaultWidth = false)
        ) {
            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .background(Color.Black.copy(alpha = 0.9f))
                    .clickable { showFullScreenPreview = false },
                contentAlignment = Alignment.Center
            ) {
                val fullScreenImageModel = if (capturedImageUri != null) {
                    ImageRequest.Builder(context)
                        .data(capturedImageUri)
                        .diskCachePolicy(CachePolicy.DISABLED)
                        .memoryCachePolicy(CachePolicy.DISABLED)
                        .build()
                } else {
                    ImageRequest.Builder(context)
                        .data(displayImagePath?.let { File(it) })
                        .diskCachePolicy(CachePolicy.DISABLED)
                        .memoryCachePolicy(CachePolicy.DISABLED)
                        .build()
                }
                AsyncImage(
                    model = fullScreenImageModel,
                    contentDescription = stringResource(R.string.cd_preview),
                    modifier = Modifier
                        .fillMaxSize()
                        .padding(16.dp),
                    contentScale = ContentScale.Fit
                )
            }
        }
    }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(title) },
        text = {
            Column {
                // 画像プレビュー（画像がある場合のみ表示）
                if (hasImage) {
                    Box(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(bottom = 8.dp),
                        contentAlignment = Alignment.Center
                    ) {
                        Box {
                            val imageModel = if (capturedImageUri != null) {
                                ImageRequest.Builder(context)
                                    .data(capturedImageUri)
                                    .diskCachePolicy(CachePolicy.DISABLED)
                                    .memoryCachePolicy(CachePolicy.DISABLED)
                                    .build()
                            } else {
                                ImageRequest.Builder(context)
                                    .data(displayImagePath?.let { File(it) })
                                    .diskCachePolicy(CachePolicy.DISABLED)
                                    .memoryCachePolicy(CachePolicy.DISABLED)
                                    .build()
                            }
                            AsyncImage(
                                model = imageModel,
                                contentDescription = stringResource(R.string.cd_preview),
                                modifier = Modifier
                                    .size(120.dp)
                                    .clip(RoundedCornerShape(8.dp))
                                    .clickable { showFullScreenPreview = true },
                                contentScale = ContentScale.Crop
                            )
                            // 削除ボタン
                            IconButton(
                                onClick = onClearImage,
                                modifier = Modifier
                                    .align(Alignment.TopEnd)
                                    .size(24.dp)
                                    .background(
                                        MaterialTheme.colorScheme.surface.copy(alpha = 0.8f),
                                        RoundedCornerShape(12.dp)
                                    )
                            ) {
                                Icon(
                                    Icons.Default.Clear,
                                    contentDescription = stringResource(R.string.cd_delete_photo),
                                    modifier = Modifier.size(16.dp)
                                )
                            }
                        }
                    }
                }

                // カメラボタン
                OutlinedButton(
                    onClick = onCameraClick,
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Icon(
                        Icons.Default.PhotoCamera,
                        contentDescription = null,
                        modifier = Modifier.size(18.dp)
                    )
                    Spacer(modifier = Modifier.width(8.dp))
                    Text(stringResource(if (hasImage) R.string.retake_photo else R.string.take_photo))
                }

                Spacer(modifier = Modifier.height(12.dp))

                OutlinedTextField(
                    value = name,
                    onValueChange = { name = it },
                    label = { Text(stringResource(if (hasImage) R.string.label_name_optional else R.string.label_name)) },
                    singleLine = true,
                    modifier = Modifier.fillMaxWidth()
                )
                Spacer(modifier = Modifier.height(8.dp))
                OutlinedTextField(
                    value = location,
                    onValueChange = { location = it },
                    label = { Text(stringResource(R.string.label_location)) },
                    singleLine = true,
                    modifier = Modifier.fillMaxWidth()
                )

                // Profile selector (only show if multiple profiles exist)
                if (profiles.size > 1) {
                    Spacer(modifier = Modifier.height(8.dp))
                    ExposedDropdownMenuBox(
                        expanded = profileDropdownExpanded,
                        onExpandedChange = { profileDropdownExpanded = it }
                    ) {
                        OutlinedTextField(
                            value = selectedProfile?.name ?: "",
                            onValueChange = {},
                            readOnly = true,
                            label = { Text(stringResource(R.string.label_list)) },
                            trailingIcon = { ExposedDropdownMenuDefaults.TrailingIcon(expanded = profileDropdownExpanded) },
                            modifier = Modifier
                                .fillMaxWidth()
                                .menuAnchor()
                        )
                        ExposedDropdownMenu(
                            expanded = profileDropdownExpanded,
                            onDismissRequest = { profileDropdownExpanded = false }
                        ) {
                            profiles.forEach { profile ->
                                DropdownMenuItem(
                                    text = { Text(profile.name) },
                                    onClick = {
                                        selectedProfileId = profile.id
                                        profileDropdownExpanded = false
                                    }
                                )
                            }
                        }
                    }
                }
            }
        },
        confirmButton = {
            TextButton(
                onClick = {
                    // クリア済みなら既存画像は使わない
                    val finalImagePath = if (isImageCleared) pendingImagePath else (pendingImagePath ?: editingItem?.imagePath)
                    onConfirm(name.trim(), location.trim(), finalImagePath, selectedProfileId)
                },
                enabled = canConfirm
            ) {
                Text(confirmText)
            }
        },
        dismissButton = {
            if (isEditing && onDelete != null) {
                TextButton(onClick = { showDeleteConfirmation = true }) {
                    Text(stringResource(R.string.delete))
                }
            }
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )

    // 削除確認ダイアログ
    if (showDeleteConfirmation && editingItem != null) {
        AlertDialog(
            onDismissRequest = { showDeleteConfirmation = false },
            title = { Text(stringResource(R.string.delete_confirm_title)) },
            text = {
                Text(
                    stringResource(
                        R.string.delete_confirm_message,
                        editingItem.name.ifBlank { stringResource(R.string.item_no_name) }
                    )
                )
            },
            confirmButton = {
                TextButton(onClick = {
                    onDelete?.invoke(editingItem)
                    showDeleteConfirmation = false
                    onDismiss()
                }) {
                    Text(stringResource(R.string.delete))
                }
            },
            dismissButton = {
                TextButton(onClick = { showDeleteConfirmation = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }
}