package org.gonbei774.pocketcheck.service

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.NotificationCompat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.PocketCheckApplication
import org.gonbei774.pocketcheck.MainActivity
import org.gonbei774.pocketcheck.R
import org.gonbei774.pocketcheck.data.LanguagePreferences
import java.util.Calendar

class ReminderReceiver : BroadcastReceiver() {

    override fun onReceive(context: Context, intent: Intent) {
        when (intent.action) {
            ACTION_SHOW_REMINDER -> {
                val pendingResult = goAsync()
                val intervalMinutes = intent.getIntExtra(ReminderScheduler.EXTRA_INTERVAL_MINUTES, 0)

                CoroutineScope(Dispatchers.IO).launch {
                    try {
                        val application = context.applicationContext as PocketCheckApplication
                        val settings = application.settingsRepository.settings.first()

                        val shouldShow = !settings.quietTimeEnabled || !isInQuietTime(
                            settings.quietTimeStartHour, settings.quietTimeStartMinute,
                            settings.quietTimeEndHour, settings.quietTimeEndMinute
                        )
                        if (shouldShow) {
                            showReminderNotification(context)
                        }

                        // 次のアラームを再スケジュール（setAndAllowWhileIdleは一回限りのため）
                        if (settings.reminderEnabled) {
                            val nextInterval = if (intervalMinutes > 0) intervalMinutes else settings.reminderIntervalMinutes
                            ReminderScheduler(context).scheduleReminder(nextInterval)
                        }
                    } finally {
                        pendingResult.finish()
                    }
                }
            }
            ACTION_DISMISS_REMINDER -> dismissReminderNotification(context)
        }
    }

    private fun showReminderNotification(context: Context) {
        val localizedContext = LanguagePreferences.wrapContext(context)
        val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        // 通知チャンネル作成（Android 8.0+）
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                CHANNEL_ID_REMINDER,
                localizedContext.getString(R.string.channel_reminder_name),
                NotificationManager.IMPORTANCE_DEFAULT
            ).apply {
                description = localizedContext.getString(R.string.channel_reminder_description)
            }
            notificationManager.createNotificationChannel(channel)
        }

        // アプリを開くIntent
        val openIntent = PendingIntent.getActivity(
            context,
            0,
            Intent(context, MainActivity::class.java),
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        // 「ない」ボタンのIntent
        val dismissIntent = PendingIntent.getBroadcast(
            context,
            1,
            Intent(context, ReminderReceiver::class.java).apply {
                action = ACTION_DISMISS_REMINDER
            },
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val notification = NotificationCompat.Builder(context, CHANNEL_ID_REMINDER)
            .setSmallIcon(R.drawable.ic_launcher_foreground)
            .setContentTitle(localizedContext.getString(R.string.notification_reminder_title))
            .setContentText(localizedContext.getString(R.string.notification_reminder_text))
            .setPriority(NotificationCompat.PRIORITY_DEFAULT)
            .setContentIntent(openIntent)
            .setAutoCancel(true)
            .addAction(0, localizedContext.getString(R.string.notification_action_yes), openIntent)
            .addAction(0, localizedContext.getString(R.string.notification_action_no), dismissIntent)
            .build()

        notificationManager.notify(NOTIFICATION_ID_REMINDER, notification)
    }

    private fun dismissReminderNotification(context: Context) {
        val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        notificationManager.cancel(NOTIFICATION_ID_REMINDER)
    }

    companion object {
        const val ACTION_SHOW_REMINDER = "org.gonbei774.pocketcheck.ACTION_SHOW_REMINDER"
        const val ACTION_DISMISS_REMINDER = "org.gonbei774.pocketcheck.ACTION_DISMISS_REMINDER"
        const val CHANNEL_ID_REMINDER = "reminder_channel"
        const val NOTIFICATION_ID_REMINDER = 1

        fun isInQuietTime(startHour: Int, startMinute: Int, endHour: Int, endMinute: Int): Boolean {
            val calendar = Calendar.getInstance()
            val currentHour = calendar.get(Calendar.HOUR_OF_DAY)
            val currentMinute = calendar.get(Calendar.MINUTE)

            val currentTime = currentHour * 60 + currentMinute
            val startTime = startHour * 60 + startMinute
            val endTime = endHour * 60 + endMinute

            return if (startTime <= endTime) {
                currentTime in startTime until endTime
            } else {
                currentTime >= startTime || currentTime < endTime
            }
        }
    }
}