package org.gonbei774.pocketcheck.data.repository

import kotlinx.coroutines.flow.Flow
import org.gonbei774.pocketcheck.data.local.ItemDao
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.data.model.ProfileItemCount

class ItemRepository(private val itemDao: ItemDao) {

    val allItems: Flow<List<Item>> = itemDao.getAllItems()

    val itemsInUse: Flow<List<Item>> = itemDao.getItemsInUse()

    val archivedItems: Flow<List<Item>> = itemDao.getArchivedItems()

    val inUseCountByProfile: Flow<List<ProfileItemCount>> = itemDao.getInUseCountByProfile()

    fun getItemsByProfile(profileId: Long): Flow<List<Item>> {
        return itemDao.getItemsByProfile(profileId)
    }

    fun getArchivedItemsByProfile(profileId: Long): Flow<List<Item>> {
        return itemDao.getArchivedItemsByProfile(profileId)
    }

    suspend fun getItemById(id: Long): Item? {
        return itemDao.getItemById(id)
    }

    suspend fun insert(item: Item): Long {
        // 新しいアイテムはそのプロファイル内で最後に追加
        val maxOrder = itemDao.getMaxDisplayOrderByProfile(item.profileId) ?: -1
        val itemWithOrder = item.copy(displayOrder = maxOrder + 1)
        return itemDao.insert(itemWithOrder)
    }

    suspend fun update(item: Item) {
        itemDao.update(item)
    }

    suspend fun delete(item: Item) {
        itemDao.delete(item)
    }

    suspend fun toggleUsageStatus(item: Item) {
        val newStatus = !item.isInUse
        val lastUsedAt = if (newStatus) System.currentTimeMillis() else null
        itemDao.updateUsageStatus(item.id, newStatus, lastUsedAt)
    }

    suspend fun reorderItems(itemIds: List<Long>) {
        itemIds.forEachIndexed { index, id ->
            itemDao.updateDisplayOrder(id, index)
        }
    }

    suspend fun archiveItem(item: Item) {
        itemDao.updateArchiveStatus(item.id, true)
    }

    suspend fun unarchiveItem(item: Item) {
        itemDao.updateArchiveStatus(item.id, false)
    }
}