package org.gonbei774.pocketcheck

import android.Manifest
import android.app.Application
import android.content.Context
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.os.Build
import androidx.core.content.ContextCompat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.data.AppLanguage
import org.gonbei774.pocketcheck.data.LanguagePreferences
import org.gonbei774.pocketcheck.data.local.AppDatabase
import org.gonbei774.pocketcheck.data.local.SettingsDataStore
import org.gonbei774.pocketcheck.data.repository.ItemRepository
import org.gonbei774.pocketcheck.data.repository.ProfileRepository
import org.gonbei774.pocketcheck.data.repository.SettingsRepository
import org.gonbei774.pocketcheck.service.ReminderReceiver
import org.gonbei774.pocketcheck.service.StatusNotificationService
import java.util.Locale

class PocketCheckApplication : Application() {

    private val applicationScope = CoroutineScope(SupervisorJob() + Dispatchers.Main)

    val database by lazy { AppDatabase.getDatabase(this) }
    val itemRepository by lazy { ItemRepository(database.itemDao()) }
    val profileRepository by lazy { ProfileRepository(database.profileDao(), database.itemDao()) }
    val settingsDataStore by lazy { SettingsDataStore(this) }
    val settingsRepository by lazy { SettingsRepository(settingsDataStore) }

    override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(updateBaseContextLocale(newBase))
    }

    private fun updateBaseContextLocale(context: Context): Context {
        val languagePrefs = LanguagePreferences(context)
        val selectedLanguage = languagePrefs.getLanguage()

        if (selectedLanguage == AppLanguage.SYSTEM) {
            return context
        }

        val locale = when (selectedLanguage) {
            AppLanguage.JAPANESE -> Locale("ja")
            AppLanguage.ENGLISH -> Locale("en")
            AppLanguage.SYSTEM -> return context
        }

        Locale.setDefault(locale)

        val config = Configuration(context.resources.configuration)
        config.setLocale(locale)

        return context.createConfigurationContext(config)
    }

    override fun onCreate() {
        super.onCreate()
        observeItemsInUse()
    }

    private fun observeItemsInUse() {
        applicationScope.launch {
            combine(
                itemRepository.itemsInUse,
                settingsRepository.settings
            ) { items, settings ->
                Pair(items, settings)
            }.collectLatest { (items, settings) ->
                val shouldShow = items.isNotEmpty() &&
                    hasNotificationPermission() &&
                    settings.persistentNotificationEnabled &&
                    !shouldHideForQuietTime(settings)

                if (shouldShow) {
                    StatusNotificationService.start(this@PocketCheckApplication)
                } else {
                    StatusNotificationService.stop(this@PocketCheckApplication)
                }
            }
        }
    }

    private fun shouldHideForQuietTime(settings: org.gonbei774.pocketcheck.data.model.Settings): Boolean {
        if (!settings.quietTimeEnabled) {
            return false
        }
        return ReminderReceiver.isInQuietTime(
            settings.quietTimeStartHour,
            settings.quietTimeStartMinute,
            settings.quietTimeEndHour,
            settings.quietTimeEndMinute
        )
    }

    fun hasNotificationPermission(): Boolean {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.POST_NOTIFICATIONS
            ) == PackageManager.PERMISSION_GRANTED
        } else {
            true
        }
    }
}