package org.gonbei774.pocketcheck.util

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.net.Uri
import android.util.Log
import androidx.exifinterface.media.ExifInterface
import org.gonbei774.pocketcheck.BuildConfig
import java.io.File
import java.io.FileOutputStream
import java.util.UUID

object ImageUtils {
    private const val MAX_SIZE = 1024
    private const val JPEG_QUALITY = 80
    private const val IMAGE_DIR = "item_images"

    /**
     * URIから画像を読み込み、圧縮してアプリ内部ストレージに保存
     * @return 保存したファイルの絶対パス
     */
    fun saveCompressedImage(context: Context, uri: Uri): String? {
        return try {
            val inputStream = context.contentResolver.openInputStream(uri) ?: return null

            // EXIF情報から回転角度を取得
            val rotation = getRotationFromUri(context, uri)

            // 元画像のサイズを取得
            val options = BitmapFactory.Options().apply {
                inJustDecodeBounds = true
            }
            BitmapFactory.decodeStream(inputStream, null, options)
            inputStream.close()

            // サンプルサイズを計算（メモリ効率のため）
            val sampleSize = calculateSampleSize(options.outWidth, options.outHeight, MAX_SIZE)

            // 実際に画像を読み込み
            val decodeOptions = BitmapFactory.Options().apply {
                inSampleSize = sampleSize
            }
            val newInputStream = context.contentResolver.openInputStream(uri) ?: return null
            val bitmap = BitmapFactory.decodeStream(newInputStream, null, decodeOptions)
            newInputStream.close()

            if (bitmap == null) return null

            // 回転を適用
            val rotatedBitmap = rotateBitmap(bitmap, rotation)

            // リサイズ
            val resizedBitmap = resizeBitmap(rotatedBitmap, MAX_SIZE)
            if (rotatedBitmap != resizedBitmap) {
                rotatedBitmap.recycle()
            }

            // 保存先ディレクトリを作成
            val imageDir = File(context.filesDir, IMAGE_DIR)
            if (!imageDir.exists()) {
                imageDir.mkdirs()
            }

            // ファイル名を生成して保存
            val fileName = "${UUID.randomUUID()}.jpg"
            val outputFile = File(imageDir, fileName)
            FileOutputStream(outputFile).use { fos ->
                resizedBitmap.compress(Bitmap.CompressFormat.JPEG, JPEG_QUALITY, fos)
            }
            resizedBitmap.recycle()

            outputFile.absolutePath
        } catch (e: Exception) {
            if (BuildConfig.DEBUG) {
                Log.e("ImageUtils", "saveCompressedImage failed", e)
            }
            null
        }
    }

    /**
     * 画像ファイルを削除
     */
    fun deleteImage(imagePath: String?): Boolean {
        if (imagePath.isNullOrBlank()) return false
        return try {
            val file = File(imagePath)
            if (file.exists()) {
                file.delete()
            } else {
                false
            }
        } catch (e: Exception) {
            if (BuildConfig.DEBUG) {
                Log.e("ImageUtils", "deleteImage failed", e)
            }
            false
        }
    }

    /**
     * 画像ファイルが存在するか確認
     */
    fun imageExists(imagePath: String?): Boolean {
        if (imagePath.isNullOrBlank()) return false
        return File(imagePath).exists()
    }

    private fun getRotationFromUri(context: Context, uri: Uri): Int {
        return try {
            val inputStream = context.contentResolver.openInputStream(uri) ?: return 0
            val exif = ExifInterface(inputStream)
            inputStream.close()

            when (exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL)) {
                ExifInterface.ORIENTATION_ROTATE_90 -> 90
                ExifInterface.ORIENTATION_ROTATE_180 -> 180
                ExifInterface.ORIENTATION_ROTATE_270 -> 270
                else -> 0
            }
        } catch (e: Exception) {
            0
        }
    }

    private fun calculateSampleSize(width: Int, height: Int, maxSize: Int): Int {
        var sampleSize = 1
        val largerDimension = maxOf(width, height)
        while (largerDimension / sampleSize > maxSize * 2) {
            sampleSize *= 2
        }
        return sampleSize
    }

    private fun rotateBitmap(bitmap: Bitmap, degrees: Int): Bitmap {
        if (degrees == 0) return bitmap
        val matrix = Matrix().apply {
            postRotate(degrees.toFloat())
        }
        return Bitmap.createBitmap(bitmap, 0, 0, bitmap.width, bitmap.height, matrix, true)
    }

    private fun resizeBitmap(bitmap: Bitmap, maxSize: Int): Bitmap {
        val width = bitmap.width
        val height = bitmap.height

        if (width <= maxSize && height <= maxSize) {
            return bitmap
        }

        val ratio = minOf(maxSize.toFloat() / width, maxSize.toFloat() / height)
        val newWidth = (width * ratio).toInt()
        val newHeight = (height * ratio).toInt()

        return Bitmap.createScaledBitmap(bitmap, newWidth, newHeight, true)
    }
}