package org.gonbei774.pocketcheck.ui.component

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.border
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.ui.draw.clip
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Alarm
import androidx.compose.material.icons.filled.AlarmOff
import androidx.compose.material.icons.filled.Check
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalDrawerSheet
import androidx.compose.material3.NavigationDrawerItem
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TimePicker
import androidx.compose.material3.rememberTimePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.platform.LocalContext
import org.gonbei774.pocketcheck.R
import org.gonbei774.pocketcheck.data.model.Profile
import org.gonbei774.pocketcheck.data.model.ProfileItemCount
import org.gonbei774.pocketcheck.util.AlarmPermissionHelper

@Composable
fun AppDrawer(
    profiles: List<Profile>,
    currentProfileId: Long,
    inUseCountByProfile: List<ProfileItemCount>,
    onProfileSelect: (Profile) -> Unit,
    onCreateProfile: (String) -> Unit,
    onRenameProfile: (Profile, String) -> Unit,
    onDeleteProfile: (Profile) -> Unit,
    onUpdateAlarm: (Profile, Boolean, Int, Int, Int) -> Unit,
    onRequestAlarmPermission: () -> Unit,
    modifier: Modifier = Modifier
) {
    val context = LocalContext.current
    var showCreateDialog by remember { mutableStateOf(false) }
    var profileToEdit by remember { mutableStateOf<Profile?>(null) }
    var profileToDelete by remember { mutableStateOf<Profile?>(null) }
    var contextMenuProfile by remember { mutableStateOf<Profile?>(null) }
    var profileForAlarm by remember { mutableStateOf<Profile?>(null) }
    var showPermissionDialog by remember { mutableStateOf(false) }

    // Convert inUseCountByProfile to a map for easy lookup
    val inUseCountMap: Map<Long, Int> = remember(inUseCountByProfile) {
        inUseCountByProfile.associate { it.profileId to it.count }
    }

    ModalDrawerSheet(modifier = modifier) {
        Column(
            modifier = Modifier.fillMaxHeight()
        ) {
            Spacer(modifier = Modifier.height(16.dp))

            // Profile list
            LazyColumn(
                modifier = Modifier.weight(1f)
            ) {
                items(profiles, key = { it.id }) { profile ->
                    val isSelected = profile.id == currentProfileId
                    val inUseCount = inUseCountMap[profile.id] ?: 0

                    Box(
                        modifier = Modifier
                            .fillMaxWidth()
                            .pointerInput(profile) {
                                detectTapGestures(
                                    onLongPress = { contextMenuProfile = profile }
                                )
                            }
                    ) {
                        NavigationDrawerItem(
                            label = {
                                Row(
                                    modifier = Modifier.fillMaxWidth(),
                                    horizontalArrangement = Arrangement.SpaceBetween,
                                    verticalAlignment = Alignment.CenterVertically
                                ) {
                                    Text(
                                        text = profile.name,
                                        modifier = Modifier.weight(1f)
                                    )
                                    Row(
                                        verticalAlignment = Alignment.CenterVertically,
                                        horizontalArrangement = Arrangement.spacedBy(4.dp)
                                    ) {
                                        if (inUseCount > 0) {
                                            Text(
                                                text = inUseCount.toString(),
                                                style = MaterialTheme.typography.bodyMedium,
                                                color = MaterialTheme.colorScheme.onSurfaceVariant
                                            )
                                        }
                                        IconButton(
                                            onClick = {
                                                // Check permission every time button is clicked
                                                if (AlarmPermissionHelper.canScheduleExactAlarms(context)) {
                                                    profileForAlarm = profile
                                                } else {
                                                    showPermissionDialog = true
                                                }
                                            },
                                            modifier = Modifier.size(32.dp)
                                        ) {
                                            Icon(
                                                imageVector = if (profile.alarmEnabled) Icons.Filled.Alarm else Icons.Filled.AlarmOff,
                                                contentDescription = stringResource(R.string.profile_alarm_settings),
                                                tint = if (profile.alarmEnabled) {
                                                    MaterialTheme.colorScheme.primary
                                                } else {
                                                    MaterialTheme.colorScheme.onSurfaceVariant
                                                },
                                                modifier = Modifier.size(20.dp)
                                            )
                                        }
                                    }
                                }
                            },
                            selected = isSelected,
                            onClick = { onProfileSelect(profile) },
                            icon = if (isSelected) {
                                {
                                    Icon(
                                        Icons.Default.Check,
                                        contentDescription = null
                                    )
                                }
                            } else null,
                            modifier = Modifier.padding(horizontal = 12.dp)
                        )

                        // Context menu for long press
                        DropdownMenu(
                            expanded = contextMenuProfile == profile,
                            onDismissRequest = { contextMenuProfile = null }
                        ) {
                            DropdownMenuItem(
                                text = { Text(stringResource(R.string.profile_rename)) },
                                onClick = {
                                    profileToEdit = profile
                                    contextMenuProfile = null
                                }
                            )
                            if (!profile.isDefault) {
                                DropdownMenuItem(
                                    text = { Text(stringResource(R.string.delete)) },
                                    onClick = {
                                        profileToDelete = profile
                                        contextMenuProfile = null
                                    }
                                )
                            }
                        }
                    }
                }
            }

            // Divider and Create button
            HorizontalDivider(modifier = Modifier.padding(horizontal = 16.dp))

            NavigationDrawerItem(
                label = { Text(stringResource(R.string.profile_create_new)) },
                selected = false,
                onClick = { showCreateDialog = true },
                icon = {
                    Icon(
                        Icons.Default.Add,
                        contentDescription = null
                    )
                },
                modifier = Modifier.padding(horizontal = 12.dp, vertical = 8.dp)
            )
        }
    }

    // Create profile dialog
    if (showCreateDialog) {
        ProfileNameDialog(
            title = stringResource(R.string.profile_create_title),
            initialName = "",
            confirmText = stringResource(R.string.profile_create_button),
            onDismiss = { showCreateDialog = false },
            onConfirm = { name ->
                onCreateProfile(name)
                showCreateDialog = false
            }
        )
    }

    // Rename profile dialog
    if (profileToEdit != null) {
        ProfileNameDialog(
            title = stringResource(R.string.profile_rename_title),
            initialName = profileToEdit!!.name,
            confirmText = stringResource(R.string.update),
            onDismiss = { profileToEdit = null },
            onConfirm = { name ->
                onRenameProfile(profileToEdit!!, name)
                profileToEdit = null
            }
        )
    }

    // Delete confirmation dialog
    if (profileToDelete != null) {
        AlertDialog(
            onDismissRequest = { profileToDelete = null },
            title = { Text(stringResource(R.string.profile_delete_title)) },
            text = {
                Text(stringResource(R.string.profile_delete_message, profileToDelete!!.name))
            },
            confirmButton = {
                TextButton(onClick = {
                    onDeleteProfile(profileToDelete!!)
                    profileToDelete = null
                }) {
                    Text(stringResource(R.string.delete))
                }
            },
            dismissButton = {
                TextButton(onClick = { profileToDelete = null }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }

    // Alarm settings dialog
    if (profileForAlarm != null) {
        ProfileAlarmDialog(
            profile = profileForAlarm!!,
            onDismiss = { profileForAlarm = null },
            onConfirm = { enabled, hour, minute, days ->
                onUpdateAlarm(profileForAlarm!!, enabled, hour, minute, days)
                profileForAlarm = null
            }
        )
    }

    // Permission request dialog
    if (showPermissionDialog) {
        AlertDialog(
            onDismissRequest = { showPermissionDialog = false },
            title = { Text(stringResource(R.string.alarm_permission_required_title)) },
            text = { Text(stringResource(R.string.alarm_permission_required_message)) },
            confirmButton = {
                TextButton(onClick = {
                    onRequestAlarmPermission()
                    showPermissionDialog = false
                }) {
                    Text(stringResource(R.string.open_settings))
                }
            },
            dismissButton = {
                TextButton(onClick = { showPermissionDialog = false }) {
                    Text(stringResource(R.string.cancel))
                }
            }
        )
    }
}

@Composable
private fun ProfileNameDialog(
    title: String,
    initialName: String,
    confirmText: String,
    onDismiss: () -> Unit,
    onConfirm: (String) -> Unit
) {
    var name by remember { mutableStateOf(initialName) }

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(title) },
        text = {
            OutlinedTextField(
                value = name,
                onValueChange = { name = it },
                label = { Text(stringResource(R.string.profile_name_label)) },
                singleLine = true,
                modifier = Modifier.fillMaxWidth()
            )
        },
        confirmButton = {
            TextButton(
                onClick = { onConfirm(name.trim()) },
                enabled = name.isNotBlank()
            ) {
                Text(confirmText)
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun ProfileAlarmDialog(
    profile: Profile,
    onDismiss: () -> Unit,
    onConfirm: (enabled: Boolean, hour: Int, minute: Int, days: Int) -> Unit
) {
    var enabled by remember { mutableStateOf(profile.alarmEnabled) }
    var selectedDays by remember { mutableIntStateOf(profile.alarmDays) }
    val timePickerState = rememberTimePickerState(
        initialHour = profile.alarmHour,
        initialMinute = profile.alarmMinute,
        is24Hour = true
    )

    val dayLabels = listOf(
        stringResource(R.string.day_sun),
        stringResource(R.string.day_mon),
        stringResource(R.string.day_tue),
        stringResource(R.string.day_wed),
        stringResource(R.string.day_thu),
        stringResource(R.string.day_fri),
        stringResource(R.string.day_sat)
    )

    AlertDialog(
        onDismissRequest = onDismiss,
        title = { Text(stringResource(R.string.profile_alarm_title, profile.name)) },
        text = {
            Column(
                modifier = Modifier.fillMaxWidth(),
                verticalArrangement = Arrangement.spacedBy(16.dp)
            ) {
                // ON/OFF switch
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.SpaceBetween,
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Text(stringResource(R.string.profile_alarm_enabled))
                    Switch(
                        checked = enabled,
                        onCheckedChange = { enabled = it }
                    )
                }

                if (enabled) {
                    // Time picker
                    TimePicker(state = timePickerState)

                    // Day of week selector
                    Text(
                        text = stringResource(R.string.profile_alarm_days),
                        style = MaterialTheme.typography.bodyMedium
                    )
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.SpaceEvenly
                    ) {
                        dayLabels.forEachIndexed { index, day ->
                            val bit = 1 shl index
                            val isSelected = (selectedDays and bit) != 0
                            DayToggleCircle(
                                label = day,
                                selected = isSelected,
                                onClick = {
                                    selectedDays = if (isSelected) {
                                        selectedDays and bit.inv()
                                    } else {
                                        selectedDays or bit
                                    }
                                }
                            )
                        }
                    }
                }
            }
        },
        confirmButton = {
            TextButton(
                onClick = {
                    onConfirm(
                        enabled,
                        timePickerState.hour,
                        timePickerState.minute,
                        selectedDays
                    )
                }
            ) {
                Text(stringResource(R.string.ok))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }
    )
}

@Composable
private fun DayToggleCircle(
    label: String,
    selected: Boolean,
    onClick: () -> Unit
) {
    val backgroundColor = if (selected) {
        MaterialTheme.colorScheme.primary
    } else {
        MaterialTheme.colorScheme.surface
    }
    val textColor = if (selected) {
        MaterialTheme.colorScheme.onPrimary
    } else {
        MaterialTheme.colorScheme.onSurface
    }
    val borderColor = if (selected) {
        MaterialTheme.colorScheme.primary
    } else {
        MaterialTheme.colorScheme.outline
    }

    Box(
        modifier = Modifier
            .size(36.dp)
            .clip(CircleShape)
            .background(backgroundColor, CircleShape)
            .border(1.dp, borderColor, CircleShape)
            .clickable(onClick = onClick),
        contentAlignment = Alignment.Center
    ) {
        Text(
            text = label,
            style = MaterialTheme.typography.bodyMedium,
            color = textColor,
            textAlign = TextAlign.Center
        )
    }
}