package org.gonbei774.pocketcheck.service

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.app.Service
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.IBinder
import androidx.core.app.NotificationCompat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.PocketCheckApplication
import org.gonbei774.pocketcheck.MainActivity
import org.gonbei774.pocketcheck.R
import org.gonbei774.pocketcheck.data.LanguagePreferences
import org.gonbei774.pocketcheck.data.model.Item

class StatusNotificationService : Service() {

    private val serviceJob = Job()
    private val serviceScope = CoroutineScope(Dispatchers.Main + serviceJob)
    private var observeJob: Job? = null
    private var isRunning = false

    override fun onBind(intent: Intent?): IBinder? = null

    override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(LanguagePreferences.wrapContext(newBase))
    }

    override fun onCreate() {
        super.onCreate()
        createNotificationChannel()
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        when (intent?.action) {
            ACTION_START -> startObserving()
            ACTION_STOP -> stopSelf()
        }
        return START_STICKY
    }

    private fun startObserving() {
        // 既に実行中なら何もしない（監視は継続中）
        if (isRunning) {
            return
        }
        isRunning = true

        val application = application as PocketCheckApplication

        // 初期通知を表示してForeground Serviceを開始（読み込み中...）
        startForeground(NOTIFICATION_ID_STATUS, createNotification(null))

        observeJob?.cancel()
        observeJob = serviceScope.launch {
            application.itemRepository.itemsInUse.collectLatest { items ->
                if (items.isEmpty()) {
                    stopSelf()
                } else {
                    updateNotification(items)
                }
            }
        }
    }

    private fun createNotificationChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                CHANNEL_ID_STATUS,
                getString(R.string.channel_status_name),
                NotificationManager.IMPORTANCE_LOW
            ).apply {
                description = getString(R.string.channel_status_description)
                setSound(null, null)
                enableVibration(false)
            }
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.createNotificationChannel(channel)
        }
    }

    private fun createNotification(items: List<Item>?): Notification {
        val openIntent = PendingIntent.getActivity(
            this,
            0,
            Intent(this, MainActivity::class.java),
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val contentText = when {
            items == null -> getString(R.string.notification_status_loading)
            items.isEmpty() -> getString(R.string.notification_status_empty)
            else -> getString(R.string.notification_status_in_use, items.size)
        }

        return NotificationCompat.Builder(this, CHANNEL_ID_STATUS)
            .setSmallIcon(R.drawable.ic_launcher_foreground)
            .setContentTitle(getString(R.string.notification_status_title))
            .setContentText(contentText)
            .setOngoing(true)
            .setSilent(true)
            .setPriority(NotificationCompat.PRIORITY_LOW)
            .setContentIntent(openIntent)
            .build()
    }

    private fun updateNotification(items: List<Item>) {
        val notificationManager = getSystemService(NotificationManager::class.java)
        notificationManager.notify(NOTIFICATION_ID_STATUS, createNotification(items))
    }

    override fun onDestroy() {
        isRunning = false
        serviceJob.cancel()
        super.onDestroy()
    }

    companion object {
        const val ACTION_START = "org.gonbei774.pocketcheck.ACTION_START_STATUS"
        const val ACTION_STOP = "org.gonbei774.pocketcheck.ACTION_STOP_STATUS"
        const val CHANNEL_ID_STATUS = "status_channel"
        const val NOTIFICATION_ID_STATUS = 2

        fun start(context: Context) {
            val intent = Intent(context, StatusNotificationService::class.java).apply {
                action = ACTION_START
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                context.startForegroundService(intent)
            } else {
                context.startService(intent)
            }
        }

        fun stop(context: Context) {
            val intent = Intent(context, StatusNotificationService::class.java).apply {
                action = ACTION_STOP
            }
            context.stopService(intent)
        }
    }
}