package org.gonbei774.pocketcheck.service

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.PocketCheckApplication

/**
 * 端末再起動後にリマインドと常駐通知を復活させるReceiver
 */
class BootReceiver : BroadcastReceiver() {

    override fun onReceive(context: Context, intent: Intent) {
        if (intent.action != Intent.ACTION_BOOT_COMPLETED) return

        val pendingResult = goAsync()
        val application = context.applicationContext as PocketCheckApplication

        CoroutineScope(Dispatchers.IO).launch {
            try {
                // リマインド通知のスケジュールを復活
                val settings = application.settingsRepository.settings.first()
                if (settings.reminderEnabled) {
                    val scheduler = ReminderScheduler(context)
                    scheduler.scheduleReminder(settings.reminderIntervalMinutes)
                }

                // 使用中アイテムがあり、通知権限があり、設定が有効な場合のみ常駐通知を復活
                val itemsInUse = application.itemRepository.itemsInUse.first()
                val isInQuietTime = settings.quietTimeEnabled &&
                    ReminderReceiver.isInQuietTime(
                        settings.quietTimeStartHour,
                        settings.quietTimeStartMinute,
                        settings.quietTimeEndHour,
                        settings.quietTimeEndMinute
                    )
                val shouldShowPersistent = itemsInUse.isNotEmpty() &&
                    application.hasNotificationPermission() &&
                    settings.persistentNotificationEnabled &&
                    !isInQuietTime

                if (shouldShowPersistent) {
                    StatusNotificationService.start(context)
                }

                // プロファイルアラームを再スケジュール
                ProfileAlarmScheduler(context).rescheduleAllAlarms()
            } finally {
                pendingResult.finish()
            }
        }
    }
}