package org.gonbei774.pocketcheck.data.local

import androidx.room.Dao
import androidx.room.Delete
import androidx.room.Insert
import androidx.room.Query
import androidx.room.Update
import kotlinx.coroutines.flow.Flow
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.data.model.ProfileItemCount

@Dao
interface ItemDao {
    @Query("SELECT * FROM items WHERE isArchived = 0 ORDER BY displayOrder ASC")
    fun getAllItems(): Flow<List<Item>>

    @Query("SELECT * FROM items WHERE isInUse = 1 AND isArchived = 0")
    fun getItemsInUse(): Flow<List<Item>>

    @Query("SELECT * FROM items WHERE isArchived = 1 ORDER BY displayOrder ASC")
    fun getArchivedItems(): Flow<List<Item>>

    @Query("SELECT * FROM items WHERE profileId = :profileId AND isArchived = 0 ORDER BY displayOrder ASC")
    fun getItemsByProfile(profileId: Long): Flow<List<Item>>

    @Query("SELECT * FROM items WHERE profileId = :profileId AND isArchived = 1 ORDER BY displayOrder ASC")
    fun getArchivedItemsByProfile(profileId: Long): Flow<List<Item>>

    @Query("SELECT profileId, COUNT(*) as count FROM items WHERE isInUse = 1 AND isArchived = 0 GROUP BY profileId")
    fun getInUseCountByProfile(): Flow<List<ProfileItemCount>>

    @Query("SELECT MAX(displayOrder) FROM items WHERE profileId = :profileId")
    suspend fun getMaxDisplayOrderByProfile(profileId: Long): Int?

    @Query("SELECT * FROM items WHERE id = :id")
    suspend fun getItemById(id: Long): Item?

    @Insert
    suspend fun insert(item: Item): Long

    @Update
    suspend fun update(item: Item)

    @Delete
    suspend fun delete(item: Item)

    @Query("UPDATE items SET isInUse = :isInUse, lastUsedAt = :lastUsedAt WHERE id = :id")
    suspend fun updateUsageStatus(id: Long, isInUse: Boolean, lastUsedAt: Long?)

    @Query("UPDATE items SET displayOrder = :displayOrder WHERE id = :id")
    suspend fun updateDisplayOrder(id: Long, displayOrder: Int)

    @Query("SELECT MAX(displayOrder) FROM items")
    suspend fun getMaxDisplayOrder(): Int?

    @Query("UPDATE items SET isArchived = :isArchived, isInUse = 0 WHERE id = :id")
    suspend fun updateArchiveStatus(id: Long, isArchived: Boolean)

    @Query("DELETE FROM items WHERE profileId = :profileId")
    suspend fun deleteItemsByProfile(profileId: Long)
}