package org.gonbei774.pocketcheck.data.local

import android.content.Context
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.data.model.Profile

@Database(entities = [Item::class, Profile::class], version = 6, exportSchema = false)
abstract class AppDatabase : RoomDatabase() {
    abstract fun itemDao(): ItemDao
    abstract fun profileDao(): ProfileDao

    companion object {
        @Volatile
        private var INSTANCE: AppDatabase? = null

        private val MIGRATION_1_2 = object : Migration(1, 2) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE items ADD COLUMN imagePath TEXT")
            }
        }

        private val MIGRATION_2_3 = object : Migration(2, 3) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE items ADD COLUMN displayOrder INTEGER NOT NULL DEFAULT 0")
                // 既存アイテムにdisplayOrderを設定（id順）
                db.execSQL(
                    """
                    UPDATE items SET displayOrder = (
                        SELECT COUNT(*) FROM items AS i2 WHERE i2.id < items.id
                    )
                    """.trimIndent()
                )
            }
        }

        private val MIGRATION_3_4 = object : Migration(3, 4) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE items ADD COLUMN isArchived INTEGER NOT NULL DEFAULT 0")
            }
        }

        private val MIGRATION_4_5 = object : Migration(4, 5) {
            override fun migrate(db: SupportSQLiteDatabase) {
                // 1. Create profiles table
                db.execSQL(
                    """
                    CREATE TABLE IF NOT EXISTS profiles (
                        id INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,
                        name TEXT NOT NULL,
                        displayOrder INTEGER NOT NULL DEFAULT 0,
                        isDefault INTEGER NOT NULL DEFAULT 0,
                        createdAt INTEGER NOT NULL
                    )
                    """.trimIndent()
                )
                // 2. Insert default profile (id=1, name="ポケット")
                db.execSQL(
                    """
                    INSERT INTO profiles (id, name, displayOrder, isDefault, createdAt)
                    VALUES (1, 'ポケット', 0, 1, ${System.currentTimeMillis()})
                    """.trimIndent()
                )
                // 3. Add profileId column to items table (default=1)
                db.execSQL("ALTER TABLE items ADD COLUMN profileId INTEGER NOT NULL DEFAULT 1")
            }
        }

        private val MIGRATION_5_6 = object : Migration(5, 6) {
            override fun migrate(db: SupportSQLiteDatabase) {
                db.execSQL("ALTER TABLE profiles ADD COLUMN alarmEnabled INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE profiles ADD COLUMN alarmHour INTEGER NOT NULL DEFAULT 8")
                db.execSQL("ALTER TABLE profiles ADD COLUMN alarmMinute INTEGER NOT NULL DEFAULT 0")
                db.execSQL("ALTER TABLE profiles ADD COLUMN alarmDays INTEGER NOT NULL DEFAULT 0")
            }
        }

        fun getDatabase(context: Context): AppDatabase {
            return INSTANCE ?: synchronized(this) {
                val instance = Room.databaseBuilder(
                    context.applicationContext,
                    AppDatabase::class.java,
                    "pocketcheck_database"
                )
                    .addMigrations(MIGRATION_1_2, MIGRATION_2_3, MIGRATION_3_4, MIGRATION_4_5, MIGRATION_5_6)
                    .addCallback(object : Callback() {
                        override fun onCreate(db: SupportSQLiteDatabase) {
                            super.onCreate(db)
                            // Create default profile for fresh installs
                            db.execSQL(
                                """
                                INSERT INTO profiles (id, name, displayOrder, isDefault, createdAt, alarmEnabled, alarmHour, alarmMinute, alarmDays)
                                VALUES (1, 'Pocket', 0, 1, ${System.currentTimeMillis()}, 0, 8, 0, 0)
                                """.trimIndent()
                            )
                        }
                    })
                    .build()
                INSTANCE = instance
                instance
            }
        }
    }
}