package org.gonbei774.pocketcheck.data

import android.content.Context
import android.content.SharedPreferences
import android.content.res.Configuration
import java.util.Locale

class LanguagePreferences(context: Context) {
    private val prefs: SharedPreferences = context.getSharedPreferences(
        PREFS_NAME,
        Context.MODE_PRIVATE
    )

    fun getLanguage(): AppLanguage {
        val languageCode = prefs.getString(KEY_LANGUAGE, AppLanguage.SYSTEM.code)
        return AppLanguage.fromCode(languageCode ?: AppLanguage.SYSTEM.code)
    }

    fun setLanguage(language: AppLanguage) {
        prefs.edit().putString(KEY_LANGUAGE, language.code).apply()
    }

    companion object {
        private const val PREFS_NAME = "language_preferences"
        private const val KEY_LANGUAGE = "app_language"

        fun wrapContext(context: Context): Context {
            val prefs = LanguagePreferences(context)
            val selectedLanguage = prefs.getLanguage()

            if (selectedLanguage == AppLanguage.SYSTEM) {
                return context
            }

            val locale = when (selectedLanguage) {
                AppLanguage.JAPANESE -> Locale("ja")
                AppLanguage.ENGLISH -> Locale("en")
                AppLanguage.SYSTEM -> return context
            }

            Locale.setDefault(locale)
            val config = Configuration(context.resources.configuration)
            config.setLocale(locale)
            return context.createConfigurationContext(config)
        }
    }
}

enum class AppLanguage(
    val code: String,
    val displayNameJa: String,
    val displayNameEn: String
) {
    SYSTEM("system", "システム設定に従う", "Follow system"),
    JAPANESE("ja", "日本語", "Japanese"),
    ENGLISH("en", "English", "English");

    fun getDisplayName(currentLanguageCode: String): String {
        return when (currentLanguageCode) {
            "ja" -> displayNameJa
            else -> displayNameEn
        }
    }

    companion object {
        fun fromCode(code: String): AppLanguage {
            return entries.find { it.code == code } ?: SYSTEM
        }
    }
}
