package org.gonbei774.pocketcheck.ui.screen.itemlist

import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.background
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Menu
import androidx.compose.material.icons.filled.Settings
import androidx.compose.material.icons.outlined.Archive
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.DrawerValue
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalNavigationDrawer
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.SwipeToDismissBox
import androidx.compose.material3.SwipeToDismissBoxState
import androidx.compose.material3.SwipeToDismissBoxValue
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.rememberDrawerState
import androidx.compose.material3.rememberSwipeToDismissBoxState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.dp
import kotlin.math.abs
import androidx.core.content.FileProvider
import org.gonbei774.pocketcheck.R
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.data.model.Item
import org.gonbei774.pocketcheck.ui.component.AddItemDialog
import org.gonbei774.pocketcheck.ui.component.AppDrawer
import org.gonbei774.pocketcheck.ui.component.ItemCard
import org.gonbei774.pocketcheck.util.AlarmPermissionHelper
import org.gonbei774.pocketcheck.util.ImageUtils
import sh.calvin.reorderable.ReorderableItem
import sh.calvin.reorderable.rememberReorderableLazyListState
import java.io.File

@OptIn(ExperimentalMaterial3Api::class, ExperimentalFoundationApi::class)
@Composable
fun ItemListScreen(
    viewModel: ItemListViewModel,
    onNavigateToSettings: () -> Unit = {},
    onRequestNotificationPermission: ((Boolean) -> Unit) -> Unit = {},
    modifier: Modifier = Modifier
) {
    val context = LocalContext.current
    val scope = rememberCoroutineScope()
    val items by viewModel.items.collectAsState()
    val isLoading by viewModel.isLoading.collectAsState()
    val showAddDialog by viewModel.showAddDialog.collectAsState()
    val editingItem by viewModel.editingItem.collectAsState()
    val capturedImageUri by viewModel.capturedImageUri.collectAsState()
    val pendingImagePath by viewModel.pendingImagePath.collectAsState()
    val isImageCleared by viewModel.isImageCleared.collectAsState()
    val itemToDelete by viewModel.itemToDelete.collectAsState()

    // Profile state
    val allProfiles by viewModel.allProfiles.collectAsState()
    val currentProfile by viewModel.currentProfile.collectAsState()
    val currentProfileId by viewModel.currentProfileId.collectAsState()
    val inUseCountByProfile by viewModel.inUseCountByProfile.collectAsState()

    // Profile menu state (for title long press)
    var showProfileMenu by remember { mutableStateOf(false) }
    var showRenameDialog by remember { mutableStateOf(false) }
    var showDeleteProfileDialog by remember { mutableStateOf(false) }

    // Drawer state
    val drawerState = rememberDrawerState(initialValue = DrawerValue.Closed)

    // カメラ撮影用の一時ファイルとURI
    val tempImageFile = remember {
        File(context.cacheDir, "temp_camera_image.jpg")
    }
    val tempImageUri = remember {
        FileProvider.getUriForFile(
            context,
            "${context.packageName}.fileprovider",
            tempImageFile
        )
    }

    // カメラ起動用のランチャー
    val cameraLauncher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.TakePicture()
    ) { success ->
        if (success) {
            viewModel.setCapturedImageUri(tempImageUri)
            // バックグラウンドで画像を圧縮・保存
            CoroutineScope(Dispatchers.IO).launch {
                val savedPath = ImageUtils.saveCompressedImage(context, tempImageUri)
                viewModel.setPendingImagePath(savedPath)
            }
        }
    }

    ModalNavigationDrawer(
        drawerState = drawerState,
        drawerContent = {
            AppDrawer(
                profiles = allProfiles,
                currentProfileId = currentProfileId,
                inUseCountByProfile = inUseCountByProfile,
                onProfileSelect = { profile ->
                    viewModel.switchProfile(profile.id)
                    scope.launch { drawerState.close() }
                },
                onCreateProfile = { name ->
                    viewModel.createProfile(name)
                    scope.launch { drawerState.close() }
                },
                onRenameProfile = { profile, newName ->
                    viewModel.renameProfile(profile, newName)
                },
                onDeleteProfile = { profile ->
                    viewModel.deleteProfile(profile)
                },
                onUpdateAlarm = { profile, enabled, hour, minute, days ->
                    viewModel.updateProfileAlarm(context, profile, enabled, hour, minute, days)
                },
                canScheduleExactAlarms = AlarmPermissionHelper.canScheduleExactAlarms(context),
                onRequestAlarmPermission = {
                    AlarmPermissionHelper.openExactAlarmSettings(context)
                }
            )
        },
        modifier = modifier
    ) {
        Scaffold(
            topBar = {
                TopAppBar(
                    title = {
                        Box(
                            modifier = Modifier
                                .combinedClickable(
                                    onClick = { },
                                    onLongClick = { showProfileMenu = true }
                                )
                        ) {
                            Text(currentProfile?.name ?: "PocketCheck")
                            DropdownMenu(
                                expanded = showProfileMenu,
                                onDismissRequest = { showProfileMenu = false }
                            ) {
                                DropdownMenuItem(
                                    text = { Text(stringResource(R.string.profile_rename)) },
                                    onClick = {
                                        showProfileMenu = false
                                        showRenameDialog = true
                                    }
                                )
                                if (currentProfile?.isDefault != true) {
                                    DropdownMenuItem(
                                        text = { Text(stringResource(R.string.delete)) },
                                        onClick = {
                                            showProfileMenu = false
                                            showDeleteProfileDialog = true
                                        }
                                    )
                                }
                            }
                        }
                    },
                    navigationIcon = {
                        IconButton(onClick = { scope.launch { drawerState.open() } }) {
                            Icon(
                                Icons.Default.Menu,
                                contentDescription = stringResource(R.string.cd_open_drawer)
                            )
                        }
                    },
                    actions = {
                        IconButton(onClick = onNavigateToSettings) {
                            Icon(Icons.Default.Settings, contentDescription = stringResource(R.string.cd_settings))
                        }
                    }
                )
            },
            bottomBar = {
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(16.dp)
                ) {
                    Button(
                        onClick = { viewModel.showAddDialog() },
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Icon(Icons.Default.Add, contentDescription = null)
                        Spacer(modifier = Modifier.width(8.dp))
                        Text(stringResource(R.string.add_item))
                    }
                }
            }
        ) { innerPadding ->
        if (items.isEmpty() && !isLoading) {
            // ローディング完了後のみ空リストメッセージを表示
            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(innerPadding),
                contentAlignment = Alignment.Center
            ) {
                Text(
                    text = stringResource(R.string.empty_item_list),
                    style = MaterialTheme.typography.bodyLarge,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }
        } else if (items.isNotEmpty()) {
            val lazyListState = rememberLazyListState()
            val reorderableLazyListState = rememberReorderableLazyListState(lazyListState) { from, to ->
                viewModel.reorderItems(from.index, to.index)
            }

            LazyColumn(
                state = lazyListState,
                modifier = Modifier
                    .fillMaxSize()
                    .padding(innerPadding),
                contentPadding = PaddingValues(16.dp),
                verticalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                items(items, key = { it.id }) { item ->
                    ReorderableItem(reorderableLazyListState, key = item.id) { isDragging ->
                        SwipeableItemCard(
                            item = item,
                            isDragging = isDragging,
                            onToggleUsage = {
                                if (!item.isInUse) {
                                    onRequestNotificationPermission { _ ->
                                        viewModel.toggleUsageStatus(item)
                                    }
                                } else {
                                    viewModel.toggleUsageStatus(item)
                                }
                            },
                            onClick = { viewModel.showEditDialog(item) },
                            onArchive = { viewModel.archiveItem(item) },
                            onDelete = { viewModel.showDeleteConfirmation(item) },
                            modifier = Modifier.longPressDraggableHandle()
                        )
                    }
                }
            }
        }

        // 削除確認ダイアログ
        if (itemToDelete != null) {
            AlertDialog(
                onDismissRequest = { viewModel.dismissDeleteConfirmation() },
                title = { Text(stringResource(R.string.delete_confirm_title)) },
                text = {
                    Text(
                        stringResource(
                            R.string.delete_confirm_message,
                            itemToDelete?.name?.ifBlank { stringResource(R.string.item_no_name) } ?: ""
                        )
                    )
                },
                confirmButton = {
                    TextButton(onClick = { viewModel.confirmDelete() }) {
                        Text(stringResource(R.string.delete))
                    }
                },
                dismissButton = {
                    TextButton(onClick = { viewModel.dismissDeleteConfirmation() }) {
                        Text(stringResource(R.string.cancel))
                    }
                }
            )
        }

            if (showAddDialog) {
                AddItemDialog(
                    editingItem = editingItem,
                    onDismiss = { viewModel.dismissDialog() },
                    onConfirm = { name, location, imagePath, profileId ->
                        if (editingItem != null) {
                            viewModel.updateItem(editingItem!!, name, location, imagePath, profileId)
                        } else {
                            viewModel.addItem(name, location, imagePath, profileId)
                        }
                    },
                    onDelete = { viewModel.deleteItem(it) },
                    onCameraClick = { cameraLauncher.launch(tempImageUri) },
                    capturedImageUri = capturedImageUri,
                    pendingImagePath = pendingImagePath,
                    onClearImage = { viewModel.clearImage() },
                    isImageCleared = isImageCleared,
                    profiles = allProfiles,
                    currentProfileId = currentProfileId
                )
            }

            // Profile rename dialog
            if (showRenameDialog && currentProfile != null) {
                var newName by remember(currentProfile) { mutableStateOf(currentProfile!!.name) }
                AlertDialog(
                    onDismissRequest = { showRenameDialog = false },
                    title = { Text(stringResource(R.string.profile_rename_title)) },
                    text = {
                        OutlinedTextField(
                            value = newName,
                            onValueChange = { newName = it },
                            label = { Text(stringResource(R.string.profile_name_label)) },
                            singleLine = true,
                            modifier = Modifier.fillMaxWidth()
                        )
                    },
                    confirmButton = {
                        TextButton(
                            onClick = {
                                viewModel.renameProfile(currentProfile!!, newName.trim())
                                showRenameDialog = false
                            },
                            enabled = newName.isNotBlank()
                        ) {
                            Text(stringResource(R.string.update))
                        }
                    },
                    dismissButton = {
                        TextButton(onClick = { showRenameDialog = false }) {
                            Text(stringResource(R.string.cancel))
                        }
                    }
                )
            }

            // Profile delete confirmation dialog
            if (showDeleteProfileDialog && currentProfile != null) {
                AlertDialog(
                    onDismissRequest = { showDeleteProfileDialog = false },
                    title = { Text(stringResource(R.string.profile_delete_title)) },
                    text = {
                        Text(stringResource(R.string.profile_delete_message, currentProfile!!.name))
                    },
                    confirmButton = {
                        TextButton(onClick = {
                            viewModel.deleteProfile(currentProfile!!)
                            showDeleteProfileDialog = false
                        }) {
                            Text(stringResource(R.string.delete))
                        }
                    },
                    dismissButton = {
                        TextButton(onClick = { showDeleteProfileDialog = false }) {
                            Text(stringResource(R.string.cancel))
                        }
                    }
                )
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun SwipeableItemCard(
    item: Item,
    isDragging: Boolean,
    onToggleUsage: () -> Unit,
    onClick: () -> Unit,
    onArchive: () -> Unit,
    onDelete: () -> Unit,
    modifier: Modifier = Modifier
) {
    val density = LocalDensity.current
    // 最小スワイプ距離（速いフリックでも一定距離は必要）
    val minSwipeDistancePx = with(density) { 80.dp.toPx() }

    // dismissStateをconfirmValueChange内で参照するためのホルダー
    val dismissStateHolder = remember { mutableStateOf<SwipeToDismissBoxState?>(null) }

    val dismissState = rememberSwipeToDismissBoxState(
        confirmValueChange = { dismissValue ->
            val currentOffset = dismissStateHolder.value?.requireOffset() ?: 0f
            val hasSwipedEnough = abs(currentOffset) >= minSwipeDistancePx

            when (dismissValue) {
                SwipeToDismissBoxValue.StartToEnd -> {
                    // 左→右スワイプ = アーカイブ（最小距離チェック）
                    if (hasSwipedEnough) {
                        onArchive()
                        true
                    } else {
                        false
                    }
                }
                SwipeToDismissBoxValue.EndToStart -> {
                    // 右→左スワイプ = 削除（最小距離チェック）
                    if (hasSwipedEnough) {
                        onDelete()
                    }
                    false // ダイアログで確認するのでリセット
                }
                SwipeToDismissBoxValue.Settled -> false
            }
        },
        // 誤操作防止: 幅の70%スワイプで発動（デフォルトより厳しい）
        positionalThreshold = { totalDistance -> totalDistance * 0.7f }
    )

    // dismissStateをホルダーに設定
    LaunchedEffect(dismissState) {
        dismissStateHolder.value = dismissState
    }

    SwipeToDismissBox(
        state = dismissState,
        backgroundContent = {
            val direction = dismissState.dismissDirection
            val color = when (direction) {
                SwipeToDismissBoxValue.StartToEnd -> MaterialTheme.colorScheme.secondary
                SwipeToDismissBoxValue.EndToStart -> MaterialTheme.colorScheme.error
                SwipeToDismissBoxValue.Settled -> Color.Transparent
            }
            val icon = when (direction) {
                SwipeToDismissBoxValue.StartToEnd -> Icons.Outlined.Archive
                SwipeToDismissBoxValue.EndToStart -> Icons.Default.Delete
                SwipeToDismissBoxValue.Settled -> null
            }
            val alignment = when (direction) {
                SwipeToDismissBoxValue.EndToStart -> Alignment.CenterEnd
                else -> Alignment.CenterStart
            }

            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .background(color)
                    .padding(horizontal = 20.dp),
                contentAlignment = alignment
            ) {
                icon?.let {
                    Icon(
                        imageVector = it,
                        contentDescription = null,
                        tint = Color.White
                    )
                }
            }
        },
        content = {
            ItemCard(
                item = item,
                onToggleUsage = onToggleUsage,
                onClick = onClick,
                isDragging = isDragging,
                modifier = modifier
            )
        }
    )
}