package org.gonbei774.pocketcheck.ui.navigation

import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import org.gonbei774.pocketcheck.data.AppTheme
import org.gonbei774.pocketcheck.data.local.SettingsDataStore
import org.gonbei774.pocketcheck.data.repository.ItemRepository
import org.gonbei774.pocketcheck.data.repository.ProfileRepository
import org.gonbei774.pocketcheck.data.repository.SettingsRepository
import org.gonbei774.pocketcheck.ui.screen.archived.ArchivedItemsScreen
import org.gonbei774.pocketcheck.ui.screen.archived.ArchivedItemsViewModel
import org.gonbei774.pocketcheck.ui.screen.itemlist.ItemListScreen
import org.gonbei774.pocketcheck.ui.screen.itemlist.ItemListViewModel
import org.gonbei774.pocketcheck.ui.screen.licenses.LicensesScreen
import org.gonbei774.pocketcheck.ui.screen.settings.SettingsScreen
import org.gonbei774.pocketcheck.ui.screen.settings.SettingsViewModel

object Routes {
    const val ITEM_LIST = "item_list"
    const val SETTINGS = "settings"
    const val ARCHIVED_ITEMS = "archived_items"
    const val LICENSES = "licenses"
}

@Composable
fun NavGraph(
    navController: NavHostController,
    itemRepository: ItemRepository,
    profileRepository: ProfileRepository,
    settingsRepository: SettingsRepository,
    settingsDataStore: SettingsDataStore,
    snackbarHostState: SnackbarHostState,
    onRequestNotificationPermission: ((Boolean) -> Unit) -> Unit,
    onScheduleReminder: (Int) -> Unit,
    onCancelReminder: () -> Unit,
    onThemeChange: (AppTheme) -> Unit = {},
    modifier: Modifier = Modifier
) {
    Scaffold(
        snackbarHost = { SnackbarHost(snackbarHostState) },
        modifier = modifier
    ) { innerPadding ->
        NavHost(
            navController = navController,
            startDestination = Routes.ITEM_LIST,
            modifier = Modifier.padding(innerPadding)
        ) {
            composable(Routes.ITEM_LIST) {
                val viewModel: ItemListViewModel = viewModel(
                    factory = ItemListViewModel.factory(itemRepository, profileRepository, settingsDataStore)
                )
                ItemListScreen(
                    viewModel = viewModel,
                    onNavigateToSettings = { navController.navigate(Routes.SETTINGS) },
                    onRequestNotificationPermission = onRequestNotificationPermission
                )
            }
            composable(Routes.SETTINGS) {
                val viewModel: SettingsViewModel = viewModel(
                    factory = SettingsViewModel.factory(settingsRepository)
                )
                SettingsScreen(
                    viewModel = viewModel,
                    onNavigateBack = { navController.popBackStack() },
                    onNavigateToArchivedItems = { navController.navigate(Routes.ARCHIVED_ITEMS) },
                    onNavigateToLicenses = { navController.navigate(Routes.LICENSES) },
                    onRequestNotificationPermission = onRequestNotificationPermission,
                    onScheduleReminder = onScheduleReminder,
                    onCancelReminder = onCancelReminder,
                    onThemeChange = onThemeChange
                )
            }
            composable(Routes.ARCHIVED_ITEMS) {
                val viewModel: ArchivedItemsViewModel = viewModel(
                    factory = ArchivedItemsViewModel.factory(itemRepository)
                )
                ArchivedItemsScreen(
                    viewModel = viewModel,
                    onNavigateBack = { navController.popBackStack() }
                )
            }
            composable(Routes.LICENSES) {
                LicensesScreen(
                    onNavigateBack = { navController.popBackStack() }
                )
            }
        }
    }
}