package org.gonbei774.pocketcheck.ui.component

import androidx.compose.animation.core.animateDpAsState
import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.shadow
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import coil.compose.AsyncImage
import org.gonbei774.pocketcheck.R
import coil.request.CachePolicy
import coil.request.ImageRequest
import org.gonbei774.pocketcheck.data.model.Item
import java.io.File

@Composable
fun ItemCard(
    item: Item,
    onToggleUsage: () -> Unit,
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
    isDragging: Boolean = false
) {
    val containerColor = if (item.isInUse) {
        MaterialTheme.colorScheme.primaryContainer
    } else {
        MaterialTheme.colorScheme.surface
    }
    val contentColor = if (item.isInUse) {
        MaterialTheme.colorScheme.onPrimaryContainer
    } else {
        MaterialTheme.colorScheme.onSurface
    }

    val elevation by animateDpAsState(
        targetValue = if (isDragging) 8.dp else 0.dp,
        label = "elevation"
    )

    Card(
        modifier = modifier
            .fillMaxWidth()
            .shadow(elevation, RoundedCornerShape(12.dp))
            .clickable(onClick = onClick),
        colors = CardDefaults.cardColors(containerColor = containerColor),
        border = if (!item.isInUse) BorderStroke(1.dp, MaterialTheme.colorScheme.outlineVariant) else null
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            // サムネイル画像（画像がある場合のみ表示）
            if (!item.imagePath.isNullOrBlank()) {
                val context = LocalContext.current
                val imageFile = File(item.imagePath)
                AsyncImage(
                    model = ImageRequest.Builder(context)
                        .data(imageFile)
                        .diskCachePolicy(CachePolicy.DISABLED)
                        .memoryCachePolicy(CachePolicy.DISABLED)
                        .build(),
                    contentDescription = item.name.ifBlank { stringResource(R.string.cd_item_image) },
                    modifier = Modifier
                        .size(48.dp)
                        .clip(RoundedCornerShape(6.dp)),
                    contentScale = ContentScale.Crop
                )
                Spacer(modifier = Modifier.width(12.dp))
            }

            Column(modifier = Modifier.weight(1f)) {
                // 名前がある場合のみ表示
                if (item.name.isNotBlank()) {
                    Text(
                        text = item.name,
                        style = MaterialTheme.typography.titleMedium,
                        color = contentColor
                    )
                } else if (!item.imagePath.isNullOrBlank()) {
                    // 名前がなく画像がある場合はプレースホルダーテキスト
                    Text(
                        text = stringResource(R.string.item_no_name),
                        style = MaterialTheme.typography.titleMedium,
                        color = contentColor.copy(alpha = 0.5f)
                    )
                }
                if (item.location.isNotBlank()) {
                    Text(
                        text = item.location,
                        style = MaterialTheme.typography.bodyMedium,
                        color = contentColor.copy(alpha = 0.7f)
                    )
                }
            }
            Switch(
                checked = item.isInUse,
                onCheckedChange = { onToggleUsage() }
            )
        }
    }
}