package org.gonbei774.pocketcheck.service

import android.app.AlarmManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import org.gonbei774.pocketcheck.PocketCheckApplication
import org.gonbei774.pocketcheck.data.model.Profile
import org.gonbei774.pocketcheck.util.AlarmPermissionHelper
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.Calendar

class ProfileAlarmScheduler(private val context: Context) {

    private val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager

    fun scheduleAlarm(profile: Profile) {
        if (!profile.alarmEnabled || profile.alarmDays == 0) {
            cancelAlarm(profile.id)
            return
        }

        if (!AlarmPermissionHelper.canScheduleExactAlarms(context)) {
            return
        }

        val nextTriggerTime = calculateNextTriggerTime(
            profile.alarmHour,
            profile.alarmMinute,
            profile.alarmDays
        )

        val intent = Intent(context, ProfileAlarmReceiver::class.java).apply {
            action = ProfileAlarmReceiver.ACTION_PROFILE_ALARM
            putExtra(EXTRA_PROFILE_ID, profile.id)
            putExtra(EXTRA_PROFILE_NAME, profile.name)
        }

        val pendingIntent = PendingIntent.getBroadcast(
            context,
            REQUEST_CODE_BASE + profile.id.toInt(),
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        alarmManager.setAlarmClock(
            AlarmManager.AlarmClockInfo(nextTriggerTime, pendingIntent),
            pendingIntent
        )
    }

    fun cancelAlarm(profileId: Long) {
        val intent = Intent(context, ProfileAlarmReceiver::class.java).apply {
            action = ProfileAlarmReceiver.ACTION_PROFILE_ALARM
        }
        val pendingIntent = PendingIntent.getBroadcast(
            context,
            REQUEST_CODE_BASE + profileId.toInt(),
            intent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        alarmManager.cancel(pendingIntent)
    }

    fun rescheduleAllAlarms() {
        if (!AlarmPermissionHelper.canScheduleExactAlarms(context)) {
            return
        }

        CoroutineScope(Dispatchers.IO).launch {
            val application = context.applicationContext as PocketCheckApplication
            val profiles = application.profileRepository.getProfilesWithAlarmEnabled()
            profiles.forEach { profile ->
                scheduleAlarm(profile)
            }
        }
    }

    private fun calculateNextTriggerTime(hour: Int, minute: Int, days: Int): Long {
        val now = Calendar.getInstance()
        val target = Calendar.getInstance().apply {
            set(Calendar.HOUR_OF_DAY, hour)
            set(Calendar.MINUTE, minute)
            set(Calendar.SECOND, 0)
            set(Calendar.MILLISECOND, 0)
        }

        // 今日から7日間をチェック
        for (i in 0..7) {
            val checkDay = Calendar.getInstance().apply {
                timeInMillis = target.timeInMillis
                add(Calendar.DAY_OF_YEAR, i)
            }

            // 曜日のビットフラグをチェック（日=1, 月=2, 火=4, ...）
            val dayOfWeek = checkDay.get(Calendar.DAY_OF_WEEK)
            val dayBit = 1 shl (dayOfWeek - Calendar.SUNDAY)

            if ((days and dayBit) != 0) {
                // この曜日が選択されている
                if (i == 0 && checkDay.timeInMillis <= now.timeInMillis) {
                    // 今日だが既に時刻が過ぎている場合はスキップ
                    continue
                }
                return checkDay.timeInMillis
            }
        }

        // 見つからない場合は翌週の最初の該当日
        return target.timeInMillis + (7 * 24 * 60 * 60 * 1000L)
    }

    companion object {
        private const val REQUEST_CODE_BASE = 10000
        const val EXTRA_PROFILE_ID = "profile_id"
        const val EXTRA_PROFILE_NAME = "profile_name"
    }
}