package org.gonbei774.pocketcheck.service

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.NotificationCompat
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.AlarmActivity
import org.gonbei774.pocketcheck.PocketCheckApplication
import org.gonbei774.pocketcheck.R

class ProfileAlarmReceiver : BroadcastReceiver() {

    override fun onReceive(context: Context, intent: Intent) {
        if (intent.action != ACTION_PROFILE_ALARM) return

        val profileId = intent.getLongExtra(ProfileAlarmScheduler.EXTRA_PROFILE_ID, -1)
        val profileName = intent.getStringExtra(ProfileAlarmScheduler.EXTRA_PROFILE_NAME) ?: return

        if (profileId == -1L) return

        // 静かな時間帯チェック＆次回スケジュールを非同期で処理
        CoroutineScope(Dispatchers.IO).launch {
            val application = context.applicationContext as PocketCheckApplication
            val settings = application.settingsRepository.settings.first()

            // 静かな時間帯でなければアラーム表示
            val isInQuietTime = settings.quietTimeEnabled && ReminderReceiver.isInQuietTime(
                settings.quietTimeStartHour,
                settings.quietTimeStartMinute,
                settings.quietTimeEndHour,
                settings.quietTimeEndMinute
            )

            if (!isInQuietTime) {
                showAlarmNotification(context, profileId, profileName)
            }

            // 次回アラームをスケジュール
            val profile = application.profileRepository.getProfileById(profileId)
            if (profile != null && profile.alarmEnabled) {
                ProfileAlarmScheduler(context).scheduleAlarm(profile)
            }
        }
    }

    private fun showAlarmNotification(context: Context, profileId: Long, profileName: String) {
        val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        // 通知チャンネル作成（Android 8+）
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                CHANNEL_ID,
                "プロファイルアラーム",
                NotificationManager.IMPORTANCE_HIGH
            ).apply {
                description = "リストのアラーム通知"
                setBypassDnd(true) // おやすみモードを無視
            }
            notificationManager.createNotificationChannel(channel)
        }

        // AlarmActivityを起動するIntent
        val fullScreenIntent = Intent(context, AlarmActivity::class.java).apply {
            putExtra(AlarmActivity.EXTRA_PROFILE_ID, profileId)
            putExtra(AlarmActivity.EXTRA_PROFILE_NAME, profileName)
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TOP
        }
        val fullScreenPendingIntent = PendingIntent.getActivity(
            context,
            profileId.toInt(),
            fullScreenIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        // 通知作成
        val notification = NotificationCompat.Builder(context, CHANNEL_ID)
            .setSmallIcon(R.drawable.ic_launcher_foreground)
            .setContentTitle(profileName)
            .setContentText("リストをチェック")
            .setPriority(NotificationCompat.PRIORITY_HIGH)
            .setCategory(NotificationCompat.CATEGORY_ALARM)
            .setFullScreenIntent(fullScreenPendingIntent, true)
            .setAutoCancel(true)
            .build()

        notificationManager.notify(NOTIFICATION_ID_BASE + profileId.toInt(), notification)
    }

    companion object {
        const val ACTION_PROFILE_ALARM = "org.gonbei774.pocketcheck.ACTION_PROFILE_ALARM"
        private const val CHANNEL_ID = "profile_alarm_channel"
        private const val NOTIFICATION_ID_BASE = 2000
    }
}