package org.gonbei774.pocketcheck.data.repository

import kotlinx.coroutines.flow.Flow
import org.gonbei774.pocketcheck.data.local.ItemDao
import org.gonbei774.pocketcheck.data.local.ProfileDao
import org.gonbei774.pocketcheck.data.model.Profile

class ProfileRepository(
    private val profileDao: ProfileDao,
    private val itemDao: ItemDao
) {

    val allProfiles: Flow<List<Profile>> = profileDao.getAllProfiles()

    suspend fun getProfileById(id: Long): Profile? {
        return profileDao.getProfileById(id)
    }

    fun getProfileByIdFlow(id: Long): Flow<Profile?> {
        return profileDao.getProfileByIdFlow(id)
    }

    suspend fun getDefaultProfile(): Profile? {
        return profileDao.getDefaultProfile()
    }

    suspend fun insert(profile: Profile): Long {
        val maxOrder = profileDao.getMaxDisplayOrder() ?: -1
        val profileWithOrder = profile.copy(displayOrder = maxOrder + 1)
        return profileDao.insert(profileWithOrder)
    }

    suspend fun insert(name: String): Profile {
        val maxOrder = profileDao.getMaxDisplayOrder() ?: -1
        val newProfile = Profile(
            name = name,
            displayOrder = maxOrder + 1,
            isDefault = false
        )
        val id = profileDao.insert(newProfile)
        return newProfile.copy(id = id)
    }

    suspend fun update(profile: Profile) {
        profileDao.update(profile)
    }

    suspend fun delete(profile: Profile) {
        if (!profile.isDefault) {
            // Delete all items in this profile first
            itemDao.deleteItemsByProfile(profile.id)
            profileDao.delete(profile)
        }
    }

    suspend fun updateAlarm(profileId: Long, enabled: Boolean, hour: Int, minute: Int, days: Int) {
        profileDao.updateAlarm(profileId, enabled, hour, minute, days)
    }

    suspend fun getProfilesWithAlarmEnabled(): List<Profile> {
        return profileDao.getProfilesWithAlarmEnabled()
    }
}