package org.gonbei774.pocketcheck.data.local

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.longPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import org.gonbei774.pocketcheck.data.model.Settings

val Context.dataStore: DataStore<Preferences> by preferencesDataStore(name = "settings")

class SettingsDataStore(private val context: Context) {

    private object PreferencesKeys {
        val REMINDER_ENABLED = booleanPreferencesKey("reminder_enabled")
        val REMINDER_INTERVAL_MINUTES = intPreferencesKey("reminder_interval_minutes")
        val QUIET_TIME_ENABLED = booleanPreferencesKey("quiet_time_enabled")
        val QUIET_TIME_START_HOUR = intPreferencesKey("quiet_time_start_hour")
        val QUIET_TIME_START_MINUTE = intPreferencesKey("quiet_time_start_minute")
        val QUIET_TIME_END_HOUR = intPreferencesKey("quiet_time_end_hour")
        val QUIET_TIME_END_MINUTE = intPreferencesKey("quiet_time_end_minute")
        val PERSISTENT_NOTIFICATION_ENABLED = booleanPreferencesKey("persistent_notification_enabled")
        val LAST_SELECTED_PROFILE_ID = longPreferencesKey("last_selected_profile_id")
    }

    val settings: Flow<Settings> = context.dataStore.data.map { preferences ->
        Settings(
            reminderEnabled = preferences[PreferencesKeys.REMINDER_ENABLED] ?: true,
            reminderIntervalMinutes = preferences[PreferencesKeys.REMINDER_INTERVAL_MINUTES] ?: 90,
            quietTimeEnabled = preferences[PreferencesKeys.QUIET_TIME_ENABLED] ?: true,
            quietTimeStartHour = preferences[PreferencesKeys.QUIET_TIME_START_HOUR] ?: 22,
            quietTimeStartMinute = preferences[PreferencesKeys.QUIET_TIME_START_MINUTE] ?: 0,
            quietTimeEndHour = preferences[PreferencesKeys.QUIET_TIME_END_HOUR] ?: 8,
            quietTimeEndMinute = preferences[PreferencesKeys.QUIET_TIME_END_MINUTE] ?: 0,
            persistentNotificationEnabled = preferences[PreferencesKeys.PERSISTENT_NOTIFICATION_ENABLED] ?: true
        )
    }

    suspend fun updateReminderEnabled(enabled: Boolean) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.REMINDER_ENABLED] = enabled
        }
    }

    suspend fun updateReminderIntervalMinutes(minutes: Int) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.REMINDER_INTERVAL_MINUTES] = minutes
        }
    }

    suspend fun updateQuietTimeStart(hour: Int, minute: Int) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.QUIET_TIME_START_HOUR] = hour
            preferences[PreferencesKeys.QUIET_TIME_START_MINUTE] = minute
        }
    }

    suspend fun updateQuietTimeEnabled(enabled: Boolean) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.QUIET_TIME_ENABLED] = enabled
        }
    }

    suspend fun updateQuietTimeEnd(hour: Int, minute: Int) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.QUIET_TIME_END_HOUR] = hour
            preferences[PreferencesKeys.QUIET_TIME_END_MINUTE] = minute
        }
    }

    suspend fun updatePersistentNotificationEnabled(enabled: Boolean) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.PERSISTENT_NOTIFICATION_ENABLED] = enabled
        }
    }

    val lastSelectedProfileId: Flow<Long> = context.dataStore.data.map { preferences ->
        preferences[PreferencesKeys.LAST_SELECTED_PROFILE_ID] ?: 1L
    }

    suspend fun updateLastSelectedProfileId(profileId: Long) {
        context.dataStore.edit { preferences ->
            preferences[PreferencesKeys.LAST_SELECTED_PROFILE_ID] = profileId
        }
    }
}