package org.gonbei774.pocketcheck

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.os.Build
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.material3.SnackbarHostState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.core.content.ContextCompat
import androidx.navigation.compose.rememberNavController
import kotlinx.coroutines.launch
import org.gonbei774.pocketcheck.data.AppLanguage
import org.gonbei774.pocketcheck.data.AppTheme
import org.gonbei774.pocketcheck.data.LanguagePreferences
import org.gonbei774.pocketcheck.data.ThemePreferences
import org.gonbei774.pocketcheck.service.ReminderScheduler
import org.gonbei774.pocketcheck.ui.navigation.NavGraph
import org.gonbei774.pocketcheck.ui.theme.PocketCheckTheme
import java.util.Locale

class MainActivity : ComponentActivity() {

    private var onPermissionResult: ((Boolean) -> Unit)? = null

    private val requestPermissionLauncher = registerForActivityResult(
        ActivityResultContracts.RequestPermission()
    ) { isGranted ->
        onPermissionResult?.invoke(isGranted)
        onPermissionResult = null
    }

    override fun attachBaseContext(newBase: Context) {
        super.attachBaseContext(updateBaseContextLocale(newBase))
    }

    private fun updateBaseContextLocale(context: Context): Context {
        val languagePrefs = LanguagePreferences(context)
        val selectedLanguage = languagePrefs.getLanguage()

        if (selectedLanguage == AppLanguage.SYSTEM) {
            return context
        }

        val locale = when (selectedLanguage) {
            AppLanguage.JAPANESE -> Locale("ja")
            AppLanguage.ENGLISH -> Locale("en")
            AppLanguage.SYSTEM -> return context
        }

        Locale.setDefault(locale)

        val config = Configuration(context.resources.configuration)
        config.setLocale(locale)

        return context.createConfigurationContext(config)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            val themePrefs = remember { ThemePreferences(this@MainActivity) }
            var currentTheme by remember { mutableStateOf(themePrefs.getTheme()) }

            PocketCheckTheme(appTheme = currentTheme) {
                val application = application as PocketCheckApplication
                val navController = rememberNavController()
                val snackbarHostState = remember { SnackbarHostState() }
                val scope = rememberCoroutineScope()

                val reminderScheduler = remember { ReminderScheduler(this@MainActivity) }

                NavGraph(
                    navController = navController,
                    itemRepository = application.itemRepository,
                    profileRepository = application.profileRepository,
                    settingsRepository = application.settingsRepository,
                    settingsDataStore = application.settingsDataStore,
                    snackbarHostState = snackbarHostState,
                    onRequestNotificationPermission = { onResult ->
                        requestNotificationPermission { granted ->
                            if (!granted) {
                                scope.launch {
                                    snackbarHostState.showSnackbar(
                                        getString(R.string.notification_permission_required)
                                    )
                                }
                            }
                            onResult(granted)
                        }
                    },
                    onScheduleReminder = { intervalMinutes ->
                        reminderScheduler.scheduleReminder(intervalMinutes)
                    },
                    onCancelReminder = {
                        reminderScheduler.cancelReminder()
                    },
                    onThemeChange = { theme ->
                        themePrefs.setTheme(theme)
                        currentTheme = theme
                    },
                    modifier = Modifier.fillMaxSize()
                )
            }
        }
    }

    private fun requestNotificationPermission(onResult: (Boolean) -> Unit) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            when {
                ContextCompat.checkSelfPermission(
                    this,
                    Manifest.permission.POST_NOTIFICATIONS
                ) == PackageManager.PERMISSION_GRANTED -> {
                    onResult(true)
                }
                else -> {
                    onPermissionResult = onResult
                    requestPermissionLauncher.launch(Manifest.permission.POST_NOTIFICATIONS)
                }
            }
        } else {
            // Android 12以下は権限不要
            onResult(true)
        }
    }

    fun hasNotificationPermission(): Boolean {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.POST_NOTIFICATIONS
            ) == PackageManager.PERMISSION_GRANTED
        } else {
            true
        }
    }
}