import 'dart:async';
import 'dart:io';

import 'package:dbus/dbus.dart';

import 'bluez_uuid.dart';

/// Types of Bluetooth address.
enum BlueZAddressType { public, random }

/// Types of writes to a GATT characteristic.
enum BlueZGattCharacteristicWriteType { command, request, reliable }

/// Defines how a GATT characteristic value can be used.
enum BlueZGattCharacteristicFlag {
  broadcast,
  read,
  writeWithoutResponse,
  write,
  notify,
  indicate,
  authenticatedSignedWrites,
  extendedProperties,
  reliableWrite,
  writableAuxiliaries,
  encryptRead,
  encryptWrite,
  encryptAuthenticatedRead,
  encryptAuthenticatedWrite,
  secureRead,
  secureWrite,
  authorize,
}

/// The capability of an agent registered with [BlueZClient.registerAgent].
/// * [displayOnly] - can only display information from the device.
/// * [displayYesNo] - able to display information from the device and respond with yes/no answers.
/// * [keyboardOnly] - only able to respond with pin code / passcode information.
/// * [noInputNoOutput] - not able to display information or provide information to devices.
/// * [keyboardDisplay] - able to display information from the device and respond with ping code / passcode information.
enum BlueZAgentCapability {
  displayOnly,
  displayYesNo,
  keyboardOnly,
  noInputNoOutput,
  keyboardDisplay,
}

/// An exception generated by the BlueZ server.
class BlueZException extends DBusMethodResponseException {
  /// Exception message as reported by BlueZ server.
  String get message =>
      response.values.isNotEmpty ? response.values[0].asString() : '';

  BlueZException(DBusMethodErrorResponse response) : super(response);
}

class BlueZInvalidArgumentsException extends BlueZException {
  BlueZInvalidArgumentsException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZInProgressException extends BlueZException {
  BlueZInProgressException(DBusMethodErrorResponse response) : super(response);
}

class BlueZAlreadyExistsException extends BlueZException {
  BlueZAlreadyExistsException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotSupportedException extends BlueZException {
  BlueZNotSupportedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotConnectedException extends BlueZException {
  BlueZNotConnectedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZAlreadyConnectedException extends BlueZException {
  BlueZAlreadyConnectedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotAvailableException extends BlueZException {
  BlueZNotAvailableException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZDoesNotExistException extends BlueZException {
  BlueZDoesNotExistException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotAuthorizedException extends BlueZException {
  BlueZNotAuthorizedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotPermittedException extends BlueZException {
  BlueZNotPermittedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNoSuchAdapterException extends BlueZException {
  BlueZNoSuchAdapterException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZAgentNotAvailableException extends BlueZException {
  BlueZAgentNotAvailableException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZNotReadyException extends BlueZException {
  BlueZNotReadyException(DBusMethodErrorResponse response) : super(response);
}

class BlueZFailedException extends BlueZException {
  BlueZFailedException(DBusMethodErrorResponse response) : super(response);
}

class BlueZAuthenticationCanceledException extends BlueZException {
  BlueZAuthenticationCanceledException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZAuthenticationFailedException extends BlueZException {
  BlueZAuthenticationFailedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZAuthenticationRejectedException extends BlueZException {
  BlueZAuthenticationRejectedException(DBusMethodErrorResponse response)
      : super(response);
}

class BlueZAuthenticationTimeoutException extends BlueZException {
  BlueZAuthenticationTimeoutException(DBusMethodErrorResponse response)
      : super(response);
}

/// Bluetooth manufacturer Id.
class BlueZManufacturerId {
  final int id;

  const BlueZManufacturerId(this.id);

  @override
  String toString() => "BlueZManufacturerId('$id')";

  @override
  bool operator ==(other) => other is BlueZManufacturerId && other.id == id;

  @override
  int get hashCode => id.hashCode;
}

final _bluezAddressTypeMap = <String, BlueZAddressType>{
  'public': BlueZAddressType.public,
  'random': BlueZAddressType.random
};

/// A Bluetooth adapter.
class BlueZAdapter {
  final String _adapterInterfaceName = 'org.bluez.Adapter1';

  final _BlueZObject _object;

  BlueZAdapter(this._object);

  /// Stream of property names as their values change.
  Stream<List<String>> get propertiesChanged {
    var interface = _object.interfaces[_adapterInterfaceName];
    if (interface == null) {
      throw 'BlueZ adapter missing $_adapterInterfaceName interface';
    }
    return interface.propertiesChangedStreamController.stream;
  }

  /// Gets the available filters that can be given to [setDiscoveryFilter].
  Future<List<String>> getDiscoveryFilters() async {
    var result = await _object.callMethod(
        _adapterInterfaceName, 'GetDiscoveryFilters', [],
        replySignature: DBusSignature('as'));
    return result.returnValues[0].asStringArray().toList();
  }

  /// Sets the device discovery filter.
  Future<void> setDiscoveryFilter(
      {List<String>? uuids,
      int? rssi,
      int? pathloss,
      String? transport,
      bool? duplicateData,
      bool? discoverable,
      String? pattern}) async {
    var filter = <String, DBusValue>{};
    if (uuids != null) {
      filter['UUIDs'] = DBusArray.string(uuids);
    }
    if (rssi != null) {
      filter['RSSI'] = DBusInt16(rssi);
    }
    if (pathloss != null) {
      filter['Pathloss'] = DBusUint16(pathloss);
    }
    if (transport != null) {
      filter['Transport'] = DBusString(transport);
    }
    if (duplicateData != null) {
      filter['DuplicateData'] = DBusBoolean(duplicateData);
    }
    if (discoverable != null) {
      filter['Discoverable'] = DBusBoolean(discoverable);
    }
    if (pattern != null) {
      filter['Pattern'] = DBusString(pattern);
    }
    await _object.callMethod(_adapterInterfaceName, 'SetDiscoveryFilter',
        [DBusDict.stringVariant(filter)],
        replySignature: DBusSignature(''));
  }

  /// Start discovery of devices on this adapter.
  Future<void> startDiscovery() async {
    await _object.callMethod(_adapterInterfaceName, 'StartDiscovery', [],
        replySignature: DBusSignature(''));
  }

  /// Stop discovery of devices on this adapter.
  Future<void> stopDiscovery() async {
    await _object.callMethod(_adapterInterfaceName, 'StopDiscovery', [],
        replySignature: DBusSignature(''));
  }

  /// Removes settings for [device] from this adapter.
  Future<void> removeDevice(BlueZDevice device) async {
    await _object.callMethod(
        _adapterInterfaceName, 'RemoveDevice', [device._object.path],
        replySignature: DBusSignature(''));
  }

  /// Bluetooth device address of this adapter.
  String get address =>
      _object.getStringProperty(_adapterInterfaceName, 'Address') ?? '';

  /// The Bluetooth address type.
  BlueZAddressType get addressType =>
      _bluezAddressTypeMap[
          _object.getStringProperty(_adapterInterfaceName, 'AddressType') ??
              ''] ??
      BlueZAddressType.public;

  /// The alternative name for this adapter.
  String get alias =>
      _object.getStringProperty(_adapterInterfaceName, 'Alias') ?? '';

  /// Sets the alternative name for this adapter.
  Future<void> setAlias(String value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'Alias', DBusString(value));
  }

  /// Bluetooth device class.
  int get deviceClass =>
      _object.getUint32Property(_adapterInterfaceName, 'Class') ?? 0;

  /// True if this adapter is discoverable by other Bluetooth devices.
  bool get discoverable =>
      _object.getBooleanProperty(_adapterInterfaceName, 'Discoverable') ??
      false;

  /// Sets if this adapter can be discovered by other Bluetooth devices.
  Future<void> setDiscoverable(bool value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'Discoverable', DBusBoolean(value));
  }

  int get discoverableTimeout =>
      _object.getUint32Property(_adapterInterfaceName, 'DiscoverableTimeout') ??
      0;

  Future<void> setDiscoverableTimeout(int value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'DiscoverableTimeout', DBusUint32(value));
  }

  /// True if currently discovering devices.
  bool get discovering =>
      _object.getBooleanProperty(_adapterInterfaceName, 'Discovering') ?? false;

  /// Local Device ID information in modalias format used by the kernel and udev.
  String get modalias =>
      _object.getStringProperty(_adapterInterfaceName, 'Modalias') ?? '';

  /// Name of this adapter.
  String get name =>
      _object.getStringProperty(_adapterInterfaceName, 'Name') ?? '';

  /// True if other Bluetooth devices can pair with this adapter.
  bool get pairable =>
      _object.getBooleanProperty(_adapterInterfaceName, 'Pairable') ?? false;

  /// Sets if other Bluetooth devices can pair with this adapter.
  Future<void> setPairable(bool value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'Pairable', DBusBoolean(value));
  }

  /// Timeout in seconds when pairing.
  int get pairableTimeout =>
      _object.getUint32Property(_adapterInterfaceName, 'PairableTimeout') ?? 0;

  /// Sets the timeout in seconds when pairing.
  Future<void> setPairableTimeout(int value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'PairableTimeout', DBusUint32(value));
  }

  /// True if this adapter is powered on.
  bool get powered =>
      _object.getBooleanProperty(_adapterInterfaceName, 'Powered') ?? false;

  /// Sets if this adapter is powered on.
  Future<void> setPowered(bool value) async {
    await _object.setProperty(
        _adapterInterfaceName, 'Powered', DBusBoolean(value));
  }

  List<String> get roles =>
      _object.getStringArrayProperty(_adapterInterfaceName, 'Roles') ?? [];

  /// List of 128-bit UUIDs that represents the available local services.
  List<BlueZUUID> get uuids =>
      (_object.getStringArrayProperty(_adapterInterfaceName, 'UUIDs') ?? [])
          .map((value) => BlueZUUID.fromString(value))
          .toList();
}

/// A GATT service running on a BlueZ device.
class BlueZGattService {
  final String _serviceInterfaceName = 'org.bluez.GattService1';

  final BlueZClient _client;
  final _BlueZObject _object;

  BlueZGattService(this._client, this._object);

  // TODO(robert-ancell): Includes

  /// True if this is a primary service.
  bool get primary =>
      _object.getBooleanProperty(_serviceInterfaceName, 'Primary') ?? false;

  /// Unique ID for this service.
  BlueZUUID get uuid => BlueZUUID.fromString(
      _object.getStringProperty(_serviceInterfaceName, 'UUID') ?? '');

  /// The Gatt characteristics provided by this service.
  List<BlueZGattCharacteristic> get characteristics =>
      _client._getGattCharacteristics(_object.path);
}

/// Result of a [BlueZGattCharacteristic.acquireWrite] call.
class BlueZGattAcquireWriteResult {
  /// Socket to allow writes to the GATT characteristic.
  final RawSocket socket;

  /// The maximum number of bytes allowed in each write to [socket].
  final int mtu;

  const BlueZGattAcquireWriteResult(this.socket, this.mtu);
}

/// Result of a [BlueZGattCharacteristic.acquireNotify] call.
class BlueZGattAcquireNotifyResult {
  /// Socket that streams values from the device.
  final RawSocket socket;

  /// The maximum number of bytes allowed in each read from [socket].
  final int mtu;

  const BlueZGattAcquireNotifyResult(this.socket, this.mtu);
}

/// A characteristic of a GATT service.
class BlueZGattCharacteristic {
  final String _gattCharacteristicInterfaceName =
      'org.bluez.GattCharacteristic1';

  final BlueZClient _client;
  final _BlueZObject _object;

  BlueZGattCharacteristic(this._client, this._object);

  /// Stream of property names as their values change.
  Stream<List<String>> get propertiesChanged {
    var interface = _object.interfaces[_gattCharacteristicInterfaceName];
    if (interface == null) {
      throw 'BlueZ characteristic missing $_gattCharacteristicInterfaceName interface';
    }
    return interface.propertiesChangedStreamController.stream;
  }

  // TODO(robert-ancell): Includes

  /// Unique ID for this characteristic.
  BlueZUUID get uuid => BlueZUUID.fromString(
      _object.getStringProperty(_gattCharacteristicInterfaceName, 'UUID') ??
          '');

  /// Cached value of this characteristic, updated when [readValue] is called or in a notification session triggered by [startNotify].
  List<int> get value =>
      _object.getByteArrayProperty(_gattCharacteristicInterfaceName, 'Value') ??
      [];

  /// Get mtu value of this characteristic
  int? get mtu =>
      _object.getUint16Property(_gattCharacteristicInterfaceName, 'MTU');

  /// True if if this characteristic has been acquired by any client using [acquireWrite].
  bool get writeAcquired =>
      _object.getBooleanProperty(
          _gattCharacteristicInterfaceName, 'WriteAcquired') ??
      false;

  /// True if if this characteristic has been acquired by any client using [acquireNotify].
  bool get notifyAcquired =>
      _object.getBooleanProperty(
          _gattCharacteristicInterfaceName, 'NotifyAcquired') ??
      false;

  /// True, if notifications or indications on this characteristic are currently enabled.
  bool get notifying =>
      _object.getBooleanProperty(
          _gattCharacteristicInterfaceName, 'Notifying') ??
      false;

  /// Defines how this characteristic value can be used.
  Set<BlueZGattCharacteristicFlag> get flags {
    var flags = <BlueZGattCharacteristicFlag>{};
    var values = _object.getStringArrayProperty(
            _gattCharacteristicInterfaceName, 'Flags') ??
        [];
    for (var value in values) {
      switch (value) {
        case 'broadcast':
          flags.add(BlueZGattCharacteristicFlag.broadcast);
          break;
        case 'read':
          flags.add(BlueZGattCharacteristicFlag.read);
          break;
        case 'write-without-response':
          flags.add(BlueZGattCharacteristicFlag.writeWithoutResponse);
          break;
        case 'write':
          flags.add(BlueZGattCharacteristicFlag.write);
          break;
        case 'notify':
          flags.add(BlueZGattCharacteristicFlag.notify);
          break;
        case 'indicate':
          flags.add(BlueZGattCharacteristicFlag.indicate);
          break;
        case 'authenticated-signed-writes':
          flags.add(BlueZGattCharacteristicFlag.authenticatedSignedWrites);
          break;
        case 'extended-properties':
          flags.add(BlueZGattCharacteristicFlag.extendedProperties);
          break;
        case 'reliable-write':
          flags.add(BlueZGattCharacteristicFlag.reliableWrite);
          break;
        case 'writable-auxiliaries':
          flags.add(BlueZGattCharacteristicFlag.writableAuxiliaries);
          break;
        case 'encrypt-read':
          flags.add(BlueZGattCharacteristicFlag.encryptRead);
          break;
        case 'encrypt-write':
          flags.add(BlueZGattCharacteristicFlag.encryptWrite);
          break;
        case 'encrypt-authenticated-read':
          flags.add(BlueZGattCharacteristicFlag.encryptAuthenticatedRead);
          break;
        case 'encrypt-authenticated-write':
          flags.add(BlueZGattCharacteristicFlag.encryptAuthenticatedWrite);
          break;
        case 'secure-read':
          flags.add(BlueZGattCharacteristicFlag.secureRead);
          break;
        case 'secure-write':
          flags.add(BlueZGattCharacteristicFlag.secureWrite);
          break;
        case 'authorize':
          flags.add(BlueZGattCharacteristicFlag.authorize);
          break;
      }
    }
    return flags;
  }

  /// The Gatt descriptors provided by this characteristic.
  List<BlueZGattDescriptor> get descriptors =>
      _client._getGattDescriptors(_object.path);

  /// Reads the value of the characteristic.
  Future<List<int>> readValue({int? offset}) async {
    var options = <String, DBusValue>{};
    if (offset != null) {
      options['offset'] = DBusUint16(offset);
    }
    var result = await _object.callMethod(_gattCharacteristicInterfaceName,
        'ReadValue', [DBusDict.stringVariant(options)],
        replySignature: DBusSignature('ay'));
    return result.returnValues[0].asByteArray().toList();
  }

  /// Writes [data] to the characteristic.
  Future<void> writeValue(Iterable<int> data,
      {int? offset,
      BlueZGattCharacteristicWriteType? type,
      bool? prepareAuthorize}) async {
    var options = <String, DBusValue>{};
    if (offset != null) {
      options['offset'] = DBusUint16(offset);
    }
    if (type != null) {
      String typeName;
      switch (type) {
        case BlueZGattCharacteristicWriteType.command:
          typeName = 'command';
          break;
        case BlueZGattCharacteristicWriteType.request:
          typeName = 'request';
          break;
        case BlueZGattCharacteristicWriteType.reliable:
          typeName = 'reliable';
          break;
      }
      options['type'] = DBusString(typeName);
    }
    if (prepareAuthorize != null) {
      options['prepare-authorize'] = DBusBoolean(prepareAuthorize);
    }
    await _object.callMethod(_gattCharacteristicInterfaceName, 'WriteValue',
        [DBusArray.byte(data), DBusDict.stringVariant(options)],
        replySignature: DBusSignature(''));
  }

  /// Acquire a [RawSocket] for writing to this characterisitic.
  /// Usage of [writeValue] will be locked causing it to return NotPermitted error.
  /// To release the lock close the returned file.
  Future<BlueZGattAcquireWriteResult> acquireWrite() async {
    var options = <String, DBusValue>{};
    var result = await _object.callMethod(_gattCharacteristicInterfaceName,
        'AcquireWrite', [DBusDict.stringVariant(options)],
        replySignature: DBusSignature('hq'));
    var handle = result.values[0].asUnixFd();
    var mtu = result.values[1].asUint16();
    return BlueZGattAcquireWriteResult(handle.toRawSocket(), mtu);
  }

  /// Acquire a [RawSocket] for receiving notifications from this characterisitic.
  /// To release the lock close the returned socket.
  Future<BlueZGattAcquireNotifyResult> acquireNotify() async {
    var options = <String, DBusValue>{};
    var result = await _object.callMethod(_gattCharacteristicInterfaceName,
        'AcquireNotify', [DBusDict.stringVariant(options)],
        replySignature: DBusSignature('hq'));
    var handle = result.values[0].asUnixFd();
    var mtu = result.values[1].asUint16();
    return BlueZGattAcquireNotifyResult(handle.toRawSocket(), mtu);
  }

  /// Starts a notification session from this characteristic if it supports value notifications or indications.
  Future<void> startNotify() async {
    await _object.callMethod(
        _gattCharacteristicInterfaceName, 'StartNotify', [],
        replySignature: DBusSignature(''));
  }

  /// Cancel any previous [startNotify] transaction.
  /// Note that notifications from a characteristic are shared between sessions thus calling stopNotify will release a single session.
  Future<void> stopNotify() async {
    await _object.callMethod(_gattCharacteristicInterfaceName, 'StopNotify', [],
        replySignature: DBusSignature(''));
  }
}

/// A GATT characteristic descriptor.
class BlueZGattDescriptor {
  final String _gattDescriptorInterfaceName = 'org.bluez.GattDescriptor1';

  final _BlueZObject _object;

  BlueZGattDescriptor(this._object);

  // TODO(robert-ancell): Includes

  /// Cached value of this descriptor, updated when [readValue] is called.
  List<int> get value =>
      _object.getByteArrayProperty(_gattDescriptorInterfaceName, 'Value') ?? [];

  /// Unique ID for this descriptor.
  BlueZUUID get uuid => BlueZUUID.fromString(
      _object.getStringProperty(_gattDescriptorInterfaceName, 'UUID') ?? '');

  /// Reads the value of the descriptor.
  Future<List<int>> readValue({int? offset}) async {
    var options = <String, DBusValue>{};
    if (offset != null) {
      options['offset'] = DBusUint16(offset);
    }
    var result = await _object.callMethod(_gattDescriptorInterfaceName,
        'ReadValue', [DBusDict.stringVariant(options)],
        replySignature: DBusSignature('ay'));
    return result.returnValues[0].asByteArray().toList();
  }

  /// Writes [data] to the descriptor.
  Future<void> writeValue(Iterable<int> data,
      {int? offset, bool? prepareAuthorize}) async {
    var options = <String, DBusValue>{};
    if (offset != null) {
      options['offset'] = DBusUint16(offset);
    }
    if (prepareAuthorize != null) {
      options['prepare-authorize'] = DBusBoolean(prepareAuthorize);
    }
    await _object.callMethod(_gattDescriptorInterfaceName, 'WriteValue',
        [DBusArray.byte(data), DBusDict.stringVariant(options)],
        replySignature: DBusSignature(''));
  }
}

/// A Bluetooth device.
class BlueZDevice {
  final String _deviceInterfaceName = 'org.bluez.Device1';

  final BlueZClient _client;
  final _BlueZObject _object;

  BlueZDevice(this._client, this._object);

  /// Stream of property names as their values change.
  Stream<List<String>> get propertiesChanged {
    var interface = _object.interfaces[_deviceInterfaceName];
    if (interface == null) {
      throw 'BlueZ device missing $_deviceInterfaceName interface';
    }
    return interface.propertiesChangedStreamController.stream;
  }

  /// Connect to this device.
  Future<void> connect() async {
    await _object.callMethod(_deviceInterfaceName, 'Connect', [],
        replySignature: DBusSignature(''));
  }

  /// Disconnect from this device
  Future<void> disconnect() async {
    await _object.callMethod(_deviceInterfaceName, 'Disconnect', [],
        replySignature: DBusSignature(''));
  }

  /// Connects to the service with [uuid].
  Future<void> connectProfile(BlueZUUID uuid) async {
    await _object.callMethod(
        _deviceInterfaceName, 'ConnectProfile', [DBusString(uuid.toString())],
        replySignature: DBusSignature(''));
  }

  /// Disconnects the service with [uuid].
  Future<void> disconnectProfile(BlueZUUID uuid) async {
    await _object.callMethod(_deviceInterfaceName, 'DisconnectProfile',
        [DBusString(uuid.toString())],
        replySignature: DBusSignature(''));
  }

  /// Pair with this device.
  Future<void> pair() async {
    await _object.callMethod(_deviceInterfaceName, 'Pair', [],
        replySignature: DBusSignature(''));
  }

  /// Cancel a pairing that is in progress.
  Future<void> cancelPairing() async {
    await _object.callMethod(_deviceInterfaceName, 'CancelPairing', [],
        replySignature: DBusSignature(''));
  }

  /// The adapter this device belongs to.
  BlueZAdapter get adapter {
    var objectPath =
        _object.getObjectPathProperty(_deviceInterfaceName, 'Adapter')!;
    return _client._getAdapter(objectPath)!;
  }

  /// MAC address of this device.
  String get address =>
      _object.getStringProperty(_deviceInterfaceName, 'Address') ?? '';

  /// The Bluetooth device address type.
  BlueZAddressType get addressType =>
      _bluezAddressTypeMap[
          _object.getStringProperty(_deviceInterfaceName, 'AddressType') ??
              ''] ??
      BlueZAddressType.public;

  /// An alternative name for this device.
  String get alias =>
      _object.getStringProperty(_deviceInterfaceName, 'Alias') ?? '';

  /// Sets the alternative name for this device.
  Future<void> setAlias(String value) async {
    await _object.setProperty(_deviceInterfaceName, 'Alias', DBusString(value));
  }

  /// External appearance of device, as found on GAP service.
  /// Appearance values are defined in the [Bluetooth specification](https://www.bluetooth.com/specifications/assigned-numbers/).
  int get appearance =>
      _object.getUint16Property(_deviceInterfaceName, 'Appearance') ?? 0;

  /// True if connections from this device will be ignored.
  bool get blocked =>
      _object.getBooleanProperty(_deviceInterfaceName, 'Blocked') ?? false;

  /// Sets if connections from this device will be ignored.
  Future<void> setBlocked(bool value) async {
    await _object.setProperty(
        _deviceInterfaceName, 'Blocked', DBusBoolean(value));
  }

  /// True if this device is currently connected.
  bool get connected =>
      _object.getBooleanProperty(_deviceInterfaceName, 'Connected') ?? false;

  /// Bluetooth device class.
  int get deviceClass =>
      _object.getUint32Property(_deviceInterfaceName, 'Class') ?? 0;

  /// True if this device only supports the pre-2.1 pairing mechanism.
  bool get legacyPairing =>
      _object.getBooleanProperty(_deviceInterfaceName, 'LegacyPairing') ??
      false;

  /// Icon name for this device.
  String get icon =>
      _object.getStringProperty(_deviceInterfaceName, 'Icon') ?? '';

  /// Manufacturer specific advertisement data.
  Map<BlueZManufacturerId, List<int>> get manufacturerData {
    var value =
        _object.getCachedProperty(_deviceInterfaceName, 'ManufacturerData') ??
            DBusDict(DBusSignature('q'), DBusSignature('v'), {});
    if (value.signature != DBusSignature('a{qv}')) {
      return {};
    }
    List<int> processValue(DBusValue value) {
      if (value.signature != DBusSignature('ay')) {
        return [];
      }
      return value.asByteArray().toList();
    }

    return value.asDict().map((key, value) => MapEntry(
        BlueZManufacturerId(key.asUint16()), processValue(value.asVariant())));
  }

  /// Remote Device ID information in modalias format used by the kernel and udev.
  String get modalias =>
      _object.getStringProperty(_deviceInterfaceName, 'Modalias') ?? '';

  /// Name of this device.
  String get name =>
      _object.getStringProperty(_deviceInterfaceName, 'Name') ?? '';

  /// True if the device is currently paired.
  bool get paired =>
      _object.getBooleanProperty(_deviceInterfaceName, 'Paired') ?? false;

  /// Signal strength received from the devide.
  int get rssi => _object.getInt16Property(_deviceInterfaceName, 'RSSI') ?? 0;

  /// Service advertisement data.
  Map<BlueZUUID, List<int>> get serviceData {
    var value =
        _object.getCachedProperty(_deviceInterfaceName, 'ServiceData') ??
            DBusDict.stringVariant({});
    if (value.signature != DBusSignature('a{sv}')) {
      return {};
    }
    List<int> processValue(DBusValue value) {
      if (value.signature != DBusSignature('ay')) {
        return [];
      }
      return value.asByteArray().toList();
    }

    return value.asDict().map((key, value) => MapEntry(
        BlueZUUID.fromString(key.asString()), processValue(value.asVariant())));
  }

  /// True if service discovery has been resolved.
  bool get servicesResolved =>
      _object.getBooleanProperty(_deviceInterfaceName, 'ServicesResolved') ??
      false;

  /// True if the remote is seen as trusted.
  bool get trusted =>
      _object.getBooleanProperty(_deviceInterfaceName, 'Trusted') ?? false;

  /// Sets if the remote is seen as trusted.
  Future<void> setTrusted(bool value) async {
    await _object.setProperty(
        _deviceInterfaceName, 'Trusted', DBusBoolean(value));
  }

  /// Advertised transmit power level.
  int get txPower =>
      _object.getInt16Property(_deviceInterfaceName, 'TxPower') ?? 0;

  /// UUIDs that indicate the available remote services.
  List<BlueZUUID> get uuids =>
      (_object.getStringArrayProperty(_deviceInterfaceName, 'UUIDs') ?? [])
          .map((value) => BlueZUUID.fromString(value))
          .toList();

  /// True if the device can wake the host from system suspend.
  bool get wakeAllowed =>
      _object.getBooleanProperty(_deviceInterfaceName, 'WakeAllowed') ?? false;

  /// Sets if the device can wake the host from system suspend.
  Future<void> setWakeAllowed(bool value) async {
    await _object.setProperty(
        _deviceInterfaceName, 'WakeAllowed', DBusBoolean(value));
  }

  /// The Gatt services provided by this device.
  List<BlueZGattService> get gattServices =>
      _client._getGattServices(_object.path);
}

class _BlueZInterface {
  final Map<String, DBusValue> properties;
  final propertiesChangedStreamController =
      StreamController<List<String>>.broadcast();

  Stream<List<String>> get propertiesChanged =>
      propertiesChangedStreamController.stream;

  _BlueZInterface(this.properties);

  void updateProperties(Map<String, DBusValue> changedProperties) {
    properties.addAll(changedProperties);
    propertiesChangedStreamController.add(changedProperties.keys.toList());
  }
}

class _BlueZObject extends DBusRemoteObject {
  final interfaces = <String, _BlueZInterface>{};

  void updateInterfaces(
      Map<String, Map<String, DBusValue>> interfacesAndProperties) {
    interfacesAndProperties.forEach((interfaceName, properties) {
      interfaces[interfaceName] = _BlueZInterface(properties);
    });
  }

  /// Returns true if removing [interfaceNames] would remove all interfaces on this object.
  bool wouldRemoveAllInterfaces(List<String> interfaceNames) {
    for (var interface in interfaces.keys) {
      if (!interfaceNames.contains(interface)) {
        return false;
      }
    }
    return true;
  }

  void removeInterfaces(List<String> interfaceNames) {
    for (var interfaceName in interfaceNames) {
      interfaces.remove(interfaceName);
    }
  }

  void updateProperties(
      String interfaceName, Map<String, DBusValue> changedProperties) {
    var interface = interfaces[interfaceName];
    if (interface != null) {
      interface.updateProperties(changedProperties);
    }
  }

  /// Gets a cached property.
  DBusValue? getCachedProperty(String interfaceName, String name) {
    var interface = interfaces[interfaceName];
    if (interface == null) {
      return null;
    }
    return interface.properties[name];
  }

  /// Gets a cached boolean property, or returns null if not present or not the correct type.
  bool? getBooleanProperty(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('b')) {
      return null;
    }
    return value.asBoolean();
  }

  /// Gets a cached byte array property, or returns null if not present or not the correct type.
  List<int>? getByteArrayProperty(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('ay')) {
      return null;
    }

    return value.asByteArray().toList();
  }

  /// Gets a cached signed 16 bit integer property, or returns null if not present or not the correct type.
  int? getInt16Property(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('n')) {
      return null;
    }
    return value.asInt16();
  }

  /// Gets a cached unsigned 16 bit integer property, or returns null if not present or not the correct type.
  int? getUint16Property(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('q')) {
      return null;
    }
    return value.asUint16();
  }

  /// Gets a cached unsigned 32 bit integer property, or returns null if not present or not the correct type.
  int? getUint32Property(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('u')) {
      return null;
    }
    return value.asUint32();
  }

  /// Gets a cached string property, or returns null if not present or not the correct type.
  String? getStringProperty(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('s')) {
      return null;
    }
    return value.asString();
  }

  /// Gets a cached string array property, or returns null if not present or not the correct type.
  List<String>? getStringArrayProperty(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('as')) {
      return null;
    }
    return value.asStringArray().toList();
  }

  /// Gets a cached object path property, or returns null if not present or not the correct type.
  DBusObjectPath? getObjectPathProperty(String interface, String name) {
    var value = getCachedProperty(interface, name);
    if (value == null) {
      return null;
    }
    if (value.signature != DBusSignature('o')) {
      return null;
    }
    return value.asObjectPath();
  }

  @override
  Future<DBusMethodSuccessResponse> callMethod(
      String? interface, String name, Iterable<DBusValue> values,
      {DBusSignature? replySignature,
      bool noReplyExpected = false,
      bool noAutoStart = false,
      bool allowInteractiveAuthorization = false}) async {
    try {
      return await super.callMethod(interface, name, values,
          replySignature: replySignature,
          noReplyExpected: noReplyExpected,
          noAutoStart: noAutoStart,
          allowInteractiveAuthorization: allowInteractiveAuthorization);
    } on DBusMethodResponseException catch (e) {
      switch (e.response.errorName) {
        case 'org.bluez.Error.InvalidArguments':
          throw BlueZInvalidArgumentsException(e.response);
        case 'org.bluez.Error.InProgress':
          throw BlueZInProgressException(e.response);
        case 'org.bluez.Error.AlreadyExists':
          throw BlueZAlreadyExistsException(e.response);
        case 'org.bluez.Error.NotSupported':
          throw BlueZNotSupportedException(e.response);
        case 'org.bluez.Error.NotConnected':
          throw BlueZNotConnectedException(e.response);
        case 'org.bluez.Error.AlreadyConnected':
          throw BlueZAlreadyConnectedException(e.response);
        case 'org.bluez.Error.NotAvailable':
          throw BlueZNotAvailableException(e.response);
        case 'org.bluez.Error.DoesNotExist':
          throw BlueZDoesNotExistException(e.response);
        case 'org.bluez.Error.NotAuthorized':
          throw BlueZNotAuthorizedException(e.response);
        case 'org.bluez.Error.NotPermitted':
          throw BlueZNotPermittedException(e.response);
        case 'org.bluez.Error.NoSuchAdapter':
          throw BlueZNoSuchAdapterException(e.response);
        case 'org.bluez.Error.AgentNotAvailable':
          throw BlueZAgentNotAvailableException(e.response);
        case 'org.bluez.Error.NotReady':
          throw BlueZNotReadyException(e.response);
        case 'org.bluez.Error.Failed':
          throw BlueZFailedException(e.response);
        case 'org.bluez.Error.AuthenticationCanceled':
          throw BlueZAuthenticationCanceledException(e.response);
        case 'org.bluez.Error.AuthenticationFailed':
          throw BlueZAuthenticationFailedException(e.response);
        case 'org.bluez.Error.AuthenticationRejected':
          throw BlueZAuthenticationRejectedException(e.response);
        case 'org.bluez.Error.AuthenticationTimeout':
          throw BlueZAuthenticationTimeoutException(e.response);
        default:
          rethrow;
      }
    }
  }

  _BlueZObject(DBusClient client, DBusObjectPath path,
      Map<String, Map<String, DBusValue>> interfacesAndProperties)
      : super(client, name: 'org.bluez', path: path) {
    updateInterfaces(interfacesAndProperties);
  }
}

class BlueZAgentResponse {
  final DBusMethodResponse response;

  BlueZAgentResponse(this.response);

  factory BlueZAgentResponse.success() =>
      BlueZAgentResponse(DBusMethodSuccessResponse());

  factory BlueZAgentResponse.rejected() =>
      BlueZAgentResponse(DBusMethodErrorResponse('org.bluez.Error.Rejected'));

  factory BlueZAgentResponse.canceled() =>
      BlueZAgentResponse(DBusMethodErrorResponse('org.bluez.Error.Canceled'));
}

class BlueZAgentPinCodeResponse {
  final DBusMethodResponse response;

  BlueZAgentPinCodeResponse(this.response);

  factory BlueZAgentPinCodeResponse.success(String pinCode) =>
      BlueZAgentPinCodeResponse(
          DBusMethodSuccessResponse([DBusString(pinCode)]));

  factory BlueZAgentPinCodeResponse.rejected() => BlueZAgentPinCodeResponse(
      DBusMethodErrorResponse('org.bluez.Error.Rejected'));

  factory BlueZAgentPinCodeResponse.canceled() => BlueZAgentPinCodeResponse(
      DBusMethodErrorResponse('org.bluez.Error.Canceled'));
}

class BlueZAgentPasskeyResponse {
  final DBusMethodResponse response;

  BlueZAgentPasskeyResponse(this.response);

  factory BlueZAgentPasskeyResponse.success(int passkey) =>
      BlueZAgentPasskeyResponse(
          DBusMethodSuccessResponse([DBusUint32(passkey)]));

  factory BlueZAgentPasskeyResponse.rejected() => BlueZAgentPasskeyResponse(
      DBusMethodErrorResponse('org.bluez.Error.Rejected'));

  factory BlueZAgentPasskeyResponse.canceled() => BlueZAgentPasskeyResponse(
      DBusMethodErrorResponse('org.bluez.Error.Canceled'));
}

/// Agent object for a client to register.
abstract class BlueZAgent {
  /// Called when this agent is unregistered.
  Future<void> release() async {}

  /// Called when a PIN code is required for authentication with [device].
  /// Return [BlueZAgentPinCodeResponse.success] with the requested PIN code.
  Future<BlueZAgentPinCodeResponse> requestPinCode(BlueZDevice device) async {
    return BlueZAgentPinCodeResponse.rejected();
  }

  /// Called when [pinCode] is required to be displayed when authenticating with [device].
  /// Return [BlueZAgentResponse.success] is this PIN is confirmed as correct, and [BlueZAgentResponse.rejected] if it is not.
  Future<BlueZAgentResponse> displayPinCode(
      BlueZDevice device, String pinCode) async {
    return BlueZAgentResponse.rejected();
  }

  /// Called when a passkey is required for authentication with [device].
  /// Return [BlueZAgentPasskeyResponse.success] with the requested passkey.
  Future<BlueZAgentPasskeyResponse> requestPasskey(BlueZDevice device) async {
    return BlueZAgentPasskeyResponse.rejected();
  }

  /// Called when [passkey] is required to be displayed when authenticating with [device].
  Future<void> displayPasskey(
      BlueZDevice device, int passkey, int entered) async {}

  /// Called when a passkey is required to be confirmed when authenticating with [device].
  /// Return [BlueZAgentResponse.success] is this passkey is confirmed as correct, and [BlueZAgentResponse.rejected] if it is not.
  Future<BlueZAgentResponse> requestConfirmation(
      BlueZDevice device, int passkey) async {
    return BlueZAgentResponse.rejected();
  }

  /// Called when confirmation is required when authenticating with [device].
  /// Return [BlueZAgentResponse.success] is this authentication should occur, and [BlueZAgentResponse.rejected] if it should not.
  Future<BlueZAgentResponse> requestAuthorization(BlueZDevice device) async {
    return BlueZAgentResponse.rejected();
  }

  /// Called when confirmation is required when accessing the service [uuid] on [device].
  /// Return [BlueZAgentResponse.success] is this authorization should occur, and [BlueZAgentResponse.rejected] if it should not.
  Future<BlueZAgentResponse> authorizeService(
      BlueZDevice device, BlueZUUID uuid) async {
    return BlueZAgentResponse.rejected();
  }

  /// Called when a request is canceled due to lack of response from the agent.
  Future<void> cancel() async {}
}

class _BlueZAgentObject extends DBusObject {
  final BlueZClient bluezClient;
  final BlueZAgent agent;

  _BlueZAgentObject(this.bluezClient, this.agent, DBusObjectPath path)
      : super(path);

  @override
  Future<DBusMethodResponse> handleMethodCall(DBusMethodCall methodCall) async {
    if (methodCall.interface != 'org.bluez.Agent1') {
      return DBusMethodErrorResponse.unknownInterface();
    }

    if (methodCall.name == 'Release') {
      if (methodCall.signature != DBusSignature('')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      await agent.release();
      return DBusMethodSuccessResponse();
    } else if (methodCall.name == 'RequestPinCode') {
      if (methodCall.signature != DBusSignature('o')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.requestPinCode(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!))
          .response;
    } else if (methodCall.name == 'DisplayPinCode') {
      if (methodCall.signature != DBusSignature('os')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.displayPinCode(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!,
              methodCall.values[1].asString()))
          .response;
    } else if (methodCall.name == 'RequestPasskey') {
      if (methodCall.signature != DBusSignature('o')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.requestPasskey(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!))
          .response;
    } else if (methodCall.name == 'DisplayPasskey') {
      if (methodCall.signature != DBusSignature('ouq')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      await agent.displayPasskey(
          bluezClient._getDevice(methodCall.values[0].asObjectPath())!,
          methodCall.values[1].asUint32(),
          methodCall.values[2].asUint16());
      return DBusMethodSuccessResponse();
    } else if (methodCall.name == 'RequestConfirmation') {
      if (methodCall.signature != DBusSignature('ou')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.requestConfirmation(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!,
              methodCall.values[1].asUint32()))
          .response;
    } else if (methodCall.name == 'RequestAuthorization') {
      if (methodCall.signature != DBusSignature('o')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.requestAuthorization(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!))
          .response;
    } else if (methodCall.name == 'AuthorizeService') {
      if (methodCall.signature != DBusSignature('os')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      return (await agent.authorizeService(
              bluezClient._getDevice(methodCall.values[0].asObjectPath())!,
              BlueZUUID.fromString(methodCall.values[1].asString())))
          .response;
    } else if (methodCall.name == 'Cancel') {
      if (methodCall.signature != DBusSignature('')) {
        return DBusMethodErrorResponse.invalidArgs();
      }
      await agent.cancel();
      return DBusMethodSuccessResponse();
    } else {
      return DBusMethodErrorResponse.unknownMethod();
    }
  }
}

/// A client that connects to BlueZ.
class BlueZClient {
  /// Stream of adapters as they are added.
  Stream<BlueZAdapter> get adapterAdded => _adapterAddedStreamController.stream;

  /// Stream of adapters as they are removed.
  Stream<BlueZAdapter> get adapterRemoved =>
      _adapterRemovedStreamController.stream;

  /// Stream of devices as they are added.
  Stream<BlueZDevice> get deviceAdded => _deviceAddedStreamController.stream;

  /// Stream of devices as they are removed.
  Stream<BlueZDevice> get deviceRemoved =>
      _deviceRemovedStreamController.stream;

  /// The bus this client is connected to.
  final DBusClient _bus;
  final bool _closeBus;

  /// The root D-Bus BlueZ object.
  late final DBusRemoteObjectManager _root;

  // Objects exported on the bus.
  final _objects = <DBusObjectPath, _BlueZObject>{};

  // Subscription to object manager signals.
  StreamSubscription? _objectManagerSubscription;

  final _adapterAddedStreamController =
      StreamController<BlueZAdapter>.broadcast();
  final _adapterRemovedStreamController =
      StreamController<BlueZAdapter>.broadcast();
  final _deviceAddedStreamController =
      StreamController<BlueZDevice>.broadcast();
  final _deviceRemovedStreamController =
      StreamController<BlueZDevice>.broadcast();

  /// Registered agent.
  _BlueZAgentObject? _agent;

  /// Creates a new BlueZ client. If [bus] is provided connect to the given D-Bus server.
  BlueZClient({DBusClient? bus})
      : _bus = bus ?? DBusClient.system(),
        _closeBus = bus == null {
    _root = DBusRemoteObjectManager(_bus,
        name: 'org.bluez', path: DBusObjectPath('/'));
  }

  /// Connects to the BlueZ daemon.
  /// Must be called before accessing methods and properties.
  Future<void> connect() async {
    // Already connected
    if (_objectManagerSubscription != null) {
      return;
    }

    // Subscribe to changes
    _objectManagerSubscription = _root.signals.listen((signal) {
      if (signal is DBusObjectManagerInterfacesAddedSignal) {
        var object = _objects[signal.changedPath];
        if (object != null) {
          object.updateInterfaces(signal.interfacesAndProperties);
        } else {
          object = _BlueZObject(
              _bus, signal.changedPath, signal.interfacesAndProperties);
          _objects[signal.changedPath] = object;
          if (_isAdapter(object)) {
            _adapterAddedStreamController.add(BlueZAdapter(object));
          } else if (_isDevice(object)) {
            _deviceAddedStreamController.add(BlueZDevice(this, object));
          }
        }
      } else if (signal is DBusObjectManagerInterfacesRemovedSignal) {
        var object = _objects[signal.changedPath];
        if (object != null) {
          // If all the interface are removed, then this object has been removed.
          // Keep the previous values around for the client to use.
          if (object.wouldRemoveAllInterfaces(signal.interfaces)) {
            _objects.remove(signal.changedPath);
          } else {
            object.removeInterfaces(signal.interfaces);
          }

          if (signal.interfaces.contains('org.bluez.Adapter1')) {
            _adapterRemovedStreamController.add(BlueZAdapter(object));
          } else if (signal.interfaces.contains('org.bluez.Device1')) {
            _deviceRemovedStreamController.add(BlueZDevice(this, object));
          }
        }
      } else if (signal is DBusPropertiesChangedSignal) {
        var object = _objects[signal.path];
        if (object != null) {
          object.updateProperties(
              signal.propertiesInterface, signal.changedProperties);
        }
      }
    });

    // Find all the objects exported.
    var objects = await _root.getManagedObjects();
    objects.forEach((objectPath, interfacesAndProperties) {
      _objects[objectPath] =
          _BlueZObject(_bus, objectPath, interfacesAndProperties);
    });

    // Report initial adapters and devices.
    for (var object in _objects.values) {
      if (_isAdapter(object)) {
        _adapterAddedStreamController.add(BlueZAdapter(object));
      } else if (_isDevice(object)) {
        _deviceAddedStreamController.add(BlueZDevice(this, object));
      }
    }
  }

  /// The adapters present on this system.
  /// Use [adapterAdded] and [adapterRemoved] to detect when this list changes.
  List<BlueZAdapter> get adapters {
    var adapters = <BlueZAdapter>[];
    for (var object in _objects.values) {
      if (_isAdapter(object)) {
        adapters.add(BlueZAdapter(object));
      }
    }
    return adapters;
  }

  /// The devices on this system.
  /// Use [deviceAdded] and [deviceRemoved] to detect when this list changes.
  List<BlueZDevice> get devices {
    var devices = <BlueZDevice>[];
    for (var object in _objects.values) {
      if (_isDevice(object)) {
        devices.add(BlueZDevice(this, object));
      }
    }
    return devices;
  }

  /// Registers an agent handler.
  /// A D-Bus object will be registered on [path], which the user must choose to not collide with any other path on the D-Bus client that was passed in the [BlueZClient] constructor.
  Future<void> registerAgent(BlueZAgent agent,
      {DBusObjectPath? path,
      var capability = BlueZAgentCapability.keyboardDisplay}) async {
    if (_agent != null) {
      throw 'Agent already registered';
    }

    var object = _objects[DBusObjectPath('/org/bluez')];
    if (object == null) {
      throw 'Missing /org/bluez object required for agent registration';
    }

    _agent = _BlueZAgentObject(
        this, agent, path ?? DBusObjectPath('/org/bluez/Agent'));
    await _bus.registerObject(_agent!);

    var capabilityString = {
          BlueZAgentCapability.displayOnly: 'DisplayOnly',
          BlueZAgentCapability.displayYesNo: 'DisplayYesNo',
          BlueZAgentCapability.keyboardOnly: 'KeyboardOnly',
          BlueZAgentCapability.noInputNoOutput: 'NoInputNoOutput',
          BlueZAgentCapability.keyboardDisplay: 'KeyboardDisplay',
        }[capability] ??
        '';

    await object.callMethod('org.bluez.AgentManager1', 'RegisterAgent',
        [_agent!.path, DBusString(capabilityString)],
        replySignature: DBusSignature(''));
  }

  /// Unregisters the agent handler previouly registered with [registerAgent].
  Future<void> unregisterAgent() async {
    if (_agent == null) {
      throw 'No agent registered';
    }

    var object = _objects[DBusObjectPath('/org/bluez')];
    if (object == null) {
      throw 'Missing /org/bluez object required for agent unregistration';
    }

    await object.callMethod(
        'org.bluez.AgentManager1', 'UnregisterAgent', [_agent!.path],
        replySignature: DBusSignature(''));
    _agent = null;
  }

  /// Requests that the agent set with [registerAgent] is the system default agent.
  Future<void> requestDefaultAgent() async {
    var object = _objects[DBusObjectPath('/org/bluez')];
    if (object == null) {
      throw 'Missing /org/bluez object required for agent unregistration';
    }

    await object.callMethod(
        'org.bluez.AgentManager1', 'RequestDefaultAgent', [_agent!.path],
        replySignature: DBusSignature(''));
  }

  /// Terminates all active connections. If a client remains unclosed, the Dart process may not terminate.
  Future<void> close() async {
    if (_objectManagerSubscription != null) {
      await _objectManagerSubscription?.cancel();
      _objectManagerSubscription = null;
    }
    if (_closeBus) {
      await _bus.close();
    }
  }

  BlueZAdapter? _getAdapter(DBusObjectPath objectPath) {
    var object = _objects[objectPath];
    if (object == null) {
      return null;
    }
    return BlueZAdapter(object);
  }

  bool _isAdapter(_BlueZObject object) {
    return object.interfaces.containsKey('org.bluez.Adapter1');
  }

  BlueZDevice? _getDevice(DBusObjectPath objectPath) {
    var object = _objects[objectPath];
    if (object == null) {
      return null;
    }
    return BlueZDevice(this, object);
  }

  bool _isDevice(_BlueZObject object) {
    return object.interfaces.containsKey('org.bluez.Device1');
  }

  List<BlueZGattService> _getGattServices(DBusObjectPath parentPath) {
    var services = <BlueZGattService>[];
    for (var object in _objects.values) {
      if (object.path.isInNamespace(parentPath) && _isGattService(object)) {
        services.add(BlueZGattService(this, object));
      }
    }
    return services;
  }

  bool _isGattService(_BlueZObject object) {
    return object.interfaces.containsKey('org.bluez.GattService1');
  }

  List<BlueZGattCharacteristic> _getGattCharacteristics(
      DBusObjectPath parentPath) {
    var characteristics = <BlueZGattCharacteristic>[];
    for (var object in _objects.values) {
      if (object.path.isInNamespace(parentPath) &&
          _isGattCharacteristic(object)) {
        characteristics.add(BlueZGattCharacteristic(this, object));
      }
    }
    return characteristics;
  }

  bool _isGattCharacteristic(_BlueZObject object) {
    return object.interfaces.containsKey('org.bluez.GattCharacteristic1');
  }

  List<BlueZGattDescriptor> _getGattDescriptors(DBusObjectPath parentPath) {
    var descriptors = <BlueZGattDescriptor>[];
    for (var object in _objects.values) {
      if (object.path.isInNamespace(parentPath) && _isGattDescriptor(object)) {
        descriptors.add(BlueZGattDescriptor(object));
      }
    }
    return descriptors;
  }

  bool _isGattDescriptor(_BlueZObject object) {
    return object.interfaces.containsKey('org.bluez.GattDescriptor1');
  }
}
