/*
 * This is the source code of Telegram for Android v. 5.x.x.
 * It is licensed under GNU GPL v. 2 or later.
 * You should have received a copy of the license in this archive (see LICENSE).
 *
 * Copyright Nikolai Kudashov, 2013-2018.
 */

package org.telegram.tgnet;

import android.graphics.Bitmap;
import android.graphics.Path;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.text.TextUtils;

import androidx.annotation.Keep;
import androidx.annotation.Nullable;

import org.telegram.messenger.ApplicationLoader;
import org.telegram.messenger.BuildVars;
import org.telegram.messenger.DialogObject;
import org.telegram.messenger.FileLoader;
import org.telegram.messenger.FileLog;
import org.telegram.messenger.ImageLoader;
import org.telegram.messenger.MediaDataController;
import org.telegram.messenger.MessageObject;
import org.telegram.messenger.MessagesController;
import org.telegram.messenger.SvgHelper;
import org.telegram.messenger.TranslateController;
import org.telegram.messenger.Utilities;
import org.telegram.tgnet.json.TLJsonBuilder;
import org.telegram.tgnet.json.TLJsonParser;
import org.telegram.tgnet.tl.TL_account;
import org.telegram.tgnet.tl.TL_bots;
import org.telegram.tgnet.tl.TL_payments;
import org.telegram.tgnet.tl.TL_phone;
import org.telegram.tgnet.tl.TL_stars;
import org.telegram.tgnet.tl.TL_stats;
import org.telegram.tgnet.tl.TL_stories;
import org.telegram.ui.Stories.MessageMediaStoryFull;
import org.telegram.ui.Stories.MessageMediaStoryFull_old;

import java.util.ArrayList;
import java.util.HashMap;

import me.vkryl.core.BitwiseUtils;

public class TLRPC {

    //public static final int MESSAGE_FLAG_UNREAD             = 0x00000001;
    //public static final int MESSAGE_FLAG_OUT                = 0x00000002;
    public static final int MESSAGE_FLAG_FWD                = 0x00000004;
    public static final int MESSAGE_FLAG_REPLY              = 0x00000008;
    //public static final int MESSAGE_FLAG_MENTION            = 0x00000010;
    //public static final int MESSAGE_FLAG_CONTENT_UNREAD     = 0x00000020;
    public static final int MESSAGE_FLAG_HAS_MARKUP         = 0x00000040;
    public static final int MESSAGE_FLAG_HAS_ENTITIES       = 0x00000080;
    public static final int MESSAGE_FLAG_HAS_FROM_ID        = 0x00000100;
    public static final int MESSAGE_FLAG_HAS_MEDIA          = 0x00000200;
    public static final int MESSAGE_FLAG_HAS_VIEWS          = 0x00000400;
    public static final int MESSAGE_FLAG_HAS_BOT_ID         = 0x00000800;
    public static final int MESSAGE_FLAG_EDITED             = 0x00008000;

    public static final int LAYER = 218;

    public static abstract class EmailVerifyPurpose extends TLObject {

        public static EmailVerifyPurpose TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            EmailVerifyPurpose result = null;
            switch (constructor) {
                case TL_emailVerifyPurposeLoginSetup.constructor:
                    result = new TL_emailVerifyPurposeLoginSetup();
                    break;
                case TL_emailVerifyPurposeLoginChange.constructor:
                    result = new TL_emailVerifyPurposeLoginChange();
                    break;
                case TL_emailVerifyPurposePassport.constructor:
                    result = new TL_emailVerifyPurposePassport();
                    break;
            }
            return TLdeserialize(EmailVerifyPurpose.class, result, stream, constructor, exception);
        }
    }

    public static class TL_emailVerifyPurposeLoginSetup extends EmailVerifyPurpose {
        public static final int constructor = 0x4345be73;

        public String phone_number;
        public String phone_code_hash;

        public void readParams(InputSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            phone_code_hash = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(phone_code_hash);
        }
    }

    public static class TL_inputReplyToMonoForum extends InputReplyTo {
        public static final int constructor = 0x69d66c45;

        public void readParams(InputSerializedData stream, boolean exception) {
            monoforum_peer_id = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            monoforum_peer_id.serializeToStream(stream);
        }
    }

    public static class TL_emailVerifyPurposeLoginChange extends EmailVerifyPurpose {
        public static final int constructor = 0x527d22eb;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_emailVerifyPurposePassport extends EmailVerifyPurpose {
        public static final int constructor = 0xbbf51685;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class EmailVerification extends TLObject {

        public static EmailVerification TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            EmailVerification result = null;
            switch (constructor) {
                case TL_emailVerificationCode.constructor:
                    result = new TL_emailVerificationCode();
                    break;
                case TL_emailVerificationGoogle.constructor:
                    result = new TL_emailVerificationGoogle();
                    break;
                case TL_emailVerificationApple.constructor:
                    result = new TL_emailVerificationApple();
                    break;
            }
            return TLdeserialize(EmailVerification.class, result, stream, constructor, exception);
        }
    }

    public static class TL_emailVerificationCode extends EmailVerification {
        public static final int constructor = 0x922e55a9;

        public String code;

        public void readParams(InputSerializedData stream, boolean exception) {
            code = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(code);
        }
    }

    public static class TL_emailVerificationGoogle extends EmailVerification {
        public static final int constructor = 0xdb909ec2;

        public String token;

        public void readParams(InputSerializedData stream, boolean exception) {
            token = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(token);
        }
    }

    public static class TL_emailVerificationApple extends EmailVerification {
        public static final int constructor = 0x96d074fd;

        public String token;

        public void readParams(InputSerializedData stream, boolean exception) {
            token = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(token);
        }
    }

    public static class TL_chatBannedRights extends TLObject {
        public static final int constructor = 0x9f120418;

        public int flags;
        public boolean view_messages;
        public boolean send_messages;
        public boolean send_media;
        public boolean send_stickers;
        public boolean send_gifs;
        public boolean send_games;
        public boolean send_inline;
        public boolean embed_links;
        public boolean send_polls;
        public boolean change_info;
        public boolean invite_users;
        public boolean pin_messages;
        public boolean manage_topics;
        public boolean send_photos;
        public boolean send_videos;
        public boolean send_roundvideos;
        public boolean send_audios;
        public boolean send_voices;
        public boolean send_docs;
        public boolean send_plain;
        public int until_date;

        public static TL_chatBannedRights TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_chatBannedRights result = TL_chatBannedRights.constructor != constructor ? null : new TL_chatBannedRights();
            return TLdeserialize(TL_chatBannedRights.class, result, stream, constructor, exception);
        }

        public static TL_chatBannedRights clone(TL_chatBannedRights rights) {
            if (rights == null) return null;
            SerializedData data = new SerializedData(rights.getObjectSize());
            rights.serializeToStream(data);
            data = new SerializedData(data.toByteArray());
            return TLdeserialize(data, data.readInt32(false), false);
        }

        @Override
        public boolean equals(@Nullable Object obj) {
            if (!(obj instanceof TL_chatBannedRights)) return false;
            return this.flags == ((TL_chatBannedRights) obj).flags;
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            view_messages = (flags & 1) != 0;
            send_messages = (flags & 2) != 0;
            send_media = (flags & 4) != 0;
            send_stickers = (flags & 8) != 0;
            send_gifs = (flags & 16) != 0;
            send_games = (flags & 32) != 0;
            send_inline = (flags & 64) != 0;
            embed_links = (flags & 128) != 0;
            send_polls = (flags & 256) != 0;
            change_info = (flags & 1024) != 0;
            invite_users = (flags & 32768) != 0;
            pin_messages = (flags & 131072) != 0;
            manage_topics = (flags & 262144) != 0;
            send_photos = (flags & 524288) != 0;
            send_videos = (flags & 1048576) != 0;
            send_roundvideos = (flags & 2097152) != 0;
            send_audios = (flags & 4194304) != 0;
            send_voices = (flags & 8388608) != 0;
            send_docs = (flags & 16777216) != 0;
            send_plain = (flags & 33554432) != 0;
            if (send_media) {
                send_photos = true;
                send_videos = true;
                send_roundvideos = true;
                send_audios = true;
                send_voices = true;
                send_docs = true;
            }
            until_date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            if (ApplicationLoader.isAndroidTestEnvironment()) {
                stream.writeInt32(flags);
                stream.writeInt32(until_date);
                return;
            }

            if (send_photos && send_videos && send_roundvideos && send_audios && send_voices && send_docs) {
                send_media = true;
            } else {
                send_media = false;
            }
            if (send_plain && send_media && send_stickers) {
                send_messages = true;
            } else {
                send_messages = false;
            }
            flags = view_messages ? (flags | 1) : (flags &~ 1);
            flags = send_messages ? (flags | 2) : (flags &~ 2);
            flags = send_media ? (flags | 4) : (flags &~ 4);
            flags = send_stickers ? (flags | 8) : (flags &~ 8);
            flags = send_gifs ? (flags | 16) : (flags &~ 16);
            flags = send_games ? (flags | 32) : (flags &~ 32);
            flags = send_inline ? (flags | 64) : (flags &~ 64);
            flags = embed_links ? (flags | 128) : (flags &~ 128);
            flags = send_polls ? (flags | 256) : (flags &~ 256);
            flags = change_info ? (flags | 1024) : (flags &~ 1024);
            flags = invite_users ? (flags | 32768) : (flags &~ 32768);
            flags = pin_messages ? (flags | 131072) : (flags &~ 131072);
            flags = manage_topics ? (flags | 262144) : (flags &~ 262144);
            flags = send_photos ? (flags | 524288) : (flags &~ 524288);
            flags = send_videos ? (flags | 1048576) : (flags &~ 1048576);
            flags = send_roundvideos ? (flags | 2097152) : (flags &~ 2097152);
            flags = send_audios ? (flags | 4194304) : (flags &~ 4194304);
            flags = send_voices ? (flags | 8388608) : (flags &~ 8388608);
            flags = send_docs ? (flags | 16777216) : (flags &~ 16777216);
            flags = send_plain ? (flags | 33554432) : (flags &~ 33554432);
            stream.writeInt32(flags);
            stream.writeInt32(until_date);
        }
    }

    public static class TL_stickers_suggestedShortName extends TLObject {
        public static final int constructor = 0x85fea03f;

        public String short_name;

        public static TL_stickers_suggestedShortName TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_stickers_suggestedShortName result = TL_stickers_suggestedShortName.constructor != constructor ? null : new TL_stickers_suggestedShortName();
            return TLdeserialize(TL_stickers_suggestedShortName.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            short_name = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(short_name);
        }
    }

    public static abstract class DraftMessage extends TLObject {

        public int flags;
        public boolean no_webpage;
        public boolean invert_media;
        public InputReplyTo reply_to;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public InputMedia media;
        public SuggestedPost suggested_post;
        public int date;
        public long effect;

        public static DraftMessage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            DraftMessage result = null;
            switch (constructor) {
                case TL_draftMessageEmpty.constructor:
                    result = new TL_draftMessageEmpty();
                    break;
                case TL_draftMessageEmpty_layer81.constructor:
                    result = new TL_draftMessageEmpty_layer81();
                    break;
                case TL_draftMessage_layer165.constructor:
                    result = new TL_draftMessage_layer165();
                    break;
                case TL_draftMessage_layer181.constructor:
                    result = new TL_draftMessage_layer181();
                    break;
                case TL_draftMessage_layer205.constructor:
                    result = new TL_draftMessage_layer205();
                    break;
                case TL_draftMessage.constructor:
                    result = new TL_draftMessage();
                    break;
            }
            return TLdeserialize(DraftMessage.class, result, stream, constructor, exception);
        }
    }

    public static class TL_draftMessageEmpty extends DraftMessage {
        public static final int constructor = 0x1b0c841a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(date);
            }
        }
    }

    public static class TL_draftMessageEmpty_layer81 extends TL_draftMessageEmpty {
        public static final int constructor = 0xba4baec5;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_draftMessage extends DraftMessage {
        public static final int constructor = 0x96eaa5eb;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 2) != 0;
            invert_media = (flags & 64) != 0;
            if ((flags & 16) != 0) {
                reply_to = InputReplyTo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            message = stream.readString(exception);
            if ((flags & 8) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            if ((flags & 32) != 0) {
                media = InputMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            date = stream.readInt32(exception);
            if ((flags & 128) != 0) {
                effect = stream.readInt64(exception);
            }
            if (hasFlag(flags, FLAG_8)) {
                suggested_post = SuggestedPost.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 2) : (flags &~ 2);
            flags = invert_media ? (flags | 64) : (flags &~ 64);
            flags = setFlag(flags, FLAG_8, suggested_post != null);
            stream.writeInt32(flags);
            if ((flags & 16) != 0) {
                reply_to.serializeToStream(stream);
            }
            stream.writeString(message);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, entities);
            }
            if ((flags & 32) != 0) {
                media.serializeToStream(stream);
            }
            stream.writeInt32(date);
            if ((flags & 128) != 0) {
                stream.writeInt64(effect);
            }
            if (hasFlag(flags, FLAG_8)) {
                suggested_post.serializeToStream(stream);
            }
        }
    }

    public static class TL_draftMessage_layer205 extends DraftMessage {
        public static final int constructor = 0x2d65321f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 2) != 0;
            invert_media = (flags & 64) != 0;
            if ((flags & 16) != 0) {
                reply_to = InputReplyTo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            message = stream.readString(exception);
            if ((flags & 8) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            if ((flags & 32) != 0) {
                media = InputMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            date = stream.readInt32(exception);
            if ((flags & 128) != 0) {
                effect = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 2) : (flags &~ 2);
            flags = invert_media ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            if ((flags & 16) != 0) {
                reply_to.serializeToStream(stream);
            }
            stream.writeString(message);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, entities);
            }
            if ((flags & 32) != 0) {
                media.serializeToStream(stream);
            }
            stream.writeInt32(date);
            if ((flags & 128) != 0) {
                stream.writeInt64(effect);
            }
        }
    }

    public static class TL_draftMessage_layer181 extends TL_draftMessage {
        public static final int constructor = 0x3fccf7ef;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 2) != 0;
            invert_media = (flags & 64) != 0;
            if ((flags & 16) != 0) {
                reply_to = InputReplyTo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            message = stream.readString(exception);
            if ((flags & 8) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            if ((flags & 32) != 0) {
                media = InputMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 2) : (flags &~ 2);
            flags = invert_media ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            if ((flags & 16) != 0) {
                reply_to.serializeToStream(stream);
            }
            stream.writeString(message);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, entities);
            }
            if ((flags & 32) != 0) {
                media.serializeToStream(stream);
            }
            stream.writeInt32(date);
        }
    }

    public static class TL_draftMessage_layer165 extends TL_draftMessage {
        public static final int constructor = 0xfd8e711f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 2) != 0;
            if ((flags & 1) != 0) {
                TL_inputReplyToMessage reply_to = new TL_inputReplyToMessage();
                reply_to.flags |= 16;
                reply_to.reply_to_msg_id = stream.readInt32(exception);
                this.reply_to = reply_to;
            }
            message = stream.readString(exception);
            if ((flags & 8) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(reply_to instanceof TLRPC.TL_inputReplyToMessage ? ((TL_inputReplyToMessage) reply_to).reply_to_msg_id : 0);
            }
            stream.writeString(message);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, entities);
            }
            stream.writeInt32(date);
        }
    }

    public static abstract class ChatPhoto extends TLObject {

        public int flags;
        public boolean has_video;
        public FileLocation photo_small;
        public FileLocation photo_big;
        public byte[] stripped_thumb;
        public int dc_id;
        public long photo_id;
        public BitmapDrawable strippedBitmap;

        public static ChatPhoto TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            return TLdeserialize(stream, constructor, exception, true);
        }

        public static ChatPhoto TLdeserialize(InputSerializedData stream, int constructor, boolean exception,  boolean allowStripedThumb) {
            ChatPhoto result = null;
            switch (constructor) {
                case 0x1c6e1c11:
                    result = new TL_chatPhoto();
                    break;
                case 0x475cdbd5:
                    result = new TL_chatPhoto_layer115();
                    break;
                case 0x37c1011c:
                    result = new TL_chatPhotoEmpty();
                    break;
                case 0x6153276a:
                    result = new TL_chatPhoto_layer97();
                    break;
                case 0xd20b9f3c:
                    result = new TL_chatPhoto_layer126();
                    break;
                case 0x4790ee05:
                    result = new TL_chatPhoto_layer127();
                    break;
            }
            return TLdeserialize(ChatPhoto.class, result, stream, constructor, exception);
        }
    }

    public static class TL_chatPhoto extends ChatPhoto {
        public static final int constructor = 0x1c6e1c11;

        public void readParams(InputSerializedData stream, boolean exception) {
            readParams(stream, exception, true);
        }

        public void readParams(InputSerializedData stream, boolean exception, boolean allowStripedThumbs) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_id = stream.readInt64(exception);
            if ((flags & 2) != 0) {
                stripped_thumb = stream.readByteArray(exception);
            }
            dc_id = stream.readInt32(exception);
            photo_small = new TL_fileLocationToBeDeprecated();
            photo_small.volume_id = -photo_id;
            photo_small.local_id = 'a';
            photo_big = new TL_fileLocationToBeDeprecated();
            photo_big.volume_id = -photo_id;
            photo_big.local_id = 'c';

            if (allowStripedThumbs && stripped_thumb != null && Build.VERSION.SDK_INT >= 21) {
                try {
                    strippedBitmap = new BitmapDrawable(ImageLoader.getStrippedPhotoBitmap(stripped_thumb, "b"));
                } catch (Throwable e) {
                    FileLog.e(e);
                }
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt64(photo_id);
            if ((flags & 2) != 0) {
                stream.writeByteArray(stripped_thumb);
            }
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhoto_layer115 extends TL_chatPhoto {
        public static final int constructor = 0x475cdbd5;

        public void readParams(InputSerializedData stream, boolean exception) {
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhotoEmpty extends ChatPhoto {
        public static final int constructor = 0x37c1011c;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_chatPhoto_layer97 extends TL_chatPhoto {
        public static final int constructor = 0x6153276a;

        public void readParams(InputSerializedData stream, boolean exception) {
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
        }
    }

    public static class TL_chatPhoto_layer126 extends TL_chatPhoto {
        public static final int constructor = 0xd20b9f3c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_chatPhoto_layer127 extends TL_chatPhoto {
        public static final int constructor = 0x4790ee05;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_video = (flags & 1) != 0;
            photo_small = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_big = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 2) != 0) {
                stripped_thumb = stream.readByteArray(exception);
                if (Build.VERSION.SDK_INT >= 21) {
                    try {
                        strippedBitmap = new BitmapDrawable(ImageLoader.getStrippedPhotoBitmap(stripped_thumb, "b"));
                    } catch (Throwable e) {
                        FileLog.e(e);
                    }
                }
            }
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_video ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            photo_small.serializeToStream(stream);
            photo_big.serializeToStream(stream);
            if ((flags & 2) != 0) {
                stream.writeByteArray(stripped_thumb);
            }
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_help_termsOfService extends TLObject {
        public static final int constructor = 0x780a0310;

        public int flags;
        public boolean popup;
        public TL_dataJSON id;
        public String text;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public int min_age_confirm;

        public static TL_help_termsOfService TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_help_termsOfService result = TL_help_termsOfService.constructor != constructor ? null : new TL_help_termsOfService();
            return TLdeserialize(TL_help_termsOfService.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            popup = (flags & 1) != 0;
            id = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            text = stream.readString(exception);
            entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            if ((flags & 2) != 0) {
                min_age_confirm = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = popup ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            id.serializeToStream(stream);
            stream.writeString(text);
            Vector.serialize(stream, entities);
            if ((flags & 2) != 0) {
                stream.writeInt32(min_age_confirm);
            }
        }
    }

    public static class PaymentReceipt extends TLObject {
        public int flags;
        public int date;
        public long bot_id;
        public long provider_id;
        public String title;
        public String description;
        public WebDocument photo;
        public TL_invoice invoice;
        public TL_paymentRequestedInfo info;
        public TL_shippingOption shipping;
        public long tip_amount;
        public String currency;
        public long total_amount;
        public String credentials_title;
        public ArrayList<User> users = new ArrayList<>();
        public String transaction_id;

        public static PaymentReceipt TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PaymentReceipt result = null;
            switch (constructor) {
                case TL_payments_paymentReceipt.constructor:
                    result = new TL_payments_paymentReceipt();
                    break;
                case TL_payments_paymentReceiptStars.constructor:
                    result = new TL_payments_paymentReceiptStars();
                    break;
            }
            return TLdeserialize(PaymentReceipt.class, result, stream, constructor, exception);
        }
    }

    public static class TL_payments_paymentReceipt extends PaymentReceipt {
        public static final int constructor = 0x70c4fe03;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            date = stream.readInt32(exception);
            bot_id = stream.readInt64(exception);
            provider_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 4) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                info = TL_paymentRequestedInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                shipping = TL_shippingOption.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                tip_amount = stream.readInt64(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            credentials_title = stream.readString(exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(date);
            stream.writeInt64(bot_id);
            stream.writeInt64(provider_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 4) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            if ((flags & 1) != 0) {
                info.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                shipping.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                stream.writeInt64(tip_amount);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(credentials_title);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_payments_paymentReceiptStars extends PaymentReceipt {
        public static final int constructor = 0xdabbf83a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            date = stream.readInt32(exception);
            bot_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 4) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            transaction_id = stream.readString(exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(date);
            stream.writeInt64(bot_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 4) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(transaction_id);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class NotifyPeer extends TLObject {

        public static NotifyPeer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            NotifyPeer result = null;
            switch (constructor) {
                case 0x226e6308:
                    result = new TL_notifyForumTopic();
                    break;
                case 0xd612e8ef:
                    result = new TL_notifyBroadcasts();
                    break;
                case 0xc007cec3:
                    result = new TL_notifyChats();
                    break;
                case 0xb4c83b4c:
                    result = new TL_notifyUsers();
                    break;
                case 0x9fd40bd8:
                    result = new TL_notifyPeer();
                    break;
            }
            return TLdeserialize(NotifyPeer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_notifyForumTopic extends NotifyPeer {
        public static final int constructor = 0x226e6308;

        public Peer peer;
        public int top_msg_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            top_msg_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(top_msg_id);
        }
    }

    public static class TL_notifyBroadcasts extends NotifyPeer {
        public static final int constructor = 0xd612e8ef;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyChats extends NotifyPeer {
        public static final int constructor = 0xc007cec3;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyUsers extends NotifyPeer {
        public static final int constructor = 0xb4c83b4c;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notifyPeer extends NotifyPeer {
        public static final int constructor = 0x9fd40bd8;

        public Peer peer;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_emojiKeywordsDifference extends TLObject {
        public static final int constructor = 0x5cc761bd;

        public String lang_code;
        public int from_version;
        public int version;
        public ArrayList<EmojiKeyword> keywords = new ArrayList<>();

        public static TL_emojiKeywordsDifference TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_emojiKeywordsDifference result = TL_emojiKeywordsDifference.constructor != constructor ? null : new TL_emojiKeywordsDifference();
            return TLdeserialize(TL_emojiKeywordsDifference.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            lang_code = stream.readString(exception);
            from_version = stream.readInt32(exception);
            version = stream.readInt32(exception);
            keywords = Vector.deserialize(stream, EmojiKeyword::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(lang_code);
            stream.writeInt32(from_version);
            stream.writeInt32(version);
            Vector.serialize(stream, keywords);
        }
    }

    public static abstract class messages_SentEncryptedMessage extends TLObject {
        public int date;
        public EncryptedFile file;

        public static messages_SentEncryptedMessage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_SentEncryptedMessage result = null;
            switch (constructor) {
                case 0x560f8935:
                    result = new TL_messages_sentEncryptedMessage();
                    break;
                case 0x9493ff32:
                    result = new TL_messages_sentEncryptedFile();
                    break;
            }
            return TLdeserialize(messages_SentEncryptedMessage.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_sentEncryptedMessage extends messages_SentEncryptedMessage {
        public static final int constructor = 0x560f8935;

        public void readParams(InputSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
        }
    }

    public static class TL_messages_sentEncryptedFile extends messages_SentEncryptedMessage {
        public static final int constructor = 0x9493ff32;

        public void readParams(InputSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
            file = EncryptedFile.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
            file.serializeToStream(stream);
        }
    }

    public static class TL_premiumSubscriptionOption extends TLObject {
        public static final int constructor = 0x5f2d1df2;

        public int flags;
        public boolean current;
        public String transaction;
        public boolean can_purchase_upgrade;
        public int months;
        public String currency;
        public long amount;
        public String bot_url;
        public String store_product;

        public static TL_premiumSubscriptionOption TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            TL_premiumSubscriptionOption result = null;
            switch (constructor) {
                case 0x5f2d1df2:
                    result = new TL_premiumSubscriptionOption();
                    break;
                case 0xb6f11ebe:
                    result = new TL_premiumSubscriptionOption_layer151();
                    break;
            }
            return TLdeserialize(TL_premiumSubscriptionOption.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            current = (flags & 2) != 0;
            if ((flags & 8) != 0) {
                transaction = stream.readString(exception);
            }
            can_purchase_upgrade = (flags & 4) != 0;
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = current ? (flags | 2) : (flags &~ 2);
            flags = can_purchase_upgrade ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            if ((flags & 8) != 0) {
                stream.writeString(transaction);
            }
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_premiumSubscriptionOption_layer151 extends TL_premiumSubscriptionOption {
        public static final int constructor = 0xb6f11ebe;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_premiumGiftOption extends TLObject {
        public static final int constructor = 0x79c059f7;

        public int flags;
        public int months;
        public String currency;
        public long amount;
        public String bot_url;
        public String store_product;

        public static TL_premiumGiftOption TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            TL_premiumGiftOption result = null;
            switch (constructor) {
                case TL_premiumGiftOption.constructor:
                    result = new TL_premiumGiftOption();
                    break;
                case TL_premiumGiftOption_layer199.constructor:
                    result = new TL_premiumGiftOption_layer199();
                    break;
            }
            return TLdeserialize(TL_premiumGiftOption.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            if ((flags & 2) != 0) {
                bot_url = stream.readString(exception);
            }
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            if ((flags & 2) != 0) {
                stream.writeString(bot_url);
            }
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_premiumGiftOption_layer199 extends TL_premiumGiftOption {
        public static final int constructor = 0x74c34319;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            months = stream.readInt32(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
            bot_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                store_product = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32(months);
            stream.writeString(currency);
            stream.writeInt64(amount);
            stream.writeString(bot_url);
            if ((flags & 1) != 0) {
                stream.writeString(store_product);
            }
        }
    }

    public static class TL_error extends TLObject {
        public static final int constructor = 0xc4b9f9bb;

        public int code;
        public String text;

        public static TL_error TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_error result = TL_error.constructor != constructor ? null : new TL_error();
            return TLdeserialize(TL_error.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            code = stream.readInt32(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(code);
            stream.writeString(text);
        }
    }

    public static abstract class UrlAuthResult extends TLObject {

        public static UrlAuthResult TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            UrlAuthResult result = null;
            switch (constructor) {
                case 0xa9d6db1f:
                    result = new TL_urlAuthResultDefault();
                    break;
                case 0x92d33a0e:
                    result = new TL_urlAuthResultRequest();
                    break;
                case 0x8f8c0e4e:
                    result = new TL_urlAuthResultAccepted();
                    break;
            }
            return TLdeserialize(UrlAuthResult.class, result, stream, constructor, exception);
        }
    }

    public static class TL_urlAuthResultDefault extends UrlAuthResult {
        public static final int constructor = 0xa9d6db1f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_urlAuthResultRequest extends UrlAuthResult {
        public static final int constructor = 0x92d33a0e;

        public int flags;
        public boolean request_write_access;
        public User bot;
        public String domain;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            request_write_access = (flags & 1) != 0;
            bot = User.TLdeserialize(stream, stream.readInt32(exception), exception);
            domain = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = request_write_access ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            bot.serializeToStream(stream);
            stream.writeString(domain);
        }
    }

    public static class TL_urlAuthResultAccepted extends UrlAuthResult {
        public static final int constructor = 0x8f8c0e4e;

        public String url;

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_messages_chatFull extends TLObject {
        public static final int constructor = 0xe5d7d19c;

        public ChatFull full_chat;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatFull TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_chatFull result = TL_messages_chatFull.constructor != constructor ? null : new TL_messages_chatFull();
            return TLdeserialize(TL_messages_chatFull.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            full_chat = ChatFull.TLdeserialize(stream, stream.readInt32(exception), exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            full_chat.serializeToStream(stream);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class DocumentAttribute extends TLObject {
        public String alt;
        public InputStickerSet stickerset;
        public double duration;
        public int flags;
        public TL_maskCoords mask_coords;
        public boolean round_message;
        public boolean supports_streaming;
        public String file_name;
        public int w;
        public int h;
        public boolean mask;
        public String title;
        public String performer;
        public boolean voice;
        public byte[] waveform;
        public int preload_prefix_size;
        public double video_start_ts;
        public boolean nosound;
        public String video_codec;

        public static DocumentAttribute TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            DocumentAttribute result = null;
            switch (constructor) {
                case 0x3a556302:
                    result = new TL_documentAttributeSticker_layer55();
                    break;
                case 0xef02ce6:
                    result = new TL_documentAttributeVideo_layer159();
                    break;
                case 0x51448e5:
                    result = new TL_documentAttributeAudio_old();
                    break;
                case 0x6319d612:
                    result = new TL_documentAttributeSticker();
                    break;
                case 0x11b58939:
                    result = new TL_documentAttributeAnimated();
                    break;
                case 0x15590068:
                    result = new TL_documentAttributeFilename();
                    break;
                case TL_documentAttributeVideo.constructor:
                    result = new TL_documentAttributeVideo();
                    break;
                case TL_documentAttributeVideo_layer187.constructor:
                    result = new TL_documentAttributeVideo_layer187();
                    break;
                case TL_documentAttributeVideo_layer184.constructor:
                    result = new TL_documentAttributeVideo_layer184();
                    break;
                case 0x5910cccb:
                    result = new TL_documentAttributeVideo_layer65();
                    break;
                case 0xded218e0:
                    result = new TL_documentAttributeAudio_layer45();
                    break;
                case 0xfb0a5727:
                    result = new TL_documentAttributeSticker_old();
                    break;
                case 0x9801d2f7:
                    result = new TL_documentAttributeHasStickers();
                    break;
                case 0x994c9882:
                    result = new TL_documentAttributeSticker_old2();
                    break;
                case 0x6c37c15c:
                    result = new TL_documentAttributeImageSize();
                    break;
                case 0x9852f9c6:
                    result = new TL_documentAttributeAudio();
                    break;
                case 0xfd149899:
                    result = new TL_documentAttributeCustomEmoji();
                    break;
            }
            return TLdeserialize(DocumentAttribute.class, result, stream, constructor, exception);
        }
    }

    public static class TL_documentAttributeSticker_layer55 extends TL_documentAttributeSticker {
        public static final int constructor = 0x3a556302;

        public void readParams(InputSerializedData stream, boolean exception) {
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
        }
    }

    public static class TL_documentAttributeAudio_old extends TL_documentAttributeAudio {
        public static final int constructor = 0x51448e5;

        public void readParams(InputSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) duration);
        }
    }

    public static class TL_documentAttributeSticker extends DocumentAttribute {
        public static final int constructor = 0x6319d612;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            mask = (flags & 2) != 0;
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                mask_coords = TL_maskCoords.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = mask ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
            if ((flags & 1) != 0) {
                mask_coords.serializeToStream(stream);
            }
        }
    }

    public static class TL_documentAttributeAnimated extends DocumentAttribute {
        public static final int constructor = 0x11b58939;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeFilename extends DocumentAttribute {
        public static final int constructor = 0x15590068;

        public void readParams(InputSerializedData stream, boolean exception) {
            file_name = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(file_name);
        }
    }

    public static class TL_documentAttributeVideo extends DocumentAttribute {
        public static final int constructor = 0x43c57c48;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            round_message = (flags & 1) != 0;
            supports_streaming = (flags & 2) != 0;
            nosound = (flags & 8) != 0;
            duration = stream.readDouble(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                preload_prefix_size = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
            if ((flags & 32) != 0) {
                video_codec = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = round_message ? (flags | 1) : (flags &~ 1);
            flags = supports_streaming ? (flags | 2) : (flags &~ 2);
            flags = nosound ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeDouble(duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
            if ((flags & 4) != 0) {
                stream.writeInt32(preload_prefix_size);
            }
            if ((flags & 16) != 0) {
                stream.writeDouble(video_start_ts);
            }
            if ((flags & 32) != 0) {
                stream.writeString(video_codec);
            }
        }
    }

    public static class TL_documentAttributeVideo_layer187 extends TL_documentAttributeVideo {
        public static final int constructor = 0x17399fad;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            round_message = (flags & 1) != 0;
            supports_streaming = (flags & 2) != 0;
            nosound = (flags & 8) != 0;
            duration = stream.readDouble(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                preload_prefix_size = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = round_message ? (flags | 1) : (flags &~ 1);
            flags = supports_streaming ? (flags | 2) : (flags &~ 2);
            flags = nosound ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeDouble(duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
            if ((flags & 4) != 0) {
                stream.writeInt32(preload_prefix_size);
            }
            if ((flags & 16) != 0) {
                stream.writeDouble(video_start_ts);
            }
        }
    }

    public static class TL_documentAttributeVideo_layer184 extends TL_documentAttributeVideo {
        public static final int constructor = 0xd38ff1c2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            round_message = (flags & 1) != 0;
            supports_streaming = (flags & 2) != 0;
            nosound = (flags & 8) != 0;
            duration = stream.readDouble(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                preload_prefix_size = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = round_message ? (flags | 1) : (flags &~ 1);
            flags = supports_streaming ? (flags | 2) : (flags &~ 2);
            flags = nosound ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeDouble(duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
            if ((flags & 4) != 0) {
                stream.writeInt32(preload_prefix_size);
            }
        }
    }

    public static class TL_documentAttributeVideo_layer159 extends TL_documentAttributeVideo {
        public static final int constructor = 0xef02ce6;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            round_message = (flags & 1) != 0;
            supports_streaming = (flags & 2) != 0;
            duration = stream.readInt32(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = round_message ? (flags | 1) : (flags &~ 1);
            flags = supports_streaming ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32((int) duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeVideo_layer65 extends TL_documentAttributeVideo {
        public static final int constructor = 0x5910cccb;

        public void readParams(InputSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) duration);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeAudio_layer45 extends TL_documentAttributeAudio {
        public static final int constructor = 0xded218e0;

        public void readParams(InputSerializedData stream, boolean exception) {
            duration = stream.readInt32(exception);
            title = stream.readString(exception);
            performer = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) duration);
            stream.writeString(title);
            stream.writeString(performer);
        }
    }

    public static class TL_documentAttributeSticker_old extends TL_documentAttributeSticker {
        public static final int constructor = 0xfb0a5727;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeHasStickers extends DocumentAttribute {
        public static final int constructor = 0x9801d2f7;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_documentAttributeSticker_old2 extends TL_documentAttributeSticker {
        public static final int constructor = 0x994c9882;

        public void readParams(InputSerializedData stream, boolean exception) {
            alt = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(alt);
        }
    }

    public static class TL_documentAttributeImageSize extends DocumentAttribute {
        public static final int constructor = 0x6c37c15c;

        public void readParams(InputSerializedData stream, boolean exception) {
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_documentAttributeAudio extends DocumentAttribute {
        public static final int constructor = 0x9852f9c6;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            voice = (flags & 1024) != 0;
            duration = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                performer = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                waveform = stream.readByteArray(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = voice ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32((int) duration);
            if ((flags & 1) != 0) {
                stream.writeString(title);
            }
            if ((flags & 2) != 0) {
                stream.writeString(performer);
            }
            if ((flags & 4) != 0) {
                stream.writeByteArray(waveform);
            }
        }
    }

    public static class TL_documentAttributeCustomEmoji extends DocumentAttribute {
        public static final int constructor = 0xfd149899;

        public boolean free;
        public boolean text_color;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            free = (flags & 1) != 0;
            text_color = (flags & 2) != 0;
            alt = stream.readString(exception);
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = free ? (flags | 1) : (flags &~ 1);
            flags = text_color ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(alt);
            stickerset.serializeToStream(stream);
        }
    }

    public static class TL_textPhone extends RichText {
        public static final int constructor = 0x1ccb966a;

        public String phone;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            phone = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(phone);
        }
    }

    public static class TL_textSuperscript extends RichText {
        public static final int constructor = 0xc7fb5e01;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textImage extends RichText {
        public static final int constructor = 0x81ccf4f;

        public long document_id;
        public long photo_id; //custom
        public int w;
        public int h;

        public void readParams(InputSerializedData stream, boolean exception) {
            document_id = stream.readInt64(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(document_id);
            stream.writeInt32(w);
            stream.writeInt32(h);
        }
    }

    public static class TL_textEmpty extends RichText {
        public static final int constructor = 0xdc3d824f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_textUrl extends RichText {
        public static final int constructor = 0x3c2884c1;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            url = stream.readString(exception);
            webpage_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(url);
            stream.writeInt64(webpage_id);
        }
    }

    public static class TL_textAnchor extends RichText {
        public static final int constructor = 0x35553762;
        public String name;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            name = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(name);
        }
    }

    public static class TL_textStrike extends RichText {
        public static final int constructor = 0x9bf8bb95;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textMarked extends RichText {
        public static final int constructor = 0x34b8621;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textFixed extends RichText {
        public static final int constructor = 0x6c3f19b9;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textEmail extends RichText {
        public static final int constructor = 0xde5a0dd6;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            email = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeString(email);
        }
    }

    public static class TL_textPlain extends RichText {
        public static final int constructor = 0x744694e0;

        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_textConcat extends RichText {
        public static final int constructor = 0x7e6260d7;

        public void readParams(InputSerializedData stream, boolean exception) {
            texts = Vector.deserialize(stream, RichText::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, texts);
        }
    }

    public static class TL_textBold extends RichText {
        public static final int constructor = 0x6724abc4;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textItalic extends RichText {
        public static final int constructor = 0xd912a59c;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textUnderline extends RichText {
        public static final int constructor = 0xc12622c4;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_textSubscript extends RichText {
        public static final int constructor = 0xed6a8504;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
        }
    }

    public static class TL_statsURL extends TLObject {
        public static final int constructor = 0x47a971e0;

        public String url;

        public static TL_statsURL TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_statsURL result = TL_statsURL.constructor != constructor ? null : new TL_statsURL();
            return TLdeserialize(TL_statsURL.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_popularContact extends TLObject {
        public static final int constructor = 0x5ce14175;

        public long client_id;
        public int importers;

        public static TL_popularContact TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_popularContact result = TL_popularContact.constructor != constructor ? null : new TL_popularContact();
            return TLdeserialize(TL_popularContact.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            client_id = stream.readInt64(exception);
            importers = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(client_id);
            stream.writeInt32(importers);
        }
    }

    public static class TL_messages_botCallbackAnswer extends TLObject {
        public static final int constructor = 0x36585ea4;

        public int flags;
        public boolean alert;
        public boolean has_url;
        public boolean native_ui;
        public String message;
        public String url;
        public int cache_time;

        public static TL_messages_botCallbackAnswer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_botCallbackAnswer result = TL_messages_botCallbackAnswer.constructor != constructor ? null : new TL_messages_botCallbackAnswer();
            return TLdeserialize(TL_messages_botCallbackAnswer.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            alert = (flags & 2) != 0;
            has_url = (flags & 8) != 0;
            native_ui = (flags & 16) != 0;
            if ((flags & 1) != 0) {
                message = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                url = stream.readString(exception);
            }
            cache_time = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = alert ? (flags | 2) : (flags &~ 2);
            flags = has_url ? (flags | 8) : (flags &~ 8);
            flags = native_ui ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(message);
            }
            if ((flags & 4) != 0) {
                stream.writeString(url);
            }
            stream.writeInt32(cache_time);
        }
    }

    public static class TL_contactStatus extends TLObject {
        public static final int constructor = 0x16d9703b;

        public long user_id;
        public UserStatus status;

        public static TL_contactStatus TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_contactStatus result = TL_contactStatus.constructor != constructor ? null : new TL_contactStatus();
            return TLdeserialize(TL_contactStatus.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            status.serializeToStream(stream);
        }
    }

    public static abstract class GroupCall extends TLObject {

        public int flags;
        public boolean join_muted;
        public boolean can_change_join_muted;
        public boolean join_date_asc;
        public boolean schedule_start_subscribed;
        public boolean can_start_video;
        public boolean record_video_active;
        public boolean conference;
        public boolean rtmp_stream;
        public boolean listeners_hidden;
        public boolean creator;
        public long id;
        public long access_hash;
        public int participants_count;
        public String title;
        public int stream_dc_id;
        public int record_start_date;
        public int schedule_date;
        public int unmuted_video_count;
        public int unmuted_video_limit;
        public int version;
        public int duration;
        public long conference_from_call;
        public String invite_link;
        public boolean messages_enabled;
        public boolean can_change_messages_enabled;
        public boolean min;
        public long send_paid_messages_stars;
        public TLRPC.Peer default_send_as;

        public static GroupCall TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            return TLdeserialize(GroupCall.class, fromConstructor(constructor), stream, constructor, exception);
        }

        private static GroupCall fromConstructor(int constructor) {
            switch (constructor) {
                case TL_groupCallDiscarded.constructor:     return new TL_groupCallDiscarded();
                case TL_groupCall.constructor:              return new TL_groupCall();
                case TL_groupCall_layer217_1.constructor:   return new TL_groupCall_layer217_1();
                case TL_groupCall_layer216.constructor:     return new TL_groupCall_layer216();
                case TL_groupCall_layer201_2.constructor:   return new TL_groupCall_layer201_2();
                case TL_groupCall_layer201.constructor:     return new TL_groupCall_layer201();
            }
            return null;
        }
    }

    public static class TL_groupCallDiscarded extends GroupCall {
        public static final int constructor = 0x7780bcb4;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            duration = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(duration);
        }
    }

    public static class TL_groupCall extends GroupCall {
        public static final int constructor = 0xefb2b617;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            conference = (flags & 16384) != 0;
            creator = (flags & 32768) != 0;
            messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_17);
            can_change_messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_18);
            min = BitwiseUtils.hasFlag(flags, FLAG_19);
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
            if ((flags & 65536) != 0) {
                invite_link = stream.readString(exception);
            }
            if (hasFlag(flags, FLAG_20)) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
            if (hasFlag(flags, FLAG_21)) {
                default_send_as = TLRPC.Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            flags = conference ? (flags | 16384) : (flags &~ 16384);
            flags = creator ? (flags | 32768) : (flags &~ 32768);
            flags = BitwiseUtils.setFlag(flags, FLAG_17, messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_18, can_change_messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_19, min);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
            if ((flags & 65536) != 0) {
                stream.writeString(invite_link);
            }
            if (hasFlag(flags, FLAG_20)) {
                stream.writeInt64(send_paid_messages_stars);
            }
            if (hasFlag(flags, FLAG_21)) {
                default_send_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_groupCall_layer217_1 extends TL_groupCall {
        public static final int constructor = 0xd7ca61d8;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            conference = (flags & 16384) != 0;
            creator = (flags & 32768) != 0;
            messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_17);
            can_change_messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_18);
            min = BitwiseUtils.hasFlag(flags, FLAG_19);
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
            if ((flags & 65536) != 0) {
                invite_link = stream.readString(exception);
            }
            if (hasFlag(flags, FLAG_20)) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            flags = conference ? (flags | 16384) : (flags &~ 16384);
            flags = creator ? (flags | 32768) : (flags &~ 32768);
            flags = BitwiseUtils.setFlag(flags, FLAG_17, messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_18, can_change_messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_19, min);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
            if ((flags & 65536) != 0) {
                stream.writeString(invite_link);
            }
            if (hasFlag(flags, FLAG_20)) {
                stream.writeInt64(send_paid_messages_stars);
            }
        }
    }

    public static class TL_groupCall_layer216 extends TL_groupCall {
        public static final int constructor = 0x553b0ba1;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            conference = (flags & 16384) != 0;
            creator = (flags & 32768) != 0;
            messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_17);
            can_change_messages_enabled = BitwiseUtils.hasFlag(flags, FLAG_18);
            min = BitwiseUtils.hasFlag(flags, FLAG_19);
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
            if ((flags & 65536) != 0) {
                invite_link = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            flags = conference ? (flags | 16384) : (flags &~ 16384);
            flags = creator ? (flags | 32768) : (flags &~ 32768);
            flags = BitwiseUtils.setFlag(flags, FLAG_17, messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_18, can_change_messages_enabled);
            flags = BitwiseUtils.setFlag(flags, FLAG_19, min);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
            if ((flags & 65536) != 0) {
                stream.writeString(invite_link);
            }
        }
    }

    public static class TL_groupCall_layer201_2 extends TL_groupCall {
        public static final int constructor = 0xd597650c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            conference = (flags & 16384) != 0;
            creator = (flags & 32768) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
            if ((flags & 65536) != 0) {
                invite_link = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            flags = conference ? (flags | 16384) : (flags &~ 16384);
            flags = creator ? (flags | 32768) : (flags &~ 32768);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
            if ((flags & 65536) != 0) {
                stream.writeString(invite_link);
            }
        }
    }

    public static class TL_groupCall_layer201 extends TL_groupCall {
        public static final int constructor = 0xcdf8d3e3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            join_muted = (flags & 2) != 0;
            can_change_join_muted = (flags & 4) != 0;
            join_date_asc = (flags & 64) != 0;
            schedule_start_subscribed = (flags & 256) != 0;
            can_start_video = (flags & 512) != 0;
            record_video_active = (flags & 2048) != 0;
            rtmp_stream = (flags & 4096) != 0;
            listeners_hidden = (flags & 8192) != 0;
            conference = (flags & 16384) != 0;
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                stream_dc_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                record_start_date = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                schedule_date = stream.readInt32(exception);
            }
            if ((flags & 1024) != 0) {
                unmuted_video_count = stream.readInt32(exception);
            }
            unmuted_video_limit = stream.readInt32(exception);
            version = stream.readInt32(exception);
            if ((flags & 16384) != 0) {
                conference_from_call = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = join_muted ? (flags | 2) : (flags &~ 2);
            flags = can_change_join_muted ? (flags | 4) : (flags &~ 4);
            flags = join_date_asc ? (flags | 64) : (flags &~ 64);
            flags = schedule_start_subscribed ? (flags | 256) : (flags &~ 256);
            flags = can_start_video ? (flags | 512) : (flags &~ 512);
            flags = record_video_active ? (flags | 2048) : (flags &~ 2048);
            flags = rtmp_stream ? (flags | 4096) : (flags &~ 4096);
            flags = listeners_hidden ? (flags | 8192) : (flags &~ 8192);
            flags = conference ? (flags | 16384) : (flags &~ 16384);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(participants_count);
            if ((flags & 8) != 0) {
                stream.writeString(title);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(stream_dc_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(record_start_date);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(schedule_date);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(unmuted_video_count);
            }
            stream.writeInt32(unmuted_video_limit);
            stream.writeInt32(version);
            if ((flags & 16384) != 0) {
                stream.writeInt64(conference_from_call);
            }
        }
    }

    public static class TL_channelBannedRights_layer92 extends TLObject {
        public static final int constructor = 0x58cf4249;

        public int flags;
        public boolean view_messages;
        public boolean send_messages;
        public boolean send_media;
        public boolean send_stickers;
        public boolean send_gifs;
        public boolean send_games;
        public boolean send_inline;
        public boolean embed_links;
        public int until_date;

        public static TL_channelBannedRights_layer92 TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_channelBannedRights_layer92 result = TL_channelBannedRights_layer92.constructor != constructor ? null : new TL_channelBannedRights_layer92();
            return TLdeserialize(TL_channelBannedRights_layer92.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            view_messages = (flags & 1) != 0;
            send_messages = (flags & 2) != 0;
            send_media = (flags & 4) != 0;
            send_stickers = (flags & 8) != 0;
            send_gifs = (flags & 16) != 0;
            send_games = (flags & 32) != 0;
            send_inline = (flags & 64) != 0;
            embed_links = (flags & 128) != 0;
            until_date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = view_messages ? (flags | 1) : (flags &~ 1);
            flags = send_messages ? (flags | 2) : (flags &~ 2);
            flags = send_media ? (flags | 4) : (flags &~ 4);
            flags = send_stickers ? (flags | 8) : (flags &~ 8);
            flags = send_gifs ? (flags | 16) : (flags &~ 16);
            flags = send_games ? (flags | 32) : (flags &~ 32);
            flags = send_inline ? (flags | 64) : (flags &~ 64);
            flags = embed_links ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32(until_date);
        }
    }

    public static abstract class DialogPeer extends TLObject {

        public static DialogPeer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            DialogPeer result = null;
            switch (constructor) {
                case 0xe56dbf05:
                    result = new TL_dialogPeer();
                    break;
                case 0x514519e2:
                    result = new TL_dialogPeerFolder();
                    break;
            }
            return TLdeserialize(DialogPeer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_dialogPeer extends DialogPeer {
        public static final int constructor = 0xe56dbf05;

        public Peer peer;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_dialogPeerFolder extends DialogPeer {
        public static final int constructor = 0x514519e2;

        public int folder_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            folder_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(folder_id);
        }
    }

    public static abstract class MessagePeerReaction extends TLObject {

        public int flags;
        public boolean big;
        public boolean unread;
        public Peer peer_id;
        public Reaction reaction;
        public int date;
        public boolean dateIsSeen; //custom

        public static MessagePeerReaction TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            MessagePeerReaction result = null;
            switch (constructor) {
                case 0x8c79b63c:
                    result = new TL_messagePeerReaction();
                    break;
                case 0xb156fe9c:
                    result = new TL_messagePeerReaction_layer154();
                    break;
                case 0x51b67eff:
                    result = new TL_messagePeerReaction_layer144();
                    break;
                case 0x932844fa:
                    result = new TL_messagePeerReaction_layer137();
                    break;
            }
            return TLdeserialize(MessagePeerReaction.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messagePeerReaction extends MessagePeerReaction {
        public static final int constructor = 0x8c79b63c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            date = stream.readInt32(exception);
            reaction = Reaction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            stream.writeInt32(date);
            reaction.serializeToStream(stream);
        }
    }

    public static class TL_messagePeerReaction_layer154 extends MessagePeerReaction {
        public static final int constructor = 0xb156fe9c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            reaction = Reaction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            reaction.serializeToStream(stream);
        }
    }

    public static class TL_messagePeerReaction_layer144 extends MessagePeerReaction {
        public static final int constructor = 0x51b67eff;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            unread = (flags & 2) != 0;
            peer_id = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            reaction = new TL_reactionEmoji();
            ((TL_reactionEmoji) reaction).emoticon = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            flags = unread ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            peer_id.serializeToStream(stream);
            if (reaction instanceof TL_reactionEmoji)
                stream.writeString(((TL_reactionEmoji) reaction).emoticon);
            else
                stream.writeString("");
        }
    }

    public static class TL_messagePeerReaction_layer137 extends MessagePeerReaction {
        public static final int constructor = 0x932844fa;

        public long user_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            reaction = new TL_reactionEmoji();
            ((TL_reactionEmoji) reaction).emoticon = stream.readString(exception);
            peer_id = new TL_peerUser();
            peer_id.user_id = user_id;
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            if (reaction instanceof TL_reactionEmoji)
                stream.writeString(((TL_reactionEmoji) reaction).emoticon);
            else
                stream.writeString("");
        }
    }

    public static abstract class auth_Authorization extends TLObject {

        public static auth_Authorization TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            auth_Authorization result = null;
            switch (constructor) {
                case 0x44747e9a:
                    result = new TL_auth_authorizationSignUpRequired();
                    break;
                case 0x33fb7bb8://TODO old constructor need remove
                    result = new TL_auth_authorization();
                    break;
                case 0x2ea2c0d4:
                    result = new TL_auth_authorization();
                    break;
            }
            return TLdeserialize(auth_Authorization.class, result, stream, constructor, exception);
        }
    }

    public static class TL_auth_authorizationSignUpRequired extends auth_Authorization {
        public static final int constructor = 0x44747e9a;

        public int flags;
        public TL_help_termsOfService terms_of_service;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                terms_of_service = TL_help_termsOfService.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                terms_of_service.serializeToStream(stream);
            }
        }
    }

    public static class TL_auth_authorization extends auth_Authorization {
        public static final int constructor = 0x2ea2c0d4;

        public int flags;
        public boolean setup_password_required;
        public int otherwise_relogin_days;
        public int tmp_sessions;
        public byte[] future_auth_token;
        public User user;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            setup_password_required = (flags & 2) != 0;
            if ((flags & 2) != 0) {
                otherwise_relogin_days = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                tmp_sessions = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                future_auth_token = stream.readByteArray(exception);
            }
            user = User.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = setup_password_required ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                stream.writeInt32(otherwise_relogin_days);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(tmp_sessions);
            }
            if ((flags & 4) != 0) {
                stream.writeByteArray(future_auth_token);
            }
            user.serializeToStream(stream);
        }
    }

    public static class PollAnswer extends TLObject {
        public TL_textWithEntities text = new TL_textWithEntities();
        public byte[] option;

        public static PollAnswer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PollAnswer result = null;
            switch (constructor) {
                case TL_pollAnswer.constructor:
                    result = new TL_pollAnswer();
                    break;
                case TL_pollAnswer_layer178.constructor:
                    result = new TL_pollAnswer_layer178();
                    break;
            }
            return TLdeserialize(PollAnswer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_pollAnswer extends PollAnswer {
        public static final int constructor = 0xff16e2ca;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = TL_textWithEntities.TLdeserialize(stream, stream.readInt32(exception), exception);
            option = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            stream.writeByteArray(option);
        }
    }

    public static class TL_pollAnswer_layer178 extends TL_pollAnswer {
        public static final int constructor = 0x6ca9c2e9;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = new TL_textWithEntities();
            text.text = stream.readString(exception);
            option = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text == null ? "" : text.text);
            stream.writeByteArray(option);
        }
    }

    public static abstract class JSONValue extends TLObject {

        public static JSONValue TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            JSONValue result = null;
            switch (constructor) {
                case 0xc7345e6a:
                    result = new TL_jsonBool();
                    break;
                case 0x3f6d7b68:
                    result = new TL_jsonNull();
                    break;
                case 0xb71e767a:
                    result = new TL_jsonString();
                    break;
                case 0xf7444763:
                    result = new TL_jsonArray();
                    break;
                case 0x99c1d49d:
                    result = new TL_jsonObject();
                    break;
                case 0x2be0dfa4:
                    result = new TL_jsonNumber();
                    break;
            }
            return TLdeserialize(JSONValue.class, result, stream, constructor, exception);
        }
    }

    public static class TL_jsonBool extends JSONValue {
        public static final int constructor = 0xc7345e6a;

        public boolean value;

        public void readParams(InputSerializedData stream, boolean exception) {
            value = stream.readBool(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeBool(value);
        }
    }

    public static class TL_jsonNull extends JSONValue {
        public static final int constructor = 0x3f6d7b68;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_jsonString extends JSONValue {
        public static final int constructor = 0xb71e767a;

        public String value;

        public void readParams(InputSerializedData stream, boolean exception) {
            value = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(value);
        }
    }

    public static class TL_jsonArray extends JSONValue {
        public static final int constructor = 0xf7444763;

        public ArrayList<JSONValue> value = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            value = Vector.deserialize(stream, JSONValue::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, value);
        }
    }

    public static class TL_jsonObject extends JSONValue {
        public static final int constructor = 0x99c1d49d;

        public ArrayList<TL_jsonObjectValue> value = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            value = Vector.deserialize(stream, TL_jsonObjectValue::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, value);
        }
    }

    public static class TL_jsonNumber extends JSONValue {
        public static final int constructor = 0x2be0dfa4;

        public double value;

        public void readParams(InputSerializedData stream, boolean exception) {
            value = stream.readDouble(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(value);
        }
    }

    public static abstract class InputWallPaper extends TLObject {

        public static InputWallPaper TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputWallPaper result = null;
            switch (constructor) {
                case 0xe630b979:
                    result = new TL_inputWallPaper();
                    break;
                case 0x967a462e:
                    result = new TL_inputWallPaperNoFile();
                    break;
                case 0x72091c80:
                    result = new TL_inputWallPaperSlug();
                    break;
            }
            return TLdeserialize(InputWallPaper.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputWallPaper extends InputWallPaper {
        public static final int constructor = 0xe630b979;

        public long id;
        public long access_hash;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputWallPaperNoFile extends InputWallPaper {
        public static final int constructor = 0x967a462e;

        public long id;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_inputWallPaperSlug extends InputWallPaper {
        public static final int constructor = 0x72091c80;

        public String slug;

        public void readParams(InputSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static class TL_messages_historyImportParsed extends TLObject {
        public static final int constructor = 0x5e0fb7b9;

        public int flags;
        public boolean pm;
        public boolean group;
        public String title;

        public static TL_messages_historyImportParsed TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_historyImportParsed result = TL_messages_historyImportParsed.constructor != constructor ? null : new TL_messages_historyImportParsed();
            return TLdeserialize(TL_messages_historyImportParsed.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            pm = (flags & 1) != 0;
            group = (flags & 2) != 0;
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = pm ? (flags | 1) : (flags &~ 1);
            flags = group ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
        }
    }

    public static class TL_folder extends TLObject {
        public static final int constructor = 0xff544e65;

        public int flags;
        public boolean autofill_new_broadcasts;
        public boolean autofill_public_groups;
        public boolean autofill_new_correspondents;
        public int id;
        public String title;
        public ChatPhoto photo;

        public static TL_folder TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_folder result = TL_folder.constructor != constructor ? null : new TL_folder();
            return TLdeserialize(TL_folder.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            autofill_new_broadcasts = (flags & 1) != 0;
            autofill_public_groups = (flags & 2) != 0;
            autofill_new_correspondents = (flags & 4) != 0;
            id = stream.readInt32(exception);
            title = stream.readString(exception);
            if ((flags & 8) != 0) {
                photo = ChatPhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = autofill_new_broadcasts ? (flags | 1) : (flags &~ 1);
            flags = autofill_public_groups ? (flags | 2) : (flags &~ 2);
            flags = autofill_new_correspondents ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title);
            if ((flags & 8) != 0) {
                photo.serializeToStream(stream);
            }
        }
    }

    public static abstract class messages_Messages extends TLObject {
        public ArrayList<Message> messages = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public ArrayList<TL_forumTopic> topics = new ArrayList<>();
        public int flags;
        public boolean inexact;
        public int pts;
        public int count;
        public int next_rate;
        public int offset_id_offset;
        public ArrayList<Document> animatedEmoji;
        public SearchPostsFlood search_flood;

        public static messages_Messages TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_Messages result = null;
            switch (constructor) {
                case TL_messages_messagesSlice.constructor:
                    result = new TL_messages_messagesSlice();
                    break;
                case TL_messages_messagesSlice_layer215.constructor:
                    result = new TL_messages_messagesSlice_layer215();
                    break;
                case TL_messages_messagesSlice_layer210.constructor:
                    result = new TL_messages_messagesSlice_layer210();
                    break;
                case TL_messages_messages.constructor:
                    result = new TL_messages_messages();
                    break;
                case TL_messages_messages_layer215.constructor:
                    result = new TL_messages_messages_layer215();
                    break;
                case TL_messages_channelMessages.constructor:
                    result = new TL_messages_channelMessages();
                    break;
                case TL_messages_messagesNotModified.constructor:
                    result = new TL_messages_messagesNotModified();
                    break;
            }
            return TLdeserialize(messages_Messages.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_messagesSlice extends messages_Messages {
        public static final int constructor = 0x5F206716;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            count = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                next_rate = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                search_flood = SearchPostsFlood.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            topics = Vector.deserialize(stream, TL_forumTopic::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(count);
            if ((flags & 1) != 0) {
                stream.writeInt32(next_rate);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            if ((flags & 8) != 0) {
                search_flood.serializeToStream(stream);
            }
            Vector.serialize(stream, messages);
            Vector.serialize(stream, topics);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_messagesSlice_layer215 extends  TL_messages_messagesSlice {
        public static final int constructor = 0x762b263d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            count = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                next_rate = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                search_flood = SearchPostsFlood.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(count);
            if ((flags & 1) != 0) {
                stream.writeInt32(next_rate);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            if ((flags & 8) != 0) {
                search_flood.serializeToStream(stream);
            }
            Vector.serialize(stream, messages);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_messagesSlice_layer210 extends TL_messages_messagesSlice {
        public static final int constructor = 0x3a54685e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            count = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                next_rate = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(count);
            if ((flags & 1) != 0) {
                stream.writeInt32(next_rate);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            Vector.serialize(stream, messages);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_messages extends messages_Messages {
        public static final int constructor = 0x1D73E7EA;

        public void readParams(InputSerializedData stream, boolean exception) {
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            topics = Vector.deserialize(stream, TL_forumTopic::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, messages);
            Vector.serialize(stream, topics);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_messages_layer215 extends messages_Messages {
        public static final int constructor = 0x8c718e87;

        public void readParams(InputSerializedData stream, boolean exception) {
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, messages);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_channelMessages extends messages_Messages {
        public static final int constructor = 0xc776ba4e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            pts = stream.readInt32(exception);
            count = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                offset_id_offset = stream.readInt32(exception);
            }
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            topics = Vector.deserialize(stream, TL_forumTopic::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(pts);
            stream.writeInt32(count);
            if ((flags & 4) != 0) {
                stream.writeInt32(offset_id_offset);
            }
            Vector.serialize(stream, messages);
            Vector.serialize(stream, topics);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_messagesNotModified extends messages_Messages {
        public static final int constructor = 0x74535f21;

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
        }
    }

    public static class PaymentForm extends TLObject {
        public int flags;
        public boolean can_save_credentials;
        public boolean password_missing;
        public long form_id;
        public long bot_id;
        public String title;
        public String description;
        public WebDocument photo;
        public TL_invoice invoice;
        public long provider_id;
        public String url;
        public String native_provider;
        public TL_dataJSON native_params;
        public ArrayList<TL_paymentFormMethod> additional_methods = new ArrayList<>();
        public TL_paymentRequestedInfo saved_info;
        public ArrayList<TL_paymentSavedCredentialsCard> saved_credentials = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static PaymentForm TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PaymentForm result = null;
            switch (constructor) {
                case TL_payments_paymentForm.constructor:
                    result = new TL_payments_paymentForm();
                    break;
                case TL_payments_paymentFormStars.constructor:
                    result = new TL_payments_paymentFormStars();
                    break;
                case TL_payments_paymentFormStarGift.constructor:
                    result = new TL_payments_paymentFormStarGift();
                    break;
            }
            return TLdeserialize(PaymentForm.class, result, stream, constructor, exception);
        }
    }

    public static class TL_payments_paymentForm extends PaymentForm {
        public static final int constructor = 0xa0058751;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_save_credentials = (flags & 4) != 0;
            password_missing = (flags & 8) != 0;
            form_id = stream.readInt64(exception);
            bot_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 32) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            provider_id = stream.readInt64(exception);
            url = stream.readString(exception);
            if ((flags & 16) != 0) {
                native_provider = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                native_params = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                additional_methods = Vector.deserialize(stream, TL_paymentFormMethod::TLdeserialize, exception);
            }
            if ((flags & 1) != 0) {
                saved_info = TL_paymentRequestedInfo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                saved_credentials = Vector.deserialize(stream, TL_paymentSavedCredentialsCard::TLdeserialize, exception);
            }
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_save_credentials ? (flags | 4) : (flags &~ 4);
            flags = password_missing ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt64(form_id);
            stream.writeInt64(bot_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            stream.writeInt64(provider_id);
            stream.writeString(url);
            if ((flags & 16) != 0) {
                stream.writeString(native_provider);
            }
            if ((flags & 16) != 0) {
                native_params.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                Vector.serialize(stream, additional_methods);
            }
            if ((flags & 1) != 0) {
                saved_info.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                Vector.serialize(stream, saved_credentials);
            }
            Vector.serialize(stream, users);
        }
    }

    public static class TL_payments_paymentFormStars extends PaymentForm {
        public static final int constructor = 0x7bf6b15c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            form_id = stream.readInt64(exception);
            bot_id = stream.readInt64(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 32) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_save_credentials ? (flags | 4) : (flags &~ 4);
            flags = password_missing ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt64(form_id);
            stream.writeInt64(bot_id);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_payments_paymentFormStarGift extends PaymentForm {
        public static final int constructor = 0xb425cfe1;

        public void readParams(InputSerializedData stream, boolean exception) {
            form_id = stream.readInt64(exception);
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(form_id);
            invoice.serializeToStream(stream);
        }
    }

    public static class TL_paymentFormMethod extends TLObject {
        public static final int constructor = 0x88f8f21b;

        public String url;
        public String title;

        public static TL_paymentFormMethod TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_paymentFormMethod result = TL_paymentFormMethod.constructor != constructor ? null : new TL_paymentFormMethod();
            return TLdeserialize(TL_paymentFormMethod.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            title = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeString(title);
        }
    }

    public static abstract class ContactLink_layer101 extends TLObject {

        public static ContactLink_layer101 TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            ContactLink_layer101 result = null;
            switch (constructor) {
                case 0xfeedd3ad:
                    result = new TL_contactLinkNone();
                    break;
                case 0xd502c2d0:
                    result = new TL_contactLinkContact();
                    break;
                case 0x5f4f9247:
                    result = new TL_contactLinkUnknown();
                    break;
            }
            return TLdeserialize(ContactLink_layer101.class, result, stream, constructor, exception);
        }
    }

    public static class TL_contactLinkNone extends ContactLink_layer101 {
        public static final int constructor = 0xfeedd3ad;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contactLinkContact extends ContactLink_layer101 {
        public static final int constructor = 0xd502c2d0;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contactLinkUnknown extends ContactLink_layer101 {
        public static final int constructor = 0x5f4f9247;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contacts_link_layer101 extends TLObject {
        public static final int constructor = 0x3ace484c;

        public ContactLink_layer101 my_link;
        public ContactLink_layer101 foreign_link;
        public User user;

        public static TL_contacts_link_layer101 TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_contacts_link_layer101 result = TL_contacts_link_layer101.constructor != constructor ? null : new TL_contacts_link_layer101();
            return TLdeserialize(TL_contacts_link_layer101.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            my_link = ContactLink_layer101.TLdeserialize(stream, stream.readInt32(exception), exception);
            foreign_link = ContactLink_layer101.TLdeserialize(stream, stream.readInt32(exception), exception);
            user = User.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            my_link.serializeToStream(stream);
            foreign_link.serializeToStream(stream);
            user.serializeToStream(stream);
        }
    }

    public static abstract class EncryptedFile extends TLObject {
        public long id;
        public long access_hash;
        public long size;
        public int dc_id;
        public int key_fingerprint;

        public static EncryptedFile TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            EncryptedFile result = null;
            switch (constructor) {
                case 0xa8008cd8:
                    result = new TL_encryptedFile();
                    break;
                case 0x4a70994c:
                    result = new TL_encryptedFile_layer142();
                    break;
                case 0xc21f497e:
                    result = new TL_encryptedFileEmpty();
                    break;
            }
            return TLdeserialize(EncryptedFile.class, result, stream, constructor, exception);
        }
    }

    public static class TL_encryptedFile extends EncryptedFile {
        public static final int constructor = 0xa8008cd8;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt64(exception);
            dc_id = stream.readInt32(exception);
            key_fingerprint = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt64(size);
            stream.writeInt32(dc_id);
            stream.writeInt32(key_fingerprint);
        }
    }

    public static class TL_encryptedFile_layer142 extends EncryptedFile {
        public static final int constructor = 0x4a70994c;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            key_fingerprint = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) size);
            stream.writeInt32(dc_id);
            stream.writeInt32(key_fingerprint);
        }
    }

    public static class TL_encryptedFileEmpty extends EncryptedFile {
        public static final int constructor = 0xc21f497e;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class Peer extends TLObject {

        public long user_id;
        public long chat_id;
        public long channel_id;

        public static Peer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            Peer result = null;
            switch (constructor) {
                case 0xa2a5371e:
                    result = new TL_peerChannel();
                    break;
                case 0xbddde532:
                    result = new TL_peerChannel_layer131();
                    break;
                case 0x59511722:
                    result = new TL_peerUser();
                    break;
                case 0x9db1bc6d:
                    result = new TL_peerUser_layer131();
                    break;
                case 0x36c6019a:
                    result = new TL_peerChat();
                    break;
                case 0xbad0e5bb:
                    result = new TL_peerChat_layer131();
                    break;
            }
            return TLdeserialize(Peer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_peerChannel_layer131 extends TL_peerChannel {
        public static final int constructor = 0xbddde532;

        public void readParams(InputSerializedData stream, boolean exception) {
            channel_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) channel_id);
        }
    }

    public static class TL_peerUser extends Peer {
        public static final int constructor = 0x59511722;

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_peerChannel extends Peer {
        public static final int constructor = 0xa2a5371e;

        public void readParams(InputSerializedData stream, boolean exception) {
            channel_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(channel_id);
        }
    }

    public static class TL_peerChat extends Peer {
        public static final int constructor = 0x36c6019a;

        public void readParams(InputSerializedData stream, boolean exception) {
            chat_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(chat_id);
        }
    }

    public static class TL_peerUser_layer131 extends TL_peerUser {
        public static final int constructor = 0x9db1bc6d;

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_peerChat_layer131 extends TL_peerChat {
        public static final int constructor = 0xbad0e5bb;

        public void readParams(InputSerializedData stream, boolean exception) {
            chat_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) chat_id);
        }
    }

    public static class TL_labeledPrice extends TLObject {
        public static final int constructor = 0xcb296bf8;

        public String label;
        public long amount;

        public static TL_labeledPrice TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_labeledPrice result = TL_labeledPrice.constructor != constructor ? null : new TL_labeledPrice();
            return TLdeserialize(TL_labeledPrice.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            label = stream.readString(exception);
            amount = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(label);
            stream.writeInt64(amount);
        }
    }

    public static class TL_messages_exportedChatInvites extends TLObject {
        public static final int constructor = 0xbdc62dcc;

        public int count;
        public ArrayList<ExportedChatInvite> invites = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_exportedChatInvites TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_exportedChatInvites result = TL_messages_exportedChatInvites.constructor != constructor ? null : new TL_messages_exportedChatInvites();
            return TLdeserialize(TL_messages_exportedChatInvites.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            invites = Vector.deserialize(stream, ExportedChatInvite::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            Vector.serialize(stream, invites);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_inputStickerSetItem extends TLObject {
        public static final int constructor = 0xffa0a496;

        public int flags;
        public InputDocument document;
        public String emoji;
        public TL_maskCoords mask_coords;

        public static TL_inputStickerSetItem TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_inputStickerSetItem result = TL_inputStickerSetItem.constructor != constructor ? null : new TL_inputStickerSetItem();
            return TLdeserialize(TL_inputStickerSetItem.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            document = InputDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            emoji = stream.readString(exception);
            if ((flags & 1) != 0) {
                mask_coords = TL_maskCoords.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            document.serializeToStream(stream);
            stream.writeString(emoji);
            if ((flags & 1) != 0) {
                mask_coords.serializeToStream(stream);
            }
        }
    }

    public static class TL_langPackDifference extends TLObject {
        public static final int constructor = 0xf385c1f6;

        public String lang_code;
        public int from_version;
        public int version;
        public ArrayList<LangPackString> strings = new ArrayList<>();

        public static TL_langPackDifference TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_langPackDifference result = TL_langPackDifference.constructor != constructor ? null : new TL_langPackDifference();
            return TLdeserialize(TL_langPackDifference.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            lang_code = stream.readString(exception);
            from_version = stream.readInt32(exception);
            version = stream.readInt32(exception);
            strings = Vector.deserialize(stream, LangPackString::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(lang_code);
            stream.writeInt32(from_version);
            stream.writeInt32(version);
            Vector.serialize(stream, strings);
        }
    }

    public static abstract class help_DeepLinkInfo extends TLObject {

        public static help_DeepLinkInfo TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_DeepLinkInfo result = null;
            switch (constructor) {
                case 0x66afa166:
                    result = new TL_help_deepLinkInfoEmpty();
                    break;
                case 0x6a4ee832:
                    result = new TL_help_deepLinkInfo();
                    break;
            }
            return TLdeserialize(help_DeepLinkInfo.class, result, stream, constructor, exception);
        }
    }

    public static class TL_help_deepLinkInfoEmpty extends help_DeepLinkInfo {
        public static final int constructor = 0x66afa166;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_deepLinkInfo extends help_DeepLinkInfo {
        public static final int constructor = 0x6a4ee832;

        public int flags;
        public boolean update_app;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            update_app = (flags & 1) != 0;
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = update_app ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, entities);
            }
        }
    }

    public static class TL_chatAdminRights extends TLObject {
        public static final int constructor = 0x5fb224d5;

        public int flags;
        public boolean change_info;
        public boolean post_messages;
        public boolean edit_messages;
        public boolean delete_messages;
        public boolean ban_users;
        public boolean invite_users;
        public boolean pin_messages;
        public boolean add_admins;
        public boolean anonymous;
        public boolean manage_call;
        public boolean other;
        public boolean manage_topics;
        public boolean post_stories;
        public boolean edit_stories;
        public boolean delete_stories;
        public boolean manage_direct_messages;

        public static TL_chatAdminRights TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_chatAdminRights result = TL_chatAdminRights.constructor != constructor ? null : new TL_chatAdminRights();
            return TLdeserialize(TL_chatAdminRights.class, result, stream, constructor, exception);
        }

        public static TL_chatAdminRights clone(TL_chatAdminRights rights) {
            if (rights == null) return null;
            SerializedData data = new SerializedData(rights.getObjectSize());
            rights.serializeToStream(data);
            data = new SerializedData(data.toByteArray());
            return TLdeserialize(data, data.readInt32(false), false);
        }

        @Override
        public boolean equals(@Nullable Object obj) {
            if (!(obj instanceof TL_chatAdminRights)) return false;
            return this.flags == ((TL_chatAdminRights) obj).flags;
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            change_info = (flags & 1) != 0;
            post_messages = (flags & 2) != 0;
            edit_messages = (flags & 4) != 0;
            delete_messages = (flags & 8) != 0;
            ban_users = (flags & 16) != 0;
            invite_users = (flags & 32) != 0;
            pin_messages = (flags & 128) != 0;
            add_admins = (flags & 512) != 0;
            anonymous = (flags & 1024) != 0;
            manage_call = (flags & 2048) != 0;
            other = (flags & 4096) != 0;
            manage_topics = (flags & 8192) != 0;
            post_stories = (flags & 16384) != 0;
            edit_stories = (flags & 32768) != 0;
            delete_stories = (flags & 65536) != 0;
            manage_direct_messages = hasFlag(flags, FLAG_17);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = change_info ? (flags | 1) : (flags &~ 1);
            flags = post_messages ? (flags | 2) : (flags &~ 2);
            flags = edit_messages ? (flags | 4) : (flags &~ 4);
            flags = delete_messages ? (flags | 8) : (flags &~ 8);
            flags = ban_users ? (flags | 16) : (flags &~ 16);
            flags = invite_users ? (flags | 32) : (flags &~ 32);
            flags = pin_messages ? (flags | 128) : (flags &~ 128);
            flags = add_admins ? (flags | 512) : (flags &~ 512);
            flags = anonymous ? (flags | 1024) : (flags &~ 1024);
            flags = manage_call ? (flags | 2048) : (flags &~ 2048);
            flags = other ? (flags | 4096) : (flags &~ 4096);
            flags = manage_topics ? (flags | 8192) : (flags &~ 8192);
            flags = post_stories ? (flags | 16384) : (flags &~ 16384);
            flags = edit_stories ? (flags | 32768) : (flags &~ 32768);
            flags = delete_stories ? (flags | 65536) : (flags &~ 65536);
            flags = setFlag(flags, FLAG_17, manage_direct_messages);
            stream.writeInt32(flags);
        }
    }

    public static abstract class PollResults extends TLObject {

        public int flags;
        public boolean min;
        public ArrayList<TL_pollAnswerVoters> results = new ArrayList<>();
        public int total_voters;
        public ArrayList<Peer> recent_voters = new ArrayList<>();
        public String solution;
        public ArrayList<MessageEntity> solution_entities = new ArrayList<>();

        public static PollResults TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PollResults result = null;
            switch (constructor) {
                case 0x5755785a:
                    result = new TL_pollResults_layer108();
                    break;
                case 0xc87024a2:
                    result = new TL_pollResults_layer111();
                    break;
                case 0xbadcc1a3:
                    result = new TL_pollResults_layer131();
                    break;
                case 0xdcb82ea3:
                    result = new TL_pollResults_layer158();
                    break;
                case 0x7adf2420:
                    result = new TL_pollResults();
                    break;
            }
            return TLdeserialize(PollResults.class, result, stream, constructor, exception);
        }
    }

    public static class TL_pollResults_layer108 extends TL_pollResults {
        public static final int constructor = 0x5755785a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                results = Vector.deserialize(stream, TL_pollAnswerVoters::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, results);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
        }
    }

    public static class TL_pollResults_layer111 extends PollResults {
        public static final int constructor = 0xc87024a2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                results = Vector.deserialize(stream, TL_pollAnswerVoters::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_peerUser user = new TL_peerUser();
                    user.user_id = (long) stream.readInt32(exception);
                    recent_voters.add(user);
                }
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, results);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32((int) (long) recent_voters.get(a).user_id);
                }
            }
        }
    }

    public static class TL_pollResults_layer131 extends TL_pollResults {
        public static final int constructor = 0xbadcc1a3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                results = Vector.deserialize(stream, TL_pollAnswerVoters::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_peerUser user = new TL_peerUser();
                    user.user_id = (long) stream.readInt32(exception);
                    recent_voters.add(user);
                }
            }
            if ((flags & 16) != 0) {
                solution = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                solution_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, results);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt32((int) (long) recent_voters.get(a).user_id);
                }
            }
            if ((flags & 16) != 0) {
                stream.writeString(solution);
            }
            if ((flags & 16) != 0) {
                Vector.serialize(stream, solution_entities);
            }
        }
    }

    public static class TL_pollResults extends PollResults {
        public static final int constructor = 0x7adf2420;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                results = Vector.deserialize(stream, TL_pollAnswerVoters::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                recent_voters = Vector.deserialize(stream, Peer::TLdeserialize, exception);
            }
            if ((flags & 16) != 0) {
                solution = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                solution_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, results);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, recent_voters);
            }
            if ((flags & 16) != 0) {
                stream.writeString(solution);
            }
            if ((flags & 16) != 0) {
                Vector.serialize(stream, solution_entities);
            }
        }
    }

    public static class TL_pollResults_layer158 extends PollResults {
        public static final int constructor = 0xdcb82ea3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            min = (flags & 1) != 0;
            if ((flags & 2) != 0) {
                results = Vector.deserialize(stream, TL_pollAnswerVoters::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                total_voters = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    TL_peerUser user = new TL_peerUser();
                    user.user_id = stream.readInt64(exception);
                    recent_voters.add(user);
                }
            }
            if ((flags & 16) != 0) {
                solution = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                solution_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = min ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, results);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(total_voters);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = recent_voters.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeInt64(recent_voters.get(a).user_id);
                }
            }
            if ((flags & 16) != 0) {
                stream.writeString(solution);
            }
            if ((flags & 16) != 0) {
                Vector.serialize(stream, solution_entities);
            }
        }
    }

    public static abstract class SecureFile extends TLObject {

        public static SecureFile TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            SecureFile result = null;
            switch (constructor) {
                case 0x64199744:
                    result = new TL_secureFileEmpty();
                    break;
                case 0x7d09c27e:
                    result = new TL_secureFile();
                    break;
            }
            return TLdeserialize(SecureFile.class, result, stream, constructor, exception);
        }
    }

    public static class TL_secureFileEmpty extends SecureFile {
        public static final int constructor = 0x64199744;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_secureFile extends SecureFile {
        public static final int constructor = 0x7d09c27e;

        public long id;
        public long access_hash;
        public long size;
        public int dc_id;
        public int date;
        public byte[] file_hash;
        public byte[] secret;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt64(exception);
            dc_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            file_hash = stream.readByteArray(exception);
            secret = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt64(size);
            stream.writeInt32(dc_id);
            stream.writeInt32(date);
            stream.writeByteArray(file_hash);
            stream.writeByteArray(secret);
        }
    }

    public static class TL_secureFile_layer142 extends TL_secureFile {
        public static final int constructor = 0xe0277a62;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            file_hash = stream.readByteArray(exception);
            secret = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) size);
            stream.writeInt32(dc_id);
            stream.writeInt32(date);
            stream.writeByteArray(file_hash);
            stream.writeByteArray(secret);
        }
    }

    public static class TL_messages_affectedMessages extends TLObject {
        public static final int constructor = 0x84d19185;

        public int pts;
        public int pts_count;

        public static TL_messages_affectedMessages TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_affectedMessages result = TL_messages_affectedMessages.constructor != constructor ? null : new TL_messages_affectedMessages();
            return TLdeserialize(TL_messages_affectedMessages.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
            pts_count = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
            stream.writeInt32(pts_count);
        }
    }

    public static class TL_messages_chatInviteImporters extends TLObject {
        public static final int constructor = 0x81b6b00a;

        public int count;
        public ArrayList<TL_chatInviteImporter> importers = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatInviteImporters TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_chatInviteImporters result = TL_messages_chatInviteImporters.constructor != constructor ? null : new TL_messages_chatInviteImporters();
            return TLdeserialize(TL_messages_chatInviteImporters.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            importers = Vector.deserialize(stream, TL_chatInviteImporter::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            Vector.serialize(stream, importers);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_pollAnswerVoters extends TLObject {
        public static final int constructor = 0x3b6ddad2;

        public int flags;
        public boolean chosen;
        public boolean correct;
        public byte[] option;
        public int voters;

        public static TL_pollAnswerVoters TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_pollAnswerVoters result = TL_pollAnswerVoters.constructor != constructor ? null : new TL_pollAnswerVoters();
            return TLdeserialize(TL_pollAnswerVoters.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            chosen = (flags & 1) != 0;
            correct = (flags & 2) != 0;
            option = stream.readByteArray(exception);
            voters = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = chosen ? (flags | 1) : (flags &~ 1);
            flags = correct ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeByteArray(option);
            stream.writeInt32(voters);
        }
    }

    public static class TL_channels_channelParticipant extends TLObject {
        public static final int constructor = 0xdfb80317;

        public ChannelParticipant participant;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_channels_channelParticipant TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_channels_channelParticipant result = TL_channels_channelParticipant.constructor != constructor ? null : new TL_channels_channelParticipant();
            return TLdeserialize(TL_channels_channelParticipant.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            participant = ChannelParticipant.TLdeserialize(stream, stream.readInt32(exception), exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            participant.serializeToStream(stream);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_authorization extends TLObject {
        public static final int constructor = 0xad01d61d;

        public int flags;
        public boolean current;
        public boolean official_app;
        public boolean password_pending;
        public boolean encrypted_requests_disabled;
        public boolean call_requests_disabled;
        public boolean unconfirmed;
        public long hash;
        public String device_model;
        public String platform;
        public String system_version;
        public int api_id;
        public String app_name;
        public String app_version;
        public int date_created;
        public int date_active;
        public String ip;
        public String country;
        public String region;

        public static TL_authorization TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_authorization result = TL_authorization.constructor != constructor ? null : new TL_authorization();
            return TLdeserialize(TL_authorization.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            current = (flags & 1) != 0;
            official_app = (flags & 2) != 0;
            password_pending = (flags & 4) != 0;
            encrypted_requests_disabled = (flags & 8) != 0;
            call_requests_disabled = (flags & 16) != 0;
            unconfirmed = (flags & 32) != 0;
            hash = stream.readInt64(exception);
            device_model = stream.readString(exception);
            platform = stream.readString(exception);
            system_version = stream.readString(exception);
            api_id = stream.readInt32(exception);
            app_name = stream.readString(exception);
            app_version = stream.readString(exception);
            date_created = stream.readInt32(exception);
            date_active = stream.readInt32(exception);
            ip = stream.readString(exception);
            country = stream.readString(exception);
            region = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = current ? (flags | 1) : (flags &~ 1);
            flags = official_app ? (flags | 2) : (flags &~ 2);
            flags = password_pending ? (flags | 4) : (flags &~ 4);
            flags = encrypted_requests_disabled ? (flags | 8) : (flags &~ 8);
            flags = call_requests_disabled ? (flags | 16) : (flags &~ 16);
            flags = unconfirmed ? (flags | 32) : (flags &~ 32);
            stream.writeInt32(flags);
            stream.writeInt64(hash);
            stream.writeString(device_model);
            stream.writeString(platform);
            stream.writeString(system_version);
            stream.writeInt32(api_id);
            stream.writeString(app_name);
            stream.writeString(app_version);
            stream.writeInt32(date_created);
            stream.writeInt32(date_active);
            stream.writeString(ip);
            stream.writeString(country);
            stream.writeString(region);
        }
    }

    public static abstract class updates_Difference extends TLObject {
        public ArrayList<Message> new_messages = new ArrayList<>();
        public ArrayList<EncryptedMessage> new_encrypted_messages = new ArrayList<>();
        public ArrayList<Update> other_updates = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public TL_updates_state state;
        public TL_updates_state intermediate_state;
        public int pts;
        public int date;
        public int seq;

        public static updates_Difference TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            updates_Difference result = null;
            switch (constructor) {
                case 0xf49ca0:
                    result = new TL_updates_difference();
                    break;
                case 0xa8fb1981:
                    result = new TL_updates_differenceSlice();
                    break;
                case 0x4afe8f6d:
                    result = new TL_updates_differenceTooLong();
                    break;
                case 0x5d75a138:
                    result = new TL_updates_differenceEmpty();
                    break;
            }
            return TLdeserialize(updates_Difference.class, result, stream, constructor, exception);
        }
    }

    public static class TL_updates_difference extends updates_Difference {
        public static final int constructor = 0xf49ca0;

        public void readParams(InputSerializedData stream, boolean exception) {
            new_messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            new_encrypted_messages = Vector.deserialize(stream, EncryptedMessage::TLdeserialize, exception);
            other_updates = Vector.deserialize(stream, Update::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
            state = TL_updates_state.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, new_messages);
            Vector.serialize(stream, new_encrypted_messages);
            Vector.serialize(stream, other_updates);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
            state.serializeToStream(stream);
        }
    }

    public static class TL_updates_differenceSlice extends updates_Difference {
        public static final int constructor = 0xa8fb1981;

        public void readParams(InputSerializedData stream, boolean exception) {
            new_messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            new_encrypted_messages = Vector.deserialize(stream, EncryptedMessage::TLdeserialize, exception);
            other_updates = Vector.deserialize(stream, Update::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
            intermediate_state = TL_updates_state.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, new_messages);
            Vector.serialize(stream, new_encrypted_messages);
            Vector.serialize(stream, other_updates);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
            intermediate_state.serializeToStream(stream);
        }
    }

    public static class TL_updates_differenceTooLong extends updates_Difference {
        public static final int constructor = 0x4afe8f6d;

        public void readParams(InputSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
        }
    }

    public static class TL_updates_differenceEmpty extends updates_Difference {
        public static final int constructor = 0x5d75a138;

        public void readParams(InputSerializedData stream, boolean exception) {
            date = stream.readInt32(exception);
            seq = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(date);
            stream.writeInt32(seq);
        }
    }

    public static abstract class PrivacyKey extends TLObject {

        public static PrivacyKey TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PrivacyKey result = null;
            switch (constructor) {
                case TL_privacyKeyStatusTimestamp.constructor:
                    result = new TL_privacyKeyStatusTimestamp();
                    break;
                case TL_privacyKeyPhoneP2P.constructor:
                    result = new TL_privacyKeyPhoneP2P();
                    break;
                case TL_privacyKeyChatInvite.constructor:
                    result = new TL_privacyKeyChatInvite();
                    break;
                case TL_privacyKeyAddedByPhone.constructor:
                    result = new TL_privacyKeyAddedByPhone();
                    break;
                case TL_privacyKeyVoiceMessages.constructor:
                    result = new TL_privacyKeyVoiceMessages();
                    break;
                case TL_privacyKeyAbout.constructor:
                    result = new TL_privacyKeyAbout();
                    break;
                case TL_privacyKeyPhoneCall.constructor:
                    result = new TL_privacyKeyPhoneCall();
                    break;
                case TL_privacyKeyForwards.constructor:
                    result = new TL_privacyKeyForwards();
                    break;
                case TL_privacyKeyPhoneNumber.constructor:
                    result = new TL_privacyKeyPhoneNumber();
                    break;
                case TL_privacyKeyProfilePhoto.constructor:
                    result = new TL_privacyKeyProfilePhoto();
                    break;
                case TL_privacyKeyBirthday.constructor:
                    result = new TL_privacyKeyBirthday();
                    break;
                case TL_privacyKeyStarGiftsAutoSave.constructor:
                    result = new TL_privacyKeyStarGiftsAutoSave();
                    break;
                case TL_privacyKeyNoPaidMessages.constructor:
                    result = new TL_privacyKeyNoPaidMessages();
                    break;
                case TL_privacyKeySavedMusic.constructor:
                    result = new TL_privacyKeySavedMusic();
                    break;
            }
            return TLdeserialize(PrivacyKey.class, result, stream, constructor, exception);
        }
    }

    public static class TL_privacyKeyStatusTimestamp extends PrivacyKey {
        public static final int constructor = 0xbc2eab30;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneP2P extends PrivacyKey {
        public static final int constructor = 0x39491cc8;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyChatInvite extends PrivacyKey {
        public static final int constructor = 0x500e6dfa;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneCall extends PrivacyKey {
        public static final int constructor = 0x3d662b7b;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyAddedByPhone extends PrivacyKey {
        public static final int constructor = 0x42ffd42b;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyAbout extends PrivacyKey {
        public static final int constructor = 0xa486b761;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyVoiceMessages extends PrivacyKey {
        public static final int constructor = 0x697f414;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyForwards extends PrivacyKey {
        public static final int constructor = 0x69ec56a3;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyPhoneNumber extends PrivacyKey {
        public static final int constructor = 0xd19ae46d;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyProfilePhoto extends PrivacyKey {
        public static final int constructor = 0x96151fed;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyBirthday extends PrivacyKey {
        public static final int constructor = 0x2000a518;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyStarGiftsAutoSave extends PrivacyKey {
        public static final int constructor = 0x2ca4fdf8;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeyNoPaidMessages extends PrivacyKey {
        public static final int constructor = 0x17d348d2;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyKeySavedMusic extends PrivacyKey {
        public static final int constructor = 0xff7a571b;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class GeoPoint extends TLObject {
        public int flags;
        public double _long;
        public double lat;
        public int accuracy_radius;
        public long access_hash;

        public static GeoPoint TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            GeoPoint result = null;
            switch (constructor) {
                case 0x296f104:
                    result = new TL_geoPoint_layer119();
                    break;
                case 0x2049d70c:
                    result = new TL_geoPoint_layer81();
                    break;
                case 0x1117dd5f:
                    result = new TL_geoPointEmpty();
                    break;
                case 0xb2a2f663:
                    result = new TL_geoPoint();
                    break;
            }
            return TLdeserialize(GeoPoint.class, result, stream, constructor, exception);
        }
    }

    public static class TL_geoPoint_layer119 extends TL_geoPoint {
        public static final int constructor = 0x296f104;

        public void readParams(InputSerializedData stream, boolean exception) {
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_geoPoint_layer81 extends TL_geoPoint {
        public static final int constructor = 0x2049d70c;

        public void readParams(InputSerializedData stream, boolean exception) {
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
        }
    }

    public static class TL_geoPointEmpty extends GeoPoint {
        public static final int constructor = 0x1117dd5f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_geoPoint extends GeoPoint {
        public static final int constructor = 0xb2a2f663;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            _long = stream.readDouble(exception);
            lat = stream.readDouble(exception);
            access_hash = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                accuracy_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeDouble(_long);
            stream.writeDouble(lat);
            stream.writeInt64(access_hash);
            if ((flags & 1) != 0) {
                stream.writeInt32(accuracy_radius);
            }
        }
    }

    public static abstract class ChatInvite extends TLObject {

        public int flags;
        public boolean channel;
        public boolean broadcast;
        public boolean isPublic;
        public boolean megagroup;
        public boolean request_needed;
        public String title;
        public String about;
        public Photo photo;
        public int participants_count;
        public ArrayList<User> participants = new ArrayList<>();
        public Chat chat;
        public int expires;
        public boolean verified;
        public boolean scam;
        public boolean fake;
        public boolean can_refulfill_subscription;
        public int color;
        public TL_stars.TL_starsSubscriptionPricing subscription_pricing;
        public long subscription_form_id;
        public TL_bots.botVerification bot_verification;

        public static ChatInvite TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            ChatInvite result = null;
            switch (constructor) {
                case TL_chatInvite.constructor:
                    result = new TL_chatInvite();
                    break;
                case TL_chatInvite_layer195.constructor:
                    result = new TL_chatInvite_layer195();
                    break;
                case TL_chatInvite_layer185.constructor:
                    result = new TL_chatInvite_layer185();
                    break;
                case TL_chatInvite_layer165.constructor:
                    result = new TL_chatInvite_layer165();
                    break;
                case 0x61695cb0:
                    result = new TL_chatInvitePeek();
                    break;
                case 0x5a686d7c:
                    result = new TL_chatInviteAlready();
                    break;
            }
            return TLdeserialize(ChatInvite.class, result, stream, constructor, exception);
        }
    }

    public static class TL_chatInvite extends ChatInvite {
        public static final int constructor = 0x5c9d3702;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            channel = (flags & 1) != 0;
            broadcast = (flags & 2) != 0;
            isPublic = (flags & 4) != 0;
            megagroup = (flags & 8) != 0;
            request_needed = (flags & 64) != 0;
            verified = (flags & 128) != 0;
            scam = (flags & 256) != 0;
            fake = (flags & 512) != 0;
            can_refulfill_subscription = (flags & 2048) != 0;
            title = stream.readString(exception);
            boolean hasAbout = (flags & 32) != 0;
            if (hasAbout) {
                about = stream.readString(exception);
            }
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 16) != 0) {
                participants = Vector.deserialize(stream, User::TLdeserialize, exception);
            }
            color = stream.readInt32(exception);
            if ((flags & 1024) != 0) {
                subscription_pricing = TL_stars.TL_starsSubscriptionPricing.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4096) != 0) {
                subscription_form_id = stream.readInt64(exception);
            }
            if ((flags & 8192) != 0) {
                bot_verification = TL_bots.botVerification.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = channel ? (flags | 1) : (flags &~ 1);
            flags = broadcast ? (flags | 2) : (flags &~ 2);
            flags = isPublic ? (flags | 4) : (flags &~ 4);
            flags = megagroup ? (flags | 8) : (flags &~ 8);
            flags = about != null ? (flags | 32) : (flags &~ 32);
            flags = request_needed ? (flags | 64) : (flags &~ 64);
            flags = verified ? (flags | 128) : (flags &~ 128);
            flags = scam ? (flags | 256) : (flags &~ 256);
            flags = fake ? (flags | 512) : (flags &~ 512);
            flags = can_refulfill_subscription ? (flags | 2048) : (flags &~ 2048);
            stream.writeInt32(flags);
            stream.writeString(title);
            if (about != null) {
                stream.writeString(about);
            }
            photo.serializeToStream(stream);
            stream.writeInt32(participants_count);
            if ((flags & 16) != 0) {
                Vector.serialize(stream, participants);
            }
            stream.writeInt32(color);
            if ((flags & 1024) != 0) {
                subscription_pricing.serializeToStream(stream);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt64(subscription_form_id);
            }
            if ((flags & 8192) != 0) {
                bot_verification.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatInvite_layer195 extends TL_chatInvite {
        public static final int constructor = 0xfe65389d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            channel = (flags & 1) != 0;
            broadcast = (flags & 2) != 0;
            isPublic = (flags & 4) != 0;
            megagroup = (flags & 8) != 0;
            request_needed = (flags & 64) != 0;
            verified = (flags & 128) != 0;
            scam = (flags & 256) != 0;
            fake = (flags & 512) != 0;
            can_refulfill_subscription = (flags & 2048) != 0;
            title = stream.readString(exception);
            boolean hasAbout = (flags & 32) != 0;
            if (hasAbout) {
                about = stream.readString(exception);
            }
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 16) != 0) {
                participants = Vector.deserialize(stream, User::TLdeserialize, exception);
            }
            color = stream.readInt32(exception);
            if ((flags & 1024) != 0) {
                subscription_pricing = TL_stars.TL_starsSubscriptionPricing.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4096) != 0) {
                subscription_form_id = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = channel ? (flags | 1) : (flags &~ 1);
            flags = broadcast ? (flags | 2) : (flags &~ 2);
            flags = isPublic ? (flags | 4) : (flags &~ 4);
            flags = megagroup ? (flags | 8) : (flags &~ 8);
            flags = about != null ? (flags | 32) : (flags &~ 32);
            flags = request_needed ? (flags | 64) : (flags &~ 64);
            flags = verified ? (flags | 128) : (flags &~ 128);
            flags = scam ? (flags | 256) : (flags &~ 256);
            flags = fake ? (flags | 512) : (flags &~ 512);
            flags = can_refulfill_subscription ? (flags | 2048) : (flags &~ 2048);
            stream.writeInt32(flags);
            stream.writeString(title);
            if (about != null) {
                stream.writeString(about);
            }
            photo.serializeToStream(stream);
            stream.writeInt32(participants_count);
            if ((flags & 16) != 0) {
                Vector.serialize(stream, participants);
            }
            stream.writeInt32(color);
            if ((flags & 1024) != 0) {
                subscription_pricing.serializeToStream(stream);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt64(subscription_form_id);
            }
        }
    }

    public static class TL_chatInvite_layer185 extends TL_chatInvite {
        public static final int constructor = 0xcde0ec40;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            channel = (flags & 1) != 0;
            broadcast = (flags & 2) != 0;
            isPublic = (flags & 4) != 0;
            megagroup = (flags & 8) != 0;
            request_needed = (flags & 64) != 0;
            verified = (flags & 128) != 0;
            scam = (flags & 256) != 0;
            fake = (flags & 512) != 0;
            title = stream.readString(exception);
            boolean hasAbout = (flags & 32) != 0;
            if (hasAbout) {
                about = stream.readString(exception);
            }
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 16) != 0) {
                participants = Vector.deserialize(stream, User::TLdeserialize, exception);
            }
            color = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = channel ? (flags | 1) : (flags &~ 1);
            flags = broadcast ? (flags | 2) : (flags &~ 2);
            flags = isPublic ? (flags | 4) : (flags &~ 4);
            flags = megagroup ? (flags | 8) : (flags &~ 8);
            flags = about != null ? (flags | 32) : (flags &~ 32);
            flags = request_needed ? (flags | 64) : (flags &~ 64);
            flags = verified ? (flags | 128) : (flags &~ 128);
            flags = scam ? (flags | 256) : (flags &~ 256);
            flags = fake ? (flags | 512) : (flags &~ 512);
            stream.writeInt32(flags);
            stream.writeString(title);
            if (about != null) {
                stream.writeString(about);
            }
            photo.serializeToStream(stream);
            stream.writeInt32(participants_count);
            if ((flags & 16) != 0) {
                Vector.serialize(stream, participants);
            }
            stream.writeInt32(color);
        }
    }

    public static class TL_chatInvite_layer165 extends ChatInvite {
        public static final int constructor = 0x300c44c1;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            channel = (flags & 1) != 0;
            broadcast = (flags & 2) != 0;
            isPublic = (flags & 4) != 0;
            megagroup = (flags & 8) != 0;
            request_needed = (flags & 64) != 0;
            verified = (flags & 128) != 0;
            scam = (flags & 256) != 0;
            fake = (flags & 512) != 0;
            title = stream.readString(exception);
            boolean hasAbout = (flags & 32) != 0;
            if (hasAbout) {
                about = stream.readString(exception);
            }
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            participants_count = stream.readInt32(exception);
            if ((flags & 16) != 0) {
                participants = Vector.deserialize(stream, User::TLdeserialize, exception);
            }
            if ((flags & 1024) != 0) {
                color = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = channel ? (flags | 1) : (flags &~ 1);
            flags = broadcast ? (flags | 2) : (flags &~ 2);
            flags = isPublic ? (flags | 4) : (flags &~ 4);
            flags = megagroup ? (flags | 8) : (flags &~ 8);
            flags = about != null ? (flags | 32) : (flags &~ 32);
            flags = request_needed ? (flags | 64) : (flags &~ 64);
            flags = verified ? (flags | 128) : (flags &~ 128);
            flags = scam ? (flags | 256) : (flags &~ 256);
            flags = fake ? (flags | 512) : (flags &~ 512);
            stream.writeInt32(flags);
            stream.writeString(title);
            if (about != null) {
                stream.writeString(about);
            }
            photo.serializeToStream(stream);
            stream.writeInt32(participants_count);
            if ((flags & 16) != 0) {
                Vector.serialize(stream, participants);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(color);
            }
        }
    }

    public static class TL_chatInvitePeek extends ChatInvite {
        public static final int constructor = 0x61695cb0;

        public void readParams(InputSerializedData stream, boolean exception) {
            chat = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            chat.serializeToStream(stream);
            stream.writeInt32(expires);
        }
    }

    public static class TL_chatInviteAlready extends ChatInvite {
        public static final int constructor = 0x5a686d7c;

        public void readParams(InputSerializedData stream, boolean exception) {
            chat = Chat.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            chat.serializeToStream(stream);
        }
    }

    public static class InputGroupCall extends TLObject {

        public long id;
        public long access_hash;
        public String slug;
        public int msg_id;

        public static InputGroupCall TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputGroupCall result = null;
            switch (constructor) {
                case TL_inputGroupCall.constructor:
                    result = new TL_inputGroupCall();
                    break;
                case TL_inputGroupCallSlug.constructor:
                    result = new TL_inputGroupCallSlug();
                    break;
                case TL_inputGroupCallInviteMessage.constructor:
                    result = new TL_inputGroupCallInviteMessage();
                    break;
            }
            return TLdeserialize(InputGroupCall.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputGroupCall extends InputGroupCall {
        public static final int constructor = 0xd8aa840f;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputGroupCallSlug extends InputGroupCall {
        public static final int constructor = 0xfe06823f;

        public void readParams(InputSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static class TL_inputGroupCallInviteMessage extends InputGroupCall {
        public static final int constructor = 0x8c10603f;

        public void readParams(InputSerializedData stream, boolean exception) {
            msg_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(msg_id);
        }
    }

    public static abstract class help_AppUpdate extends TLObject {

        public static help_AppUpdate TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_AppUpdate result = null;
            switch (constructor) {
                case 0xccbbce30:
                    result = new TL_help_appUpdate();
                    break;
                case 0xc45a6536:
                    result = new TL_help_noAppUpdate();
                    break;
            }
            return TLdeserialize(help_AppUpdate.class, result, stream, constructor, exception);
        }
    }

    @Keep
    public static class TL_help_appUpdate extends help_AppUpdate {
        public static final int constructor = 0xccbbce30;

        public int flags;
        public boolean can_not_skip;
        public int id;
        public String version;
        public String text;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public Document document;
        public String url;
        public Document sticker;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_not_skip = (flags & 1) != 0;
            id = stream.readInt32(exception);
            version = stream.readString(exception);
            text = stream.readString(exception);
            entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            if ((flags & 2) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                url = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                sticker = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_not_skip ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(version);
            stream.writeString(text);
            Vector.serialize(stream, entities);
            if ((flags & 2) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeString(url);
            }
            if ((flags & 8) != 0) {
                sticker.serializeToStream(stream);
            }
        }
    }

    public static class TL_help_noAppUpdate extends help_AppUpdate {
        public static final int constructor = 0xc45a6536;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_affectedFoundMessages extends TLObject {
        public static final int constructor = 0xef8d3e6c;

        public int pts;
        public int pts_count;
        public int offset;
        public ArrayList<Integer> messages = new ArrayList<>();

        public static TL_messages_affectedFoundMessages TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_affectedFoundMessages result = TL_messages_affectedFoundMessages.constructor != constructor ? null : new TL_messages_affectedFoundMessages();
            return TLdeserialize(TL_messages_affectedFoundMessages.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            pts = stream.readInt32(exception);
            pts_count = stream.readInt32(exception);
            offset = stream.readInt32(exception);
            messages = Vector.deserializeInt(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(pts);
            stream.writeInt32(pts_count);
            stream.writeInt32(offset);
            Vector.serializeInt(stream, messages);
        }
    }

    public static class TL_channelAdminLogEvent extends TLObject {
        public static final int constructor = 0x1fad68cd;

        public long id;
        public int date;
        public long user_id;
        public ChannelAdminLogEventAction action;

        public static TL_channelAdminLogEvent TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_channelAdminLogEvent result = TL_channelAdminLogEvent.constructor != constructor ? null : new TL_channelAdminLogEvent();
            return TLdeserialize(TL_channelAdminLogEvent.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            date = stream.readInt32(exception);
            user_id = stream.readInt64(exception);
            action = ChannelAdminLogEventAction.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt32(date);
            stream.writeInt64(user_id);
            action.serializeToStream(stream);
        }
    }

    public static abstract class messages_FavedStickers extends TLObject {
        public long hash;
        public ArrayList<TL_stickerPack> packs = new ArrayList<>();
        public ArrayList<Document> stickers = new ArrayList<>();

        public static messages_FavedStickers TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_FavedStickers result = null;
            switch (constructor) {
                case 0x9e8fa6d3:
                    result = new TL_messages_favedStickersNotModified();
                    break;
                case 0x2cb51097:
                    result = new TL_messages_favedStickers();
                    break;
            }
            return TLdeserialize(messages_FavedStickers.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_favedStickersNotModified extends messages_FavedStickers {
        public static final int constructor = 0x9e8fa6d3;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_favedStickers extends messages_FavedStickers {
        public static final int constructor = 0x2cb51097;

        public void readParams(InputSerializedData stream, boolean exception) {
            hash = stream.readInt64(exception);
            packs = Vector.deserialize(stream, TL_stickerPack::TLdeserialize, exception);
            stickers = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(hash);
            Vector.serialize(stream, packs);
            Vector.serialize(stream, stickers);
        }
    }

    public static class TL_langPackLanguage extends TLObject {
        public static final int constructor = 0xeeca5ce3;

        public int flags;
        public boolean official;
        public boolean rtl;
        public String name;
        public String native_name;
        public String lang_code;
        public String base_lang_code;
        public String plural_code;
        public int strings_count;
        public int translated_count;
        public String translations_url;

        public static TL_langPackLanguage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_langPackLanguage result = TL_langPackLanguage.constructor != constructor ? null : new TL_langPackLanguage();
            return TLdeserialize(TL_langPackLanguage.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            official = (flags & 1) != 0;
            rtl = (flags & 4) != 0;
            name = stream.readString(exception);
            native_name = stream.readString(exception);
            lang_code = stream.readString(exception);
            if ((flags & 2) != 0) {
                base_lang_code = stream.readString(exception);
            }
            plural_code = stream.readString(exception);
            strings_count = stream.readInt32(exception);
            translated_count = stream.readInt32(exception);
            translations_url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = official ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeString(name);
            stream.writeString(native_name);
            stream.writeString(lang_code);
            if ((flags & 2) != 0) {
                stream.writeString(base_lang_code);
            }
            stream.writeString(plural_code);
            stream.writeInt32(strings_count);
            stream.writeInt32(translated_count);
            stream.writeString(translations_url);
        }
    }

    public static class TL_chatInviteImporter extends TLObject {
        public static final int constructor = 0x8c5adfd9;

        public int flags;
        public boolean requested;
        public long user_id;
        public int date;
        public String about;
        public long approved_by;
        public boolean via_chatlist;

        public static TL_chatInviteImporter TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_chatInviteImporter result = TL_chatInviteImporter.constructor != constructor ? null : new TL_chatInviteImporter();
            return TLdeserialize(TL_chatInviteImporter.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            requested = (flags & 1) != 0;
            via_chatlist = (flags & 8) != 0;
            user_id = stream.readInt64(exception);
            date = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                about = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                approved_by = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = requested ? (flags | 1) : (flags &~ 1);
            flags = via_chatlist ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt64(user_id);
            stream.writeInt32(date);
            if ((flags & 4) != 0) {
                stream.writeString(about);
            }
            if ((flags & 2) != 0) {
                stream.writeInt64(approved_by);
            }
        }
    }

    public static abstract class SendMessageAction extends TLObject {
        public int progress;

        public static SendMessageAction TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            SendMessageAction result = null;
            switch (constructor) {
                case TL_sendMessageTextDraftAction.constructor:
                    result = new TL_sendMessageTextDraftAction();
                    break;
                case TL_sendMessageGamePlayAction.constructor:
                    result = new TL_sendMessageGamePlayAction();
                    break;
                case TL_sendMessageRecordAudioAction.constructor:
                    result = new TL_sendMessageRecordAudioAction();
                    break;
                case 0x92042ff7:
                    result = new TL_sendMessageUploadVideoAction_old();
                    break;
                case 0xe6ac8a6f:
                    result = new TL_sendMessageUploadAudioAction_old();
                    break;
                case 0xf351d7ab:
                    result = new TL_sendMessageUploadAudioAction();
                    break;
                case 0xd1d34a26:
                    result = new TL_sendMessageUploadPhotoAction();
                    break;
                case 0x8faee98e:
                    result = new TL_sendMessageUploadDocumentAction_old();
                    break;
                case 0xe9763aec:
                    result = new TL_sendMessageUploadVideoAction();
                    break;
                case 0xfd5ec8f5:
                    result = new TL_sendMessageCancelAction();
                    break;
                case 0x176f8ba1:
                    result = new TL_sendMessageGeoLocationAction();
                    break;
                case 0x628cbc6f:
                    result = new TL_sendMessageChooseContactAction();
                    break;
                case 0xb05ac6b1:
                    result = new TL_sendMessageChooseStickerAction();
                    break;
                case 0x88f27fbc:
                    result = new TL_sendMessageRecordRoundAction();
                    break;
                case 0x243e1c66:
                    result = new TL_sendMessageUploadRoundAction();
                    break;
                case TL_sendMessageUploadRoundAction_layer66.constructor:
                    result = new TL_sendMessageUploadRoundAction_layer66();
                    break;
                case 0x16bf744e:
                    result = new TL_sendMessageTypingAction();
                    break;
                case 0xdbda9246:
                    result = new TL_sendMessageHistoryImportAction();
                    break;
                case 0x990a3c1a:
                    result = new TL_sendMessageUploadPhotoAction_old();
                    break;
                case 0xaa0cd9e4:
                    result = new TL_sendMessageUploadDocumentAction();
                    break;
                case 0xd92c2285:
                    result = new TL_speakingInGroupCallAction();
                    break;
                case 0xa187d66f:
                    result = new TL_sendMessageRecordVideoAction();
                    break;
                case 0x25972bcb:
                    result = new TL_sendMessageEmojiInteraction();
                    break;
                case 0xb665902e:
                    result = new TL_sendMessageEmojiInteractionSeen();
                    break;
            }
            return TLdeserialize(SendMessageAction.class, result, stream, constructor, exception);
        }
    }

    public static class TL_sendMessageTextDraftAction extends SendMessageAction {
        public static final int constructor = 0x376d975c;

        public long random_id;
        public TL_textWithEntities text;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(random_id);
            text.serializeToStream(stream);
        }

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {
            random_id = stream.readInt64(exception);
            text = TL_textWithEntities.TLdeserialize(stream, stream.readInt32(exception), exception);
        }
    }

    public static class TL_sendMessageGamePlayAction extends SendMessageAction {
        public static final int constructor = 0xdd6a8f48;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordAudioAction extends SendMessageAction {
        public static final int constructor = 0xd52f73f7;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadVideoAction_old extends TL_sendMessageUploadVideoAction {
        public static final int constructor = 0x92042ff7;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageEmojiInteraction extends SendMessageAction {
        public static final int constructor = 0x25972bcb;

        public String emoticon;
        public int msg_id;
        public TL_dataJSON interaction;

        public void readParams(InputSerializedData stream, boolean exception) {
            emoticon = stream.readString(exception);
            msg_id = stream.readInt32(exception);
            interaction = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(emoticon);
            stream.writeInt32(msg_id);
            interaction.serializeToStream(stream);
        }
    }

    public static class TL_sendMessageUploadAudioAction_old extends TL_sendMessageUploadAudioAction {
        public static final int constructor = 0xe6ac8a6f;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadAudioAction extends SendMessageAction {
        public static final int constructor = 0xf351d7ab;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadPhotoAction extends SendMessageAction {
        public static final int constructor = 0xd1d34a26;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadDocumentAction_old extends TL_sendMessageUploadDocumentAction {
        public static final int constructor = 0x8faee98e;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadVideoAction extends SendMessageAction {
        public static final int constructor = 0xe9763aec;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageCancelAction extends SendMessageAction {
        public static final int constructor = 0xfd5ec8f5;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageGeoLocationAction extends SendMessageAction {
        public static final int constructor = 0x176f8ba1;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageChooseContactAction extends SendMessageAction {
        public static final int constructor = 0x628cbc6f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageChooseStickerAction extends SendMessageAction {
        public static final int constructor = 0xb05ac6b1;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordRoundAction extends SendMessageAction {
        public static final int constructor = 0x88f27fbc;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadRoundAction extends SendMessageAction {
        public static final int constructor = 0x243e1c66;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadRoundAction_layer66 extends TL_sendMessageUploadRoundAction {
        public static final int constructor = 0xbb718624;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageEmojiInteractionSeen extends SendMessageAction {
        public static final int constructor = 0xb665902e;

        public String emoticon;

        public void readParams(InputSerializedData stream, boolean exception) {
            emoticon = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(emoticon);
        }
    }

    public static class TL_sendMessageTypingAction extends SendMessageAction {
        public static final int constructor = 0x16bf744e;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageHistoryImportAction extends SendMessageAction {
        public static final int constructor = 0xdbda9246;

        public int progress;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_sendMessageUploadPhotoAction_old extends TL_sendMessageUploadPhotoAction {
        public static final int constructor = 0x990a3c1a;

        public void readParams(InputSerializedData stream, boolean exception) {
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageUploadDocumentAction extends SendMessageAction {
        public static final int constructor = 0xaa0cd9e4;

        public void readParams(InputSerializedData stream, boolean exception) {
            progress = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(progress);
        }
    }

    public static class TL_speakingInGroupCallAction extends SendMessageAction {
        public static final int constructor = 0xd92c2285;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_sendMessageRecordVideoAction extends SendMessageAction {
        public static final int constructor = 0xa187d66f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class auth_SentCodeType extends TLObject {
        public int flags;
        public String url;
        public int length;
        public String pattern;
        public String prefix;
        public boolean apple_signin_allowed;
        public boolean google_signin_allowed;
        public String email_pattern;
        public int next_phone_login_date;
        public byte[] nonce;
        public long play_integrity_project_id;
        public byte[] play_integrity_nonce;
        public String receipt;
        public int push_timeout;
        public int reset_available_period;
        public int reset_pending_date;
        public String beginning;
        public boolean verifiedFirebase; //custom

        public static auth_SentCodeType TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            auth_SentCodeType result = null;
            switch (constructor) {
                case 0x3dbb5986:
                    result = new TL_auth_sentCodeTypeApp();
                    break;
                case 0x5353e5a7:
                    result = new TL_auth_sentCodeTypeCall();
                    break;
                case 0xf450f59b:
                    result = new TL_auth_sentCodeTypeEmailCode();
                    break;
                case 0xa5491dea:
                    result = new TL_auth_sentCodeTypeSetUpEmailRequired();
                    break;
                case 0xab03c6d9:
                    result = new TL_auth_sentCodeTypeFlashCall();
                    break;
                case 0x82006484:
                    result = new TL_auth_sentCodeTypeMissedCall();
                    break;
                case 0xc000bba2:
                    result = new TL_auth_sentCodeTypeSms();
                    break;
                case 0xd9565c39:
                    result = new TL_auth_sentCodeTypeFragmentSms();
                    break;
                case 0x9fd736:
                    result = new TL_auth_sentCodeTypeFirebaseSms();
                    break;
                case 0xa416ac81:
                    result = new TL_auth_sentCodeTypeSmsWord();
                    break;
                case 0xb37794af:
                    result = new TL_auth_sentCodeTypeSmsPhrase();
                    break;
            }
            return TLdeserialize(auth_SentCodeType.class, result, stream, constructor, exception);
        }
    }

    public static class TL_auth_sentCodeTypeApp extends auth_SentCodeType {
        public static final int constructor = 0x3dbb5986;

        public void readParams(InputSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeCall extends auth_SentCodeType {
        public static final int constructor = 0x5353e5a7;

        public void readParams(InputSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeEmailCode extends auth_SentCodeType {
        public static final int constructor = 0xf450f59b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            apple_signin_allowed = (flags & 1) != 0;
            google_signin_allowed = (flags & 2) != 0;
            email_pattern = stream.readString(exception);
            length = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                reset_available_period = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                reset_pending_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = apple_signin_allowed ? (flags | 1) : (flags &~ 1);
            flags = google_signin_allowed ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(email_pattern);
            stream.writeInt32(length);
            if ((flags & 8) != 0) {
                stream.writeInt32(reset_available_period);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(reset_pending_date);
            }
        }
    }

    public static class TL_auth_sentCodeTypeSetUpEmailRequired extends auth_SentCodeType {
        public static final int constructor = 0xa5491dea;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            apple_signin_allowed = (flags & 1) != 0;
            google_signin_allowed = (flags & 2) != 0;
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = apple_signin_allowed ? (flags | 1) : (flags &~ 1);
            flags = google_signin_allowed ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
        }
    }

    public static class TL_auth_sentCodeTypeFlashCall extends auth_SentCodeType {
        public static final int constructor = 0xab03c6d9;

        public void readParams(InputSerializedData stream, boolean exception) {
            pattern = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(pattern);
        }
    }

    public static class TL_auth_sentCodeTypeMissedCall extends auth_SentCodeType {
        public static final int constructor = 0x82006484;

        public void readParams(InputSerializedData stream, boolean exception) {
            prefix = stream.readString(exception);
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(prefix);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeSms extends auth_SentCodeType {
        public static final int constructor = 0xc000bba2;

        public void readParams(InputSerializedData stream, boolean exception) {
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeFragmentSms extends auth_SentCodeType {
        public static final int constructor = 0xd9565c39;

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeFirebaseSms extends auth_SentCodeType {
        public static final int constructor = 0x9fd736;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                nonce = stream.readByteArray(exception);
            }
            if ((flags & 4) != 0) {
                play_integrity_project_id = stream.readInt64(exception);
                play_integrity_nonce = stream.readByteArray(exception);
            }
            if ((flags & 2) != 0) {
                receipt = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                push_timeout = stream.readInt32(exception);
            }
            length = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeByteArray(nonce);
            }
            if ((flags & 4) != 0) {
                stream.writeInt64(play_integrity_project_id);
                stream.writeByteArray(play_integrity_nonce);
            }
            if ((flags & 2) != 0) {
                stream.writeString(receipt);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(push_timeout);
            }
            stream.writeInt32(length);
        }
    }

    public static class TL_auth_sentCodeTypeSmsWord extends auth_SentCodeType {
        public static final int constructor = 0xa416ac81;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                beginning = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(beginning);
            }
        }
    }

    public static class TL_auth_sentCodeTypeSmsPhrase extends auth_SentCodeType {
        public static final int constructor = 0xb37794af;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                beginning = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeString(beginning);
            }
        }
    }

    public static abstract class messages_StickerSetInstallResult extends TLObject {
        public ArrayList<StickerSetCovered> sets = new ArrayList<>();

        public static messages_StickerSetInstallResult TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_StickerSetInstallResult result = null;
            switch (constructor) {
                case 0x38641628:
                    result = new TL_messages_stickerSetInstallResultSuccess();
                    break;
                case 0x35e410a8:
                    result = new TL_messages_stickerSetInstallResultArchive();
                    break;
            }
            return TLdeserialize(messages_StickerSetInstallResult.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_stickerSetInstallResultSuccess extends messages_StickerSetInstallResult {
        public static final int constructor = 0x38641628;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_stickerSetInstallResultArchive extends messages_StickerSetInstallResult {
        public static final int constructor = 0x35e410a8;

        public void readParams(InputSerializedData stream, boolean exception) {
            sets = Vector.deserialize(stream, StickerSetCovered::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, sets);
        }
    }

    public static class PeerSettings extends TLObject {

        public int flags;
        public boolean report_spam;
        public boolean add_contact;
        public boolean block_contact;
        public boolean share_contact;
        public boolean need_contacts_exception;
        public boolean report_geo;
        public boolean autoarchived;
        public boolean invite_members;
        public boolean request_chat_broadcast;
        public int geo_distance;
        public String request_chat_title;
        public int request_chat_date;
        public boolean business_bot_paused;
        public boolean business_bot_can_reply;
        public long business_bot_id;
        public String business_bot_manage_url;
        public long charge_paid_message_stars;
        public String registration_month;
        public String phone_country;
        public int name_change_date;
        public int photo_change_date;

        public static PeerSettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PeerSettings result = null;
            switch (constructor) {
                case TL_peerSettings.constructor:
                    result = new TL_peerSettings();
                    break;
                case TL_peerSettings_layer199_3.constructor:
                    result = new TL_peerSettings_layer199_3();
                    break;
                case TL_peerSettings_layer199_2.constructor:
                    result = new TL_peerSettings_layer199_2();
                    break;
                case TL_peerSettings_layer199.constructor:
                    result = new TL_peerSettings_layer199();
                    break;
                case TL_peerSettings_layer176.constructor:
                    result = new TL_peerSettings_layer176();
                    break;
            }
            return TLdeserialize(PeerSettings.class, result, stream, constructor, exception);
        }
    }

    public static class TL_peerSettings extends PeerSettings {
        public static final int constructor = 0xf47741f7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            business_bot_paused = (flags & 2048) != 0;
            business_bot_can_reply = (flags & 4096) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_id = stream.readInt64(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_manage_url = stream.readString(exception);
            }
            if ((flags & 16384) != 0) {
                charge_paid_message_stars = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                registration_month = stream.readString(exception);
            }
            if ((flags & 65536) != 0) {
                phone_country = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                name_change_date = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                photo_change_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            flags = business_bot_paused ? (flags | 2048) : (flags &~ 2048);
            flags = business_bot_can_reply ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt64(business_bot_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeString(business_bot_manage_url);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(charge_paid_message_stars);
            }
            if ((flags & 32768) != 0) {
                stream.writeString(registration_month);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(phone_country);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(name_change_date);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(photo_change_date);
            }
        }
    }

    public static class TL_peerSettings_layer199_3 extends TL_peerSettings {
        public static final int constructor = 0xd8c39ec;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            business_bot_paused = (flags & 2048) != 0;
            business_bot_can_reply = (flags & 4096) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_id = stream.readInt64(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_manage_url = stream.readString(exception);
            }
            if ((flags & 16384) != 0) {
                charge_paid_message_stars = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                registration_month = stream.readString(exception);
            }
            if ((flags & 65536) != 0) {
                phone_country = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            flags = business_bot_paused ? (flags | 2048) : (flags &~ 2048);
            flags = business_bot_can_reply ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt64(business_bot_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeString(business_bot_manage_url);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(charge_paid_message_stars);
            }
            if ((flags & 32768) != 0) {
                stream.writeString(registration_month);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(phone_country);
            }
            if ((flags & 131072) != 0) {
                stream.writeString("");
            }
        }
    }

    public static class TL_peerSettings_layer199_2 extends TL_peerSettings {
        public static final int constructor = 0xa8639d72;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            business_bot_paused = (flags & 2048) != 0;
            business_bot_can_reply = (flags & 4096) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_id = stream.readInt64(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_manage_url = stream.readString(exception);
            }
            if ((flags & 16384) != 0) {
                charge_paid_message_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            flags = business_bot_paused ? (flags | 2048) : (flags &~ 2048);
            flags = business_bot_can_reply ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt64(business_bot_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeString(business_bot_manage_url);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(charge_paid_message_stars);
            }
        }
    }

    public static class TL_peerSettings_layer199 extends TL_peerSettings {
        public static final int constructor = 0xacd66c5e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            business_bot_paused = (flags & 2048) != 0;
            business_bot_can_reply = (flags & 4096) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_id = stream.readInt64(exception);
            }
            if ((flags & 8192) != 0) {
                business_bot_manage_url = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            flags = business_bot_paused ? (flags | 2048) : (flags &~ 2048);
            flags = business_bot_can_reply ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt64(business_bot_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeString(business_bot_manage_url);
            }
        }
    }

    public static class TL_peerSettings_layer176 extends TL_peerSettings {
        public static final int constructor = 0xa518110d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            report_spam = (flags & 1) != 0;
            add_contact = (flags & 2) != 0;
            block_contact = (flags & 4) != 0;
            share_contact = (flags & 8) != 0;
            need_contacts_exception = (flags & 16) != 0;
            report_geo = (flags & 32) != 0;
            autoarchived = (flags & 128) != 0;
            invite_members = (flags & 256) != 0;
            request_chat_broadcast = (flags & 1024) != 0;
            if ((flags & 64) != 0) {
                geo_distance = stream.readInt32(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_title = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                request_chat_date = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = report_spam ? (flags | 1) : (flags &~ 1);
            flags = add_contact ? (flags | 2) : (flags &~ 2);
            flags = block_contact ? (flags | 4) : (flags &~ 4);
            flags = share_contact ? (flags | 8) : (flags &~ 8);
            flags = need_contacts_exception ? (flags | 16) : (flags &~ 16);
            flags = report_geo ? (flags | 32) : (flags &~ 32);
            flags = autoarchived ? (flags | 128) : (flags &~ 128);
            flags = invite_members ? (flags | 256) : (flags &~ 256);
            flags = request_chat_broadcast ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(geo_distance);
            }
            if ((flags & 512) != 0) {
                stream.writeString(request_chat_title);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(request_chat_date);
            }
        }
    }

    public static class TL_readParticipantDate extends TLObject {
        public static final int constructor = 0x4a4ff172;

        public long user_id;
        public int date;

        public static TL_readParticipantDate TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_readParticipantDate result = TL_readParticipantDate.constructor != constructor ? null : new TL_readParticipantDate();
            return TLdeserialize(TL_readParticipantDate.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt32(date);
        }
    }

    public static abstract class InputDialogPeer extends TLObject {

        public static InputDialogPeer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputDialogPeer result = null;
            switch (constructor) {
                case 0xfcaafeb7:
                    result = new TL_inputDialogPeer();
                    break;
                case 0x64600527:
                    result = new TL_inputDialogPeerFolder();
                    break;
            }
            return TLdeserialize(InputDialogPeer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputDialogPeer extends InputDialogPeer {
        public static final int constructor = 0xfcaafeb7;

        public InputPeer peer;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_inputDialogPeerFolder extends InputDialogPeer {
        public static final int constructor = 0x64600527;

        public int folder_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            folder_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(folder_id);
        }
    }

    public static abstract class payments_PaymentResult extends TLObject {

        public static payments_PaymentResult TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            payments_PaymentResult result = null;
            switch (constructor) {
                case 0xd8411139:
                    result = new TL_payments_paymentVerificationNeeded();
                    break;
                case 0x4e5f810d:
                    result = new TL_payments_paymentResult();
                    break;
            }
            return TLdeserialize(payments_PaymentResult.class, result, stream, constructor, exception);
        }
    }

    public static class TL_payments_paymentVerificationNeeded extends payments_PaymentResult {
        public static final int constructor = 0xd8411139;

        public String url;

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_payments_paymentResult extends payments_PaymentResult {
        public static final int constructor = 0x4e5f810d;

        public Updates updates;

        public void readParams(InputSerializedData stream, boolean exception) {
            updates = Updates.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            updates.serializeToStream(stream);
        }
    }

    public static class TL_channels_adminLogResults extends TLObject {
        public static final int constructor = 0xed8af74d;

        public ArrayList<TL_channelAdminLogEvent> events = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_channels_adminLogResults TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_channels_adminLogResults result = TL_channels_adminLogResults.constructor != constructor ? null : new TL_channels_adminLogResults();
            return TLdeserialize(TL_channels_adminLogResults.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            events = Vector.deserialize(stream, TL_channelAdminLogEvent::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, events);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_inputPhoneContact extends TLObject {
        public static final int constructor = 0x6a1dc4be;

        public int flags;
        public long client_id;
        public String phone;
        public String first_name;
        public String last_name;
        public TL_textWithEntities note;

        public static TL_inputPhoneContact TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_inputPhoneContact result = TL_inputPhoneContact.constructor != constructor ? null : new TL_inputPhoneContact();
            return TLdeserialize(TL_inputPhoneContact.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            client_id = stream.readInt64(exception);
            phone = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            if (hasFlag(flags, FLAG_0)) {
                note = TL_textWithEntities.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(client_id);
            stream.writeString(phone);
            stream.writeString(first_name);
            stream.writeString(last_name);
            if (hasFlag(flags, FLAG_0)) {
                note.serializeToStream(stream);
            }
        }
    }

    public static abstract class ThemeSettings extends TLObject {

        public int flags;
        public boolean message_colors_animated;
        public BaseTheme base_theme;
        public int accent_color;
        public int outbox_accent_color;
        public ArrayList<Integer> message_colors = new ArrayList<>();
        public WallPaper wallpaper;

        public static ThemeSettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            ThemeSettings result = null;
            switch (constructor) {
                case 0xfa58b6d4:
                    result = new TL_themeSettings();
                    break;
                case 0x8db4e76c:
                    result = new TL_themeSettings_layer132();
                    break;
                case 0x9c14984a:
                    result = new TL_themeSettings_layer131();
                    break;
            }
            return TLdeserialize(ThemeSettings.class, result, stream, constructor, exception);
        }
    }

    public static class TL_themeSettings extends ThemeSettings {
        public static final int constructor = 0xfa58b6d4;

        public static TL_themeSettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_themeSettings result = TL_themeSettings.constructor != constructor ? null : new TL_themeSettings();
            return TLdeserialize(TL_themeSettings.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message_colors_animated = (flags & 4) != 0;
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 8) != 0) {
                outbox_accent_color = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                message_colors = Vector.deserializeInt(stream, exception);
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = message_colors_animated ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 8) != 0) {
                stream.writeInt32(outbox_accent_color);
            }
            if ((flags & 1) != 0) {
                Vector.serializeInt(stream, message_colors);
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_themeSettings_layer132 extends ThemeSettings {
        public static final int constructor = 0x8db4e76c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message_colors_animated = (flags & 4) != 0;
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                message_colors = Vector.deserializeInt(stream, exception);
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = message_colors_animated ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 1) != 0) {
                Vector.serializeInt(stream, message_colors);
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_themeSettings_layer131 extends ThemeSettings {
        public static final int constructor = 0x9c14984a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            base_theme = BaseTheme.TLdeserialize(stream, stream.readInt32(exception), exception);
            accent_color = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                int color = stream.readInt32(exception);
                if (color != 0) {
                    message_colors.add(color);
                }
            }
            if ((flags & 1) != 0) {
                int color = stream.readInt32(exception);
                if (color != 0) {
                    message_colors.add(0, color);
                }
            }
            if ((flags & 2) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            base_theme.serializeToStream(stream);
            stream.writeInt32(accent_color);
            if ((flags & 1) != 0) {
                stream.writeInt32(message_colors.size() > 1 ? message_colors.get(1) : 0);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(message_colors.size() > 0 ? message_colors.get(0) : 0);
            }
            if ((flags & 2) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_pageCaption extends TLObject {
        public static final int constructor = 0x6f747657;

        public RichText text;
        public RichText credit;

        public static TL_pageCaption TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_pageCaption result = TL_pageCaption.constructor != constructor ? null : new TL_pageCaption();
            return TLdeserialize(TL_pageCaption.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            credit = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            text.serializeToStream(stream);
            credit.serializeToStream(stream);
        }
    }

    public static abstract class PrivacyRule extends TLObject {

        public static PrivacyRule TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PrivacyRule result = null;
            switch (constructor) {
                case 0xf888fa1a:
                    result = new TL_privacyValueDisallowContacts();
                    break;
                case 0xe4621141:
                    result = new TL_privacyValueDisallowUsers();
                    break;
                case 0x6b134e8e:
                    result = new TL_privacyValueAllowChatParticipants();
                    break;
                case 0x41c87565:
                    result = new TL_privacyValueDisallowChatParticipants();
                    break;
                case 0x65427b82:
                    result = new TL_privacyValueAllowAll();
                    break;
                case 0x8b73e763:
                    result = new TL_privacyValueDisallowAll();
                    break;
                case 0xb8905fb2:
                    result = new TL_privacyValueAllowUsers();
                    break;
                case 0xfffe1bac:
                    result = new TL_privacyValueAllowContacts();
                    break;
                case 0xf7e8d89b:
                    result = new TL_privacyValueAllowCloseFriends();
                    break;
                case 0xece9814b:
                    result = new TL_privacyValueAllowPremium();
                    break;
                case TL_privacyValueAllowBots.constructor:
                    result = new TL_privacyValueAllowBots();
                    break;
                case TL_privacyValueDisallowBots.constructor:
                    result = new TL_privacyValueDisallowBots();
                    break;
            }
            return TLdeserialize(PrivacyRule.class, result, stream, constructor, exception);
        }
    }

    public static class TL_privacyValueDisallowContacts extends PrivacyRule {
        public static final int constructor = 0xf888fa1a;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueDisallowUsers extends PrivacyRule {
        public static final int constructor = 0xe4621141;

        public ArrayList<Long> users = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            users = Vector.deserializeLong(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serializeLong(stream, users);
        }
    }

    public static class TL_privacyValueAllowChatParticipants extends PrivacyRule {
        public static final int constructor = 0x6b134e8e;

        public ArrayList<Long> chats = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            chats = Vector.deserializeLong(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serializeLong(stream, chats);
        }
    }

    public static class TL_privacyValueDisallowChatParticipants extends PrivacyRule {
        public static final int constructor = 0x41c87565;

        public ArrayList<Long> chats = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            chats = Vector.deserializeLong(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serializeLong(stream, chats);
        }
    }

    public static class TL_privacyValueAllowAll extends PrivacyRule {
        public static final int constructor = 0x65427b82;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueDisallowAll extends PrivacyRule {
        public static final int constructor = 0x8b73e763;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueAllowUsers extends PrivacyRule {
        public static final int constructor = 0xb8905fb2;

        public ArrayList<Long> users = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            users = Vector.deserializeLong(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serializeLong(stream, users);
        }
    }

    public static class TL_privacyValueAllowContacts extends PrivacyRule {
        public static final int constructor = 0xfffe1bac;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueAllowCloseFriends extends PrivacyRule {
        public static final int constructor = 0xf7e8d89b;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueAllowPremium extends PrivacyRule {
        public static final int constructor = 0xece9814b;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueAllowBots extends PrivacyRule {
        public static final int constructor = 0x21461b5d;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_privacyValueDisallowBots extends PrivacyRule {
        public static final int constructor = 0xf6a5f82f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_pageTableCell extends TLObject {
        public static final int constructor = 0x34566b6a;

        public int flags;
        public boolean header;
        public boolean align_center;
        public boolean align_right;
        public boolean valign_middle;
        public boolean valign_bottom;
        public RichText text;
        public int colspan;
        public int rowspan;

        public static TL_pageTableCell TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_pageTableCell result = TL_pageTableCell.constructor != constructor ? null : new TL_pageTableCell();
            return TLdeserialize(TL_pageTableCell.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            header = (flags & 1) != 0;
            align_center = (flags & 8) != 0;
            align_right = (flags & 16) != 0;
            valign_middle = (flags & 32) != 0;
            valign_bottom = (flags & 64) != 0;
            if ((flags & 128) != 0) {
                text = RichText.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                colspan = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                rowspan = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = header ? (flags | 1) : (flags &~ 1);
            flags = align_center ? (flags | 8) : (flags &~ 8);
            flags = align_right ? (flags | 16) : (flags &~ 16);
            flags = valign_middle ? (flags | 32) : (flags &~ 32);
            flags = valign_bottom ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            if ((flags & 128) != 0) {
                text.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(colspan);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(rowspan);
            }
        }
    }

    public static class TL_messageMediaUnsupported_old extends TL_messageMediaUnsupported {
        public static final int constructor = 0x29632a36;

        public void readParams(InputSerializedData stream, boolean exception) {
            bytes = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(bytes);
        }
    }

    public static class TL_messageMediaAudio_layer45 extends MessageMedia {
        public static final int constructor = 0xc6b68300;

        public void readParams(InputSerializedData stream, boolean exception) {
            audio_unused = Audio.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            audio_unused.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaPhoto_old extends TL_messageMediaPhoto {
        public static final int constructor = 0xc8c45a2a;

        public void readParams(InputSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
        }
    }

    public static class MessageExtendedMedia extends TLObject {

        public String attachPath; // custom
        public float downloadProgress, uploadProgress; // custom

        public static MessageExtendedMedia TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            MessageExtendedMedia result = null;
            switch (constructor) {
                case 0xad628cc8:
                    result = new TL_messageExtendedMediaPreview();
                    break;
                case 0xee479c64:
                    result = new TL_messageExtendedMedia();
                    break;
            }
            return TLdeserialize(MessageExtendedMedia.class, result, stream, constructor, exception);
        }

    }

    public static class TL_messageExtendedMediaPreview extends MessageExtendedMedia {
        public static final int constructor = 0xad628cc8;

        public int flags;
        public int w;
        public int h;
        public PhotoSize thumb;
        public int video_duration;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                w = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                h = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                thumb = PhotoSize.TLdeserialize(0, 0, 0, stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                video_duration = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(w);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(h);
            }
            if ((flags & 2) != 0) {
                thumb.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(video_duration);
            }
        }
    }

    public static class TL_messageExtendedMedia extends MessageExtendedMedia {
        public static final int constructor = 0xee479c64;

        public MessageMedia media;

        public void readParams(InputSerializedData stream, boolean exception) {
            media = MessageMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            media.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaInvoice extends MessageMedia {
        public static final int constructor = 0xf6a548d3;

        public WebDocument webPhoto;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                webPhoto = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                receipt_msg_id = stream.readInt32(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            start_param = stream.readString(exception);
            if ((flags & 16) != 0) {
                extended_media.clear();
                extended_media.add(MessageExtendedMedia.TLdeserialize(stream, stream.readInt32(exception), exception));
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                webPhoto.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(receipt_msg_id);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(start_param);
            if ((flags & 16) != 0) {
                extended_media.get(0).serializeToStream(stream);
            }
        }
    }

    public static class TL_messageMediaInvoice_layer145 extends TL_messageMediaInvoice {
        public static final int constructor = 0x84551347;

        public WebDocument photo;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                receipt_msg_id = stream.readInt32(exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            start_param = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(receipt_msg_id);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            stream.writeString(start_param);
        }
    }

    public static class TL_messageMediaUnsupported extends MessageMedia {
        public static final int constructor = 0x9f84f49e;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageMediaEmpty extends MessageMedia {
        public static final int constructor = 0x3ded6320;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messageMediaVenue extends MessageMedia {
        public static final int constructor = 0x2ec0533f;

        public String icon; //custom
        public String emoji; //custom
        public long query_id; //custom
        public String result_id; //custom
        public TL_stories.TL_geoPointAddress geoAddress; //custom

        public void readParams(InputSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            venue_type = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            stream.writeString(venue_type);
        }
    }

    public static class TL_messageMediaVenue_layer71 extends MessageMedia {
        public static final int constructor = 0x7912b71f;

        public void readParams(InputSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
        }
    }

    public static class TL_messageMediaVideo_old extends TL_messageMediaVideo_layer45 {
        public static final int constructor = 0xa2d24290;

        public void readParams(InputSerializedData stream, boolean exception) {
            video_unused = Video.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            video_unused.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaDocument extends MessageMedia {
        public static final int constructor = 0x52d8ccd9;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            video = (flags & 64) != 0;
            round = (flags & 128) != 0;
            voice = (flags & 256) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                alt_documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
            }
            if ((flags & 512) != 0) {
                video_cover = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                video_timestamp = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = video ? (flags | 64) : (flags &~ 64);
            flags = round ? (flags | 128) : (flags &~ 128);
            flags = voice ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                Vector.serialize(stream, alt_documents);
            }
            if ((flags & 512) != 0) {
                video_cover.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                stream.writeInt32(video_timestamp);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer197_2 extends TL_messageMediaDocument {
        public static final int constructor = 0xdbbdf614;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            video = (flags & 64) != 0;
            round = (flags & 128) != 0;
            voice = (flags & 256) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                alt_documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
            }
            if ((flags & 512) != 0) {
                video_cover = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = video ? (flags | 64) : (flags &~ 64);
            flags = round ? (flags | 128) : (flags &~ 128);
            flags = voice ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                Vector.serialize(stream, alt_documents);
            }
            if ((flags & 512) != 0) {
                video_cover.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer197 extends TL_messageMediaDocument {
        public static final int constructor = 0xdd570bd5;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            video = (flags & 64) != 0;
            round = (flags & 128) != 0;
            voice = (flags & 256) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                alt_documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = video ? (flags | 64) : (flags &~ 64);
            flags = round ? (flags | 128) : (flags &~ 128);
            flags = voice ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                Vector.serialize(stream, alt_documents);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer187 extends TL_messageMediaDocument {
        public static final int constructor = 0x4cf4d72d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            video = (flags & 64) != 0;
            round = (flags & 128) != 0;
            voice = (flags & 256) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                alt_documents.add(Document.TLdeserialize(stream, stream.readInt32(exception), exception));
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = video ? (flags | 64) : (flags &~ 64);
            flags = round ? (flags | 128) : (flags &~ 128);
            flags = voice ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                alt_documents.get(0).serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer159 extends TL_messageMediaDocument {
        public static final int constructor = 0x9cb070d7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            nopremium = (flags & 8) != 0;
            spoiler = (flags & 16) != 0;
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                document = new TL_documentEmpty();
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = nopremium ? (flags | 8) : (flags &~ 8);
            flags = spoiler ? (flags | 16) : (flags &~ 16);
            flags = document != null ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_layer74 extends TL_messageMediaDocument {
        public static final int constructor = 0x7c4414d3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                document = new TL_documentEmpty();
            }
            if ((flags & 2) != 0) {
                captionLegacy = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeString(captionLegacy);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaDocument_old extends TL_messageMediaDocument {
        public static final int constructor = 0x2fda2204;

        public void readParams(InputSerializedData stream, boolean exception) {
            document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            document.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaDocument_layer68 extends TL_messageMediaDocument {
        public static final int constructor = 0xf3e02ea8;

        public void readParams(InputSerializedData stream, boolean exception) {
            document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            document.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaPhoto extends MessageMedia {
        public static final int constructor = 0x695150d7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            spoiler = (flags & 8) != 0;
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                photo = new TL_photoEmpty();
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = spoiler ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_messageMediaPoll extends MessageMedia {
        public static final int constructor = 0x4bd6e798;

        public Poll poll;
        public PollResults results;

        public void readParams(InputSerializedData stream, boolean exception) {
            poll = Poll.TLdeserialize(stream, stream.readInt32(exception), exception);
            results = PollResults.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            poll.serializeToStream(stream);
            results.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaPhoto_layer74 extends TL_messageMediaPhoto {
        public static final int constructor = 0xb5223b0f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            } else {
                photo = new TL_photoEmpty();
            }
            if ((flags & 2) != 0) {
                captionLegacy = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                ttl_seconds = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                stream.writeString(captionLegacy);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(ttl_seconds);
            }
        }
    }

    public static class TL_inputMediaInvoice extends InputMedia {
        public static final int constructor = 0x405fef0d;

        public int flags;
        public String title;
        public String description;
        public TL_inputWebDocument photo;
        public TL_invoice invoice;
        public byte[] payload;
        public String provider;
        public TL_dataJSON provider_data;
        public String start_param;
        public InputMedia extend_media;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = TL_inputWebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            invoice = TL_invoice.TLdeserialize(stream, stream.readInt32(exception), exception);
            payload = stream.readByteArray(exception);
            if ((flags & 8) != 0) {
                provider = stream.readString(exception);
            }
            provider_data = TL_dataJSON.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 2) != 0) {
                start_param = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                extend_media = InputMedia.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            invoice.serializeToStream(stream);
            stream.writeByteArray(payload);
            if ((flags & 8) != 0) {
                stream.writeString(provider);
            }
            provider_data.serializeToStream(stream);
            if ((flags & 2) != 0) {
                stream.writeString(start_param);
            }
            if ((flags & 4) != 0) {
                extend_media.serializeToStream(stream);
            }
        }
    }

    public static class TL_messageMediaGeoLive extends MessageMedia {
        public static final int constructor = 0xb940c666;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                heading = stream.readInt32(exception);
            }
            period = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                proximity_notification_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(heading);
            }
            stream.writeInt32(period);
            if ((flags & 2) != 0) {
                stream.writeInt32(proximity_notification_radius);
            }
        }
    }

    public static class TL_messageMediaGeoLive_layer119 extends TL_messageMediaGeoLive {
        public static final int constructor = 0x7c3c2609;

        public void readParams(InputSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            period = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
            stream.writeInt32(period);
        }
    }

    public static class TL_messageMediaGame extends MessageMedia {
        public static final int constructor = 0xfdb19008;

        public void readParams(InputSerializedData stream, boolean exception) {
            game = TL_game.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            game.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaContact_layer81 extends TL_messageMediaContact {
        public static final int constructor = 0x5e7d2f39;

        public void readParams(InputSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageMediaPhoto_layer68 extends TL_messageMediaPhoto {
        public static final int constructor = 0x3d8ce53d;

        public void readParams(InputSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaVideo_layer45 extends MessageMedia {
        public static final int constructor = 0x5bcf1675;

        public void readParams(InputSerializedData stream, boolean exception) {
            video_unused = Video.TLdeserialize(stream, stream.readInt32(exception), exception);
            captionLegacy = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            video_unused.serializeToStream(stream);
            stream.writeString(captionLegacy);
        }
    }

    public static class TL_messageMediaContact_layer131 extends TL_messageMediaContact {
        public static final int constructor = 0xcbf24940;

        public void readParams(InputSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            stream.writeInt32((int) user_id);
        }
    }

    public static class TL_messageMediaContact extends MessageMedia {
        public static final int constructor = 0x70322949;

        public void readParams(InputSerializedData stream, boolean exception) {
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_messageMediaVideoStream extends MessageMedia {
        public static final int constructor = 0xca5cab89;

        public int flags;
        public InputGroupCall call;
        public boolean rtmp_stream;

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            rtmp_stream = hasFlag(flags, FLAG_0);
            call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        @Override
        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = setFlag(flags, FLAG_0, rtmp_stream);
            stream.writeInt32(flags);
            call.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaDice extends MessageMedia {
        public static final int constructor = 0x3f7ee58b;

        public int value;
        public String emoticon;

        public void readParams(InputSerializedData stream, boolean exception) {
            value = stream.readInt32(exception);
            emoticon = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(value);
            stream.writeString(emoticon);
        }
    }

    public static class TL_messageMediaDice_layer111 extends TL_messageMediaDice {
        public static final int constructor = 0x638fe46b;

        public void readParams(InputSerializedData stream, boolean exception) {
            value = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(value);
        }
    }

    public static class TL_messageMediaGeo extends MessageMedia {
        public static final int constructor = 0x56e0d474;

        public void readParams(InputSerializedData stream, boolean exception) {
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            geo.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaWebPage extends MessageMedia {
        public static final int constructor = 0xddf10c3b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            force_large_media = (flags & 1) != 0;
            force_small_media = (flags & 2) != 0;
            manual = (flags & 8) != 0;
            safe = (flags & 16) != 0;
            webpage = WebPage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = force_large_media ? (flags | 1) : (flags &~ 1);
            flags = force_small_media ? (flags | 2) : (flags &~ 2);
            flags = manual ? (flags | 8) : (flags &~ 8);
            flags = safe ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            webpage.serializeToStream(stream);
        }
    }

    public static class TL_messageMediaWebPage_layer165 extends TL_messageMediaWebPage {
        public static final int constructor = 0xa32dd600;

        public void readParams(InputSerializedData stream, boolean exception) {
            webpage = WebPage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            webpage.serializeToStream(stream);
        }
    }

    public static abstract class LangPackString extends TLObject {
        public int flags;
        public String key;
        public String zero_value;
        public String one_value;
        public String two_value;
        public String few_value;
        public String many_value;
        public String other_value;
        public String value;

        public static LangPackString TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            LangPackString result = null;
            switch (constructor) {
                case 0x6c47ac9f:
                    result = new TL_langPackStringPluralized();
                    break;
                case 0xcad181f6:
                    result = new TL_langPackString();
                    break;
                case 0x2979eeb2:
                    result = new TL_langPackStringDeleted();
                    break;
            }
            return TLdeserialize(LangPackString.class, result, stream, constructor, exception);
        }
    }

    public static class TL_langPackStringPluralized extends LangPackString {
        public static final int constructor = 0x6c47ac9f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            key = stream.readString(exception);
            if ((flags & 1) != 0) {
                zero_value = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                one_value = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                two_value = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                few_value = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                many_value = stream.readString(exception);
            }
            other_value = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(key);
            if ((flags & 1) != 0) {
                stream.writeString(zero_value);
            }
            if ((flags & 2) != 0) {
                stream.writeString(one_value);
            }
            if ((flags & 4) != 0) {
                stream.writeString(two_value);
            }
            if ((flags & 8) != 0) {
                stream.writeString(few_value);
            }
            if ((flags & 16) != 0) {
                stream.writeString(many_value);
            }
            stream.writeString(other_value);
        }
    }

    public static class TL_langPackString extends LangPackString {
        public static final int constructor = 0xcad181f6;

        public void readParams(InputSerializedData stream, boolean exception) {
            key = stream.readString(exception);
            value = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(key);
            stream.writeString(value);
        }
    }

    public static class TL_langPackStringDeleted extends LangPackString {
        public static final int constructor = 0x2979eeb2;

        public void readParams(InputSerializedData stream, boolean exception) {
            key = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(key);
        }
    }

    public static abstract class auth_SentCode extends TLObject {

        public int flags;
        public auth_SentCodeType type;
        public String phone_code_hash;
        public auth_CodeType next_type;
        public int timeout;
        public auth_Authorization authorization;

        public static auth_SentCode TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            auth_SentCode result = null;
            switch (constructor) {
                case TL_auth_sentCodeSuccess.constructor:
                    result = new TL_auth_sentCodeSuccess();
                    break;
                case TL_auth_sentCode.constructor:
                    result = new TL_auth_sentCode();
                    break;
                case TL_auth_sentCodePaymentRequired.constructor:
                    result = new TL_auth_sentCodePaymentRequired();
                    break;
            }
            return TLdeserialize(auth_SentCode.class, result, stream, constructor, exception);
        }
    }

    public static class TL_auth_sentCodeSuccess extends auth_SentCode {
        public static final int constructor = 0x2390fe44;

        public void readParams(InputSerializedData stream, boolean exception) {
            authorization = auth_Authorization.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            authorization.serializeToStream(stream);
        }
    }

    public static class TL_auth_sentCode extends auth_SentCode {
        public static final int constructor = 0x5e002502;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = auth_SentCodeType.TLdeserialize(stream, stream.readInt32(exception), exception);
            phone_code_hash = stream.readString(exception);
            if ((flags & 2) != 0) {
                next_type = auth_CodeType.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                timeout = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            stream.writeString(phone_code_hash);
            if ((flags & 2) != 0) {
                next_type.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(timeout);
            }
        }
    }

    public static class TL_auth_sentCodePaymentRequired extends auth_SentCode {
        public static final int constructor = 0xe0955a3c;

        public String store_product;
        public String phone_code_hash;
        public String support_email_address;
        public String support_email_subject;
        public String currency;
        public long amount;

        public void readParams(InputSerializedData stream, boolean exception) {
            store_product = stream.readString(exception);
            phone_code_hash = stream.readString(exception);
            support_email_address = stream.readString(exception);
            support_email_subject = stream.readString(exception);
            currency = stream.readString(exception);
            amount = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(store_product);
            stream.writeString(phone_code_hash);
            stream.writeString(support_email_address);
            stream.writeString(support_email_subject);
            stream.writeString(currency);
            stream.writeInt64(amount);
        }
    }

    public static abstract class BotInlineResult extends TLObject {

        public int flags;
        public String id;
        public String type;
        public Photo photo;
        public Document document;
        public String title;
        public String description;
        public String url;
        public WebDocument thumb;
        public WebDocument content;
        public BotInlineMessage send_message;
        public long query_id;

        public static BotInlineResult TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            BotInlineResult result = null;
            switch (constructor) {
                case 0x11965f3a:
                    result = new TL_botInlineResult();
                    break;
                case 0x17db940b:
                    result = new TL_botInlineMediaResult();
                    break;
            }
            return TLdeserialize(BotInlineResult.class, result, stream, constructor, exception);
        }
    }

    public static class TL_botInlineResult extends BotInlineResult {
        public static final int constructor = 0x11965f3a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readString(exception);
            type = stream.readString(exception);
            if ((flags & 2) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                url = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                thumb = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                content = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            send_message = BotInlineMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(id);
            stream.writeString(type);
            if ((flags & 2) != 0) {
                stream.writeString(title);
            }
            if ((flags & 4) != 0) {
                stream.writeString(description);
            }
            if ((flags & 8) != 0) {
                stream.writeString(url);
            }
            if ((flags & 16) != 0) {
                thumb.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                content.serializeToStream(stream);
            }
            send_message.serializeToStream(stream);
        }
    }

    public static class TL_botInlineMediaResult extends BotInlineResult {
        public static final int constructor = 0x17db940b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readString(exception);
            type = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            send_message = BotInlineMessage.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(id);
            stream.writeString(type);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            send_message.serializeToStream(stream);
        }
    }

    public static class TL_notificationSoundDefault extends NotificationSound {
        public static final int constructor = 0x97e8bebe;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notificationSoundNone extends NotificationSound {
        public static final int constructor = 0x6f0c34df;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_notificationSoundRingtone extends NotificationSound {
        public static final int constructor = 0xff6c8049;

        public long id;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_notificationSoundLocal extends NotificationSound {
        public static final int constructor = 0x830b9ae4;

        public String title;
        public String data;

        public void readParams(InputSerializedData stream, boolean exception) {
            title = stream.readString(exception);
            data = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(title);
            stream.writeString(data);
        }
    }

    public static abstract class NotificationSound extends TLObject {

        public static NotificationSound TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            NotificationSound result = null;
            switch (constructor) {
                case 0x97e8bebe:
                    result = new TL_notificationSoundDefault();
                    break;
                case 0x6f0c34df:
                    result = new TL_notificationSoundNone();
                    break;
                case 0xff6c8049:
                    result = new TL_notificationSoundRingtone();
                    break;
                case 0x830b9ae4:
                    result = new TL_notificationSoundLocal();
                    break;
            }
            return TLdeserialize(NotificationSound.class, result, stream, constructor, exception);
        }
    }

    public static abstract class PeerNotifySettings extends TLObject {
        public int flags;
        public int mute_until;
        public String sound;
        public boolean show_previews;
        public int events_mask;
        public boolean silent;
        public NotificationSound ios_sound;
        public NotificationSound android_sound;
        public NotificationSound other_sound;
        public boolean stories_muted;
        public boolean stories_hide_sender;
        public NotificationSound stories_ios_sound;
        public NotificationSound stories_android_sound;
        public NotificationSound stories_other_sound;

        public static PeerNotifySettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PeerNotifySettings result = null;
            switch (constructor) {
                case 0x99622c0c:
                    result = new TL_peerNotifySettings();
                    break;
                case 0xa83b0426:
                    result = new TL_peerNotifySettings_layer156();
                    break;
                case 0x9acda4c0:
                    result = new TL_peerNotifySettings_layer77();
                    break;
                case 0xaf509d20:
                    result = new TL_peerNotifySettings_layer139();
                    break;
                case 0x8d5e11ee:
                    result = new TL_peerNotifySettings_layer47();
                    break;
                case 0x70a68512:
                    result = new TL_peerNotifySettingsEmpty_layer77();
                    break;
            }
            return TLdeserialize(PeerNotifySettings.class, result, stream, constructor, exception);
        }
    }

    public static class TL_peerNotifySettings_layer77 extends TL_peerNotifySettings {
        public static final int constructor = 0x9acda4c0;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            show_previews = (flags & 1) != 0;
            silent = (flags & 2) != 0;
            mute_until = stream.readInt32(exception);
            sound = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = show_previews ? (flags | 1) : (flags &~ 1);
            flags = silent ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeInt32(mute_until);
            stream.writeString(sound);
        }
    }

    public static class TL_peerNotifySettings extends PeerNotifySettings {
        public static final int constructor = 0x99622c0c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                ios_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16) != 0) {
                android_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                other_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                stories_muted = stream.readBool(exception);
            }
            if ((flags & 128) != 0) {
                stories_hide_sender = stream.readBool(exception);
            }
            if ((flags & 256) != 0) {
                stories_ios_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                stories_android_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                stories_other_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                ios_sound.serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                android_sound.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                other_sound.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                stream.writeBool(stories_muted);
            }
            if ((flags & 128) != 0) {
                stream.writeBool(stories_hide_sender);
            }
            if ((flags & 256) != 0) {
                stories_ios_sound.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stories_android_sound.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                stories_other_sound.serializeToStream(stream);
            }
        }
    }

    public static class TL_peerNotifySettings_layer156 extends TL_peerNotifySettings {
        public static final int constructor = 0xa83b0426;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                ios_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16) != 0) {
                android_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                other_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                ios_sound.serializeToStream(stream);
            }
            if ((flags & 16) != 0) {
                android_sound.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                other_sound.serializeToStream(stream);
            }
        }
    }

    public static class TL_peerNotifySettings_layer139 extends TL_peerNotifySettings {
        public static final int constructor = 0xaf509d20;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                sound = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                stream.writeString(sound);
            }
        }
    }

    public static class TL_peerNotifySettings_layer47 extends TL_peerNotifySettings {
        public static final int constructor = 0x8d5e11ee;

        public void readParams(InputSerializedData stream, boolean exception) {
            mute_until = stream.readInt32(exception);
            sound = stream.readString(exception);
            show_previews = stream.readBool(exception);
            events_mask = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(mute_until);
            stream.writeString(sound);
            stream.writeBool(show_previews);
            stream.writeInt32(events_mask);
        }
    }

    public static class TL_peerNotifySettingsEmpty_layer77 extends PeerNotifySettings {
        public static final int constructor = 0x70a68512;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class contacts_Blocked extends TLObject {

        public ArrayList<TL_peerBlocked> blocked = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public int count;

        public static contacts_Blocked TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            contacts_Blocked result = null;
            switch (constructor) {
                case 0xade1591:
                    result = new TL_contacts_blocked();
                    break;
                case 0xe1664194:
                    result = new TL_contacts_blockedSlice();
                    break;
            }
            return TLdeserialize(contacts_Blocked.class, result, stream, constructor, exception);
        }
    }

    public static class TL_contacts_blocked extends contacts_Blocked {
        public static final int constructor = 0xade1591;

        public void readParams(InputSerializedData stream, boolean exception) {
            blocked = Vector.deserialize(stream, TL_peerBlocked::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, blocked);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_contacts_blockedSlice extends contacts_Blocked {
        public static final int constructor = 0xe1664194;

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            blocked = Vector.deserialize(stream, TL_peerBlocked::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            Vector.serialize(stream, blocked);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_inputSecureValue extends TLObject {
        public static final int constructor = 0xdb21d0a7;

        public int flags;
        public SecureValueType type;
        public TL_secureData data;
        public InputSecureFile front_side;
        public InputSecureFile reverse_side;
        public InputSecureFile selfie;
        public ArrayList<InputSecureFile> translation = new ArrayList<>();
        public ArrayList<InputSecureFile> files = new ArrayList<>();
        public SecurePlainData plain_data;

        public static TL_inputSecureValue TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_inputSecureValue result = TL_inputSecureValue.constructor != constructor ? null : new TL_inputSecureValue();
            return TLdeserialize(TL_inputSecureValue.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                data = TL_secureData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                front_side = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                reverse_side = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                selfie = InputSecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                translation = Vector.deserialize(stream, InputSecureFile::TLdeserialize, exception);
            }
            if ((flags & 16) != 0) {
                files = Vector.deserialize(stream, InputSecureFile::TLdeserialize, exception);
            }
            if ((flags & 32) != 0) {
                plain_data = SecurePlainData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            if ((flags & 1) != 0) {
                data.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                front_side.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                reverse_side.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                selfie.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                Vector.serialize(stream, translation);
            }
            if ((flags & 16) != 0) {
                Vector.serialize(stream, files);
            }
            if ((flags & 32) != 0) {
                plain_data.serializeToStream(stream);
            }
        }
    }

    public static abstract class help_AppConfig extends TLObject {

        public static help_AppConfig TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_AppConfig result = null;
            switch (constructor) {
                case 0xdd18782e:
                    result = new TL_help_appConfig();
                    break;
                case 0x7cde641d:
                    result = new TL_help_appConfigNotModified();
                    break;
            }
            return TLdeserialize(help_AppConfig.class, result, stream, constructor, exception);
        }
    }

    public static class TL_help_appConfig extends help_AppConfig {
        public static final int constructor = 0xdd18782e;

        public int hash;
        public JSONValue config;

        public void readParams(InputSerializedData stream, boolean exception) {
            hash = stream.readInt32(exception);
            config = JSONValue.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(hash);
            config.serializeToStream(stream);
        }
    }

    public static class TL_help_appConfigNotModified extends help_AppConfig {
        public static final int constructor = 0x7cde641d;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class messages_DhConfig extends TLObject {
        public byte[] random;
        public int g;
        public byte[] p;
        public int version;

        public static messages_DhConfig TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_DhConfig result = null;
            switch (constructor) {
                case 0xc0e24635:
                    result = new TL_messages_dhConfigNotModified();
                    break;
                case 0x2c221edd:
                    result = new TL_messages_dhConfig();
                    break;
            }
            return TLdeserialize(messages_DhConfig.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_dhConfigNotModified extends messages_DhConfig {
        public static final int constructor = 0xc0e24635;

        public void readParams(InputSerializedData stream, boolean exception) {
            random = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(random);
        }
    }

    public static class TL_messages_dhConfig extends messages_DhConfig {
        public static final int constructor = 0x2c221edd;

        public void readParams(InputSerializedData stream, boolean exception) {
            g = stream.readInt32(exception);
            p = stream.readByteArray(exception);
            version = stream.readInt32(exception);
            random = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(g);
            stream.writeByteArray(p);
            stream.writeInt32(version);
            stream.writeByteArray(random);
        }
    }
    
    public static class DisallowedGiftsSettings extends TLObject {
        public static final int constructor = 0x71f276c4;
        
        public int flags;
        public boolean disallow_unlimited_stargifts;
        public boolean disallow_limited_stargifts;
        public boolean disallow_unique_stargifts;
        public boolean disallow_premium_gifts;
        public boolean disallow_stargifts_from_channels;

        public static DisallowedGiftsSettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final DisallowedGiftsSettings result = DisallowedGiftsSettings.constructor != constructor ? null : new DisallowedGiftsSettings();
            return TLdeserialize(DisallowedGiftsSettings.class, result, stream, constructor, exception);
        }

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            disallow_unlimited_stargifts = (flags & 1) != 0;
            disallow_limited_stargifts = (flags & 2) != 0;
            disallow_unique_stargifts = (flags & 4) != 0;
            disallow_premium_gifts = (flags & 8) != 0;
            disallow_stargifts_from_channels = (flags & 16) != 0;
        }

        @Override
        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = disallow_unlimited_stargifts ? flags | 1 : flags &~ 1;
            flags = disallow_limited_stargifts ? flags | 2 : flags &~ 2;
            flags = disallow_unique_stargifts ? flags | 4 : flags &~ 4;
            flags = disallow_premium_gifts ? flags | 8 : flags &~ 8;
            flags = disallow_stargifts_from_channels ? flags | 16 : flags &~ 16;
            stream.writeInt32(flags);
        }
    }

    public static class GlobalPrivacySettings extends TLObject {

        public int flags;
        public boolean archive_and_mute_new_noncontact_peers;
        public boolean keep_archived_unmuted;
        public boolean keep_archived_folders;
        public boolean hide_read_marks;
        public boolean new_noncontact_peers_require_premium;
        public boolean display_gifts_button;
        public long noncontact_peers_paid_stars;
        public DisallowedGiftsSettings disallowed_stargifts;

        public static GlobalPrivacySettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            GlobalPrivacySettings result = null;
            switch (constructor) {
                case TL_globalPrivacySettings.constructor:
                    result = new TL_globalPrivacySettings();
                    break;
                case TL_globalPrivacySettings_layer200.constructor:
                    result = new TL_globalPrivacySettings_layer200();
                    break;
            }
            return TLdeserialize(GlobalPrivacySettings.class, result, stream, constructor, exception);
        }
    }

    public static class TL_globalPrivacySettings extends GlobalPrivacySettings {
        public static final int constructor = 0xfe41b34f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            archive_and_mute_new_noncontact_peers = (flags & 1) != 0;
            keep_archived_unmuted = (flags & 2) != 0;
            keep_archived_folders = (flags & 4) != 0;
            hide_read_marks = (flags & 8) != 0;
            new_noncontact_peers_require_premium = (flags & 16) != 0;
            display_gifts_button = (flags & 128) != 0;
            if ((flags & 32) != 0) {
                noncontact_peers_paid_stars = stream.readInt64(exception);
            }
            if ((flags & 64) != 0) {
                disallowed_stargifts = DisallowedGiftsSettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = archive_and_mute_new_noncontact_peers ? (flags | 1) : (flags &~ 1);
            flags = keep_archived_unmuted ? (flags | 2) : (flags &~ 2);
            flags = keep_archived_folders ? (flags | 4) : (flags &~ 4);
            flags = hide_read_marks ? (flags | 8) : (flags &~ 8);
            flags = new_noncontact_peers_require_premium ? (flags | 16) : (flags &~ 16);
            flags = display_gifts_button ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            if ((flags & 32) != 0) {
                stream.writeInt64(noncontact_peers_paid_stars);
            }
            if ((flags & 64) != 0) {
                disallowed_stargifts.serializeToStream(stream);
            }
        }
    }

    public static class TL_globalPrivacySettings_layer200 extends TL_globalPrivacySettings {
        public static final int constructor = 0xc9d8df1c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            archive_and_mute_new_noncontact_peers = (flags & 1) != 0;
            keep_archived_unmuted = (flags & 2) != 0;
            keep_archived_folders = (flags & 4) != 0;
            hide_read_marks = (flags & 8) != 0;
            new_noncontact_peers_require_premium = (flags & 16) != 0;
            if ((flags & 32) != 0) {
                noncontact_peers_paid_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = archive_and_mute_new_noncontact_peers ? (flags | 1) : (flags &~ 1);
            flags = keep_archived_unmuted ? (flags | 2) : (flags &~ 2);
            flags = keep_archived_folders ? (flags | 4) : (flags &~ 4);
            flags = hide_read_marks ? (flags | 8) : (flags &~ 8);
            flags = new_noncontact_peers_require_premium ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            if ((flags & 32) != 0) {
                stream.writeInt64(noncontact_peers_paid_stars);
            }
        }
    }

    public static class TL_help_premiumPromo_layer144 extends TL_help_premiumPromo {
        public static final int constructor = 0x8a4f3c29;

        public void readParams(InputSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            status_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            video_sections = Vector.deserializeString(stream, exception);
            videos = Vector.deserialize(stream, Document::TLdeserialize, exception);
            currency = stream.readString(exception);
            monthly_amount = stream.readInt64(exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            Vector.serialize(stream, status_entities);
            Vector.serializeString(stream, video_sections);
            Vector.serialize(stream, videos);
            stream.writeString(currency);
            stream.writeInt64(monthly_amount);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_help_premiumPromo_layer140 extends TL_help_premiumPromo {
        public static final int constructor = 0xe0360f1b;

        public void readParams(InputSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            status_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            video_sections = Vector.deserializeString(stream, exception);
            videos = Vector.deserialize(stream, Document::TLdeserialize, exception);
            currency = stream.readString(exception);
            monthly_amount = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            Vector.serialize(stream, status_entities);
            Vector.serializeString(stream, video_sections);
            Vector.serialize(stream, videos);
            stream.writeString(currency);
            stream.writeInt64(monthly_amount);
        }
    }

    public static class TL_help_premiumPromo extends TLObject {
        public static final int constructor = 0x5334759c;

        public String status_text;
        public ArrayList<MessageEntity> status_entities = new ArrayList<>();
        public ArrayList<String> video_sections = new ArrayList<>();
        public ArrayList<Document> videos = new ArrayList<>();
        public ArrayList<TL_premiumSubscriptionOption> period_options = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public String currency;
        public long monthly_amount;

        public static TL_help_premiumPromo TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            TL_help_premiumPromo result = null;
            switch (constructor) {
                case 0x5334759c:
                    result = new TL_help_premiumPromo();
                    break;
                case 0x8a4f3c29:
                    result = new TL_help_premiumPromo_layer144();
                    break;
                case 0xe0360f1b:
                    result = new TL_help_premiumPromo_layer140();
                    break;
            }
            result = TLdeserialize(TL_help_premiumPromo.class, result, stream, constructor, exception);
            if (result != null && result.currency != null) {
                TL_help_premiumPromo finalResult = result;
                result.period_options.add(new TL_premiumSubscriptionOption() {{
                    months = 1;
                    currency = finalResult.currency;
                    amount = finalResult.monthly_amount;
                    store_product = "telegram_premium";
                }});
            }
            return result;
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            status_text = stream.readString(exception);
            status_entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            video_sections = Vector.deserializeString(stream, exception);
            videos = Vector.deserialize(stream, Document::TLdeserialize, exception);
            period_options = Vector.deserialize(stream, TL_premiumSubscriptionOption::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(status_text);
            Vector.serialize(stream, status_entities);
            Vector.serializeString(stream, video_sections);
            Vector.serialize(stream, videos);
            Vector.serialize(stream, period_options);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class help_UserInfo extends TLObject {

        public static help_UserInfo TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_UserInfo result = null;
            switch (constructor) {
                case 0xf3ae2eed:
                    result = new TL_help_userInfoEmpty();
                    break;
                case 0x1eb3758:
                    result = new TL_help_userInfo();
                    break;
            }
            return TLdeserialize(help_UserInfo.class, result, stream, constructor, exception);
        }
    }

    public static class TL_help_userInfoEmpty extends help_UserInfo {
        public static final int constructor = 0xf3ae2eed;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_userInfo extends help_UserInfo {
        public static final int constructor = 0x1eb3758;

        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public String author;
        public int date;

        public void readParams(InputSerializedData stream, boolean exception) {
            message = stream.readString(exception);
            entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            author = stream.readString(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(message);
            Vector.serialize(stream, entities);
            stream.writeString(author);
            stream.writeInt32(date);
        }
    }

    public static class TL_secureValueHash extends TLObject {
        public static final int constructor = 0xed1ecdb0;

        public SecureValueType type;
        public byte[] hash;

        public static TL_secureValueHash TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_secureValueHash result = TL_secureValueHash.constructor != constructor ? null : new TL_secureValueHash();
            return TLdeserialize(TL_secureValueHash.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            hash = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(hash);
        }
    }

    public static abstract class messages_StickerSet extends TLObject {

        public StickerSet set;
        public ArrayList<TL_stickerPack> packs = new ArrayList<>();
        public ArrayList<TL_stickerKeyword> keywords = new ArrayList<>();
        public ArrayList<Document> documents = new ArrayList<>();

        public static TL_messages_stickerSet TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            TL_messages_stickerSet result = null;
            switch (constructor) {
                case 0x6e153f16:
                    result = new TL_messages_stickerSet();
                    break;
                case 0xb60a24a6:
                    result = new TL_messages_stickerSet_layer146();
                    break;
                case 0xd3f924eb:
                    result = new TL_messages_stickerSetNotModified();
                    break;
            }
            return TLdeserialize(TL_messages_stickerSet.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_stickerSet_layer146 extends TL_messages_stickerSet {
        public static final int constructor = 0xb60a24a6;

        public void readParams(InputSerializedData stream, boolean exception) {
            set = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            packs = Vector.deserialize(stream, TL_stickerPack::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            set.serializeToStream(stream);
            Vector.serialize(stream, packs);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_messages_stickerSet extends messages_StickerSet {
        public static final int constructor = 0x6e153f16;

        public void readParams(InputSerializedData stream, boolean exception) {
            set = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            packs = Vector.deserialize(stream, TL_stickerPack::TLdeserialize, exception);
            keywords = Vector.deserialize(stream, TL_stickerKeyword::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            set.serializeToStream(stream);
            Vector.serialize(stream, packs);
            Vector.serialize(stream, keywords);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_messages_stickerSetNotModified extends TL_messages_stickerSet {
        public static final int constructor = 0xd3f924eb;

        public void readParams(InputSerializedData stream, boolean exception) {

        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class InputGeoPoint extends TLObject {

        public int flags;
        public double lat;
        public double _long;
        public int accuracy_radius;

        public static InputGeoPoint TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputGeoPoint result = null;
            switch (constructor) {
                case 0x48222faf:
                    result = new TL_inputGeoPoint();
                    break;
                case 0xe4c123d6:
                    result = new TL_inputGeoPointEmpty();
                    break;
            }
            return TLdeserialize(InputGeoPoint.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputGeoPoint extends InputGeoPoint {
        public static final int constructor = 0x48222faf;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            lat = stream.readDouble(exception);
            _long = stream.readDouble(exception);
            if ((flags & 1) != 0) {
                accuracy_radius = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeDouble(lat);
            stream.writeDouble(_long);
            if ((flags & 1) != 0) {
                stream.writeInt32(accuracy_radius);
            }
        }
    }

    public static class TL_inputGeoPointEmpty extends InputGeoPoint {
        public static final int constructor = 0xe4c123d6;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_inviteText extends TLObject {
        public static final int constructor = 0x18cb9f78;

        public String message;

        public static TL_help_inviteText TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_help_inviteText result = TL_help_inviteText.constructor != constructor ? null : new TL_help_inviteText();
            return TLdeserialize(TL_help_inviteText.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            message = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(message);
        }
    }

    public static abstract class Audio extends TLObject {
        public long id;
        public long access_hash;
        public int date;
        public int duration;
        public String mime_type;
        public int size;
        public int dc_id;
        public long user_id;
        public byte[] key;
        public byte[] iv;

        public static Audio TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            Audio result = null;
            switch (constructor) {
                case 0x586988d8:
                    result = new TL_audioEmpty_layer45();
                    break;
                case 0xf9e35055:
                    result = new TL_audio_layer45();
                    break;
                case 0x427425e7:
                    result = new TL_audio_old();
                    break;
                case 0x555555F6:
                    result = new TL_audioEncrypted();
                    break;
                case 0xc7ac6496:
                    result = new TL_audio_old2();
                    break;
            }
            return TLdeserialize(Audio.class, result, stream, constructor, exception);
        }
    }

    public static class TL_audioEmpty_layer45 extends Audio {
        public static final int constructor = 0x586988d8;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_audio_layer45 extends Audio {
        public static final int constructor = 0xf9e35055;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            mime_type = stream.readString(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeString(mime_type);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_audio_old extends TL_audio_layer45 {
        public static final int constructor = 0x427425e7;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_audioEncrypted extends TL_audio_layer45 {
        public static final int constructor = 0x555555F6;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
            key = stream.readByteArray(exception);
            iv = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
            stream.writeByteArray(key);
            stream.writeByteArray(iv);
        }
    }

    public static class TL_audio_old2 extends TL_audio_layer45 {
        public static final int constructor = 0xc7ac6496;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            user_id = stream.readInt32(exception);
            date = stream.readInt32(exception);
            duration = stream.readInt32(exception);
            mime_type = stream.readString(exception);
            size = stream.readInt32(exception);
            dc_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeInt32((int) user_id);
            stream.writeInt32(date);
            stream.writeInt32(duration);
            stream.writeString(mime_type);
            stream.writeInt32(size);
            stream.writeInt32(dc_id);
        }
    }

    public static class TL_help_country extends TLObject {
        public static final int constructor = 0xc3878e23;

        public int flags;
        public boolean hidden;
        public String iso2;
        public String default_name;
        public String name;
        public ArrayList<TL_help_countryCode> country_codes = new ArrayList<>();

        public static TL_help_country TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_help_country result = TL_help_country.constructor != constructor ? null : new TL_help_country();
            return TLdeserialize(TL_help_country.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            hidden = (flags & 1) != 0;
            iso2 = stream.readString(exception);
            default_name = stream.readString(exception);
            if ((flags & 2) != 0) {
                name = stream.readString(exception);
            }
            country_codes = Vector.deserialize(stream, TL_help_countryCode::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = hidden ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(iso2);
            stream.writeString(default_name);
            if ((flags & 2) != 0) {
                stream.writeString(name);
            }
            Vector.serialize(stream, country_codes);
        }
    }

    public static abstract class SecurePasswordKdfAlgo extends TLObject {

        public static SecurePasswordKdfAlgo TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            SecurePasswordKdfAlgo result = null;
            switch (constructor) {
                case 0xbbf2dda0:
                    result = new TL_securePasswordKdfAlgoPBKDF2HMACSHA512iter100000();
                    break;
                case 0x86471d92:
                    result = new TL_securePasswordKdfAlgoSHA512();
                    break;
                case 0x4a8537:
                    result = new TL_securePasswordKdfAlgoUnknown();
                    break;
            }
            return TLdeserialize(SecurePasswordKdfAlgo.class, result, stream, constructor, exception);
        }
    }

    public static class TL_securePasswordKdfAlgoPBKDF2HMACSHA512iter100000 extends SecurePasswordKdfAlgo {
        public static final int constructor = 0xbbf2dda0;

        public byte[] salt;

        public void readParams(InputSerializedData stream, boolean exception) {
            salt = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(salt);
        }
    }

    public static class TL_securePasswordKdfAlgoSHA512 extends SecurePasswordKdfAlgo {
        public static final int constructor = 0x86471d92;

        public byte[] salt;

        public void readParams(InputSerializedData stream, boolean exception) {
            salt = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeByteArray(salt);
        }
    }

    public static class TL_securePasswordKdfAlgoUnknown extends SecurePasswordKdfAlgo {
        public static final int constructor = 0x4a8537;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_historyImport extends TLObject {
        public static final int constructor = 0x1662af0b;

        public long id;

        public static TL_messages_historyImport TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_historyImport result = TL_messages_historyImport.constructor != constructor ? null : new TL_messages_historyImport();
            return TLdeserialize(TL_messages_historyImport.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static abstract class InputGame extends TLObject {
        public InputUser bot_id;
        public String short_name;
        public long id;
        public long access_hash;

        public static InputGame TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputGame result = null;
            switch (constructor) {
                case 0xc331e80a:
                    result = new TL_inputGameShortName();
                    break;
                case 0x32c3e77:
                    result = new TL_inputGameID();
                    break;
            }
            return TLdeserialize(InputGame.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputGameShortName extends InputGame {
        public static final int constructor = 0xc331e80a;

        public void readParams(InputSerializedData stream, boolean exception) {
            bot_id = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
            short_name = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            bot_id.serializeToStream(stream);
            stream.writeString(short_name);
        }
    }

    public static class TL_inputGameID extends InputGame {
        public static final int constructor = 0x32c3e77;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static abstract class MessageReplies extends TLObject {

        public int flags;
        public boolean comments;
        public int replies;
        public int replies_pts;
        public ArrayList<Peer> recent_repliers = new ArrayList<>();
        public long channel_id;
        public int max_id;
        public int read_max_id;

        public static MessageReplies TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            MessageReplies result = null;
            switch (constructor) {
                case 0x4128faac:
                    result = new TL_messageReplies_layer131();
                    break;
                case 0x83d60fc2:
                    result = new TL_messageReplies();
                    break;
            }
            return TLdeserialize(MessageReplies.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messageReplies_layer131 extends TL_messageReplies {
        public static final int constructor = 0x4128faac;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            comments = (flags & 1) != 0;
            replies = stream.readInt32(exception);
            replies_pts = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                recent_repliers = Vector.deserialize(stream, Peer::TLdeserialize, exception);
            }
            if ((flags & 1) != 0) {
                channel_id = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                read_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = comments ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(replies);
            stream.writeInt32(replies_pts);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, recent_repliers);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32((int) channel_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(read_max_id);
            }
        }
    }

    public static class TL_messageReplies extends MessageReplies {
        public static final int constructor = 0x83d60fc2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            comments = (flags & 1) != 0;
            replies = stream.readInt32(exception);
            replies_pts = stream.readInt32(exception);
            if ((flags & 2) != 0) {
                recent_repliers = Vector.deserialize(stream, Peer::TLdeserialize, exception);
            }
            if ((flags & 1) != 0) {
                channel_id = stream.readInt64(exception);
            }
            if ((flags & 4) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                read_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = comments ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt32(replies);
            stream.writeInt32(replies_pts);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, recent_repliers);
            }
            if ((flags & 1) != 0) {
                stream.writeInt64(channel_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(read_max_id);
            }
        }
    }

    public static abstract class messages_SponsoredMessages extends TLObject {
        public int flags;
        public int posts_between;
        public int start_delay;
        public int between_delay;
        public ArrayList<TL_sponsoredMessage> messages = new ArrayList<>();
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static messages_SponsoredMessages TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_SponsoredMessages result = null;
            switch (constructor) {
                case TL_messages_sponsoredMessagesEmpty.constructor:
                    result = new TL_messages_sponsoredMessagesEmpty();
                    break;
                case TL_messages_sponsoredMessages.constructor:
                    result = new TL_messages_sponsoredMessages();
                    break;
            }
            return TLdeserialize(messages_SponsoredMessages.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_sponsoredMessagesEmpty extends messages_SponsoredMessages {
        public static final int constructor = 0x1839490f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_sponsoredMessages extends messages_SponsoredMessages {
        public static final int constructor = 0xffda656d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                posts_between = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                start_delay = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                between_delay = stream.readInt32(exception);
            }
            messages = Vector.deserialize(stream, TL_sponsoredMessage::TLdeserialize, exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(posts_between);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(start_delay);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(between_delay);
            }
            Vector.serialize(stream, messages);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messageViews extends TLObject {
        public static final int constructor = 0x455b853d;

        public int flags;
        public int views;
        public int forwards;
        public MessageReplies replies;

        public static TL_messageViews TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messageViews result = TL_messageViews.constructor != constructor ? null : new TL_messageViews();
            return TLdeserialize(TL_messageViews.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                views = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                forwards = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                replies = MessageReplies.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(views);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(forwards);
            }
            if ((flags & 4) != 0) {
                replies.serializeToStream(stream);
            }
        }
    }

    public static abstract class ReplyMarkup extends TLObject {
        public int flags;
        public boolean resize;
        public boolean single_use;
        public boolean is_persistent;
        public boolean selective;
        public String placeholder;
        public ArrayList<TL_keyboardButtonRow> rows = new ArrayList<>();

        public static ReplyMarkup TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            ReplyMarkup result = null;
            switch (constructor) {
                case 0x85dd99d1:
                    result = new TL_replyKeyboardMarkup();
                    break;
                case 0xa03e5b85:
                    result = new TL_replyKeyboardHide();
                    break;
                case 0x86b40b08:
                    result = new TL_replyKeyboardForceReply();
                    break;
                case 0x3502758c:
                    result = new TL_replyKeyboardMarkup_layer129();
                    break;
                case 0xf4108aa0:
                    result = new TL_replyKeyboardForceReply_layer129();
                    break;
                case 0x48a30254:
                    result = new TL_replyInlineMarkup();
                    break;
            }
            return TLdeserialize(ReplyMarkup.class, result, stream, constructor, exception);
        }
    }

    public static class TL_replyKeyboardMarkup extends ReplyMarkup {
        public static final int constructor = 0x85dd99d1;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            resize = (flags & 1) != 0;
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            is_persistent = (flags & 16) != 0;
            rows = Vector.deserialize(stream, TL_keyboardButtonRow::TLdeserialize, exception);
            if ((flags & 8) != 0) {
                placeholder = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = resize ? (flags | 1) : (flags &~ 1);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            flags = is_persistent ? (flags | 16) : (flags &~ 16);
            stream.writeInt32(flags);
            Vector.serialize(stream, rows);
            if ((flags & 8) != 0) {
                stream.writeString(placeholder);
            }
        }
    }

    public static class TL_replyKeyboardHide extends ReplyMarkup {
        public static final int constructor = 0xa03e5b85;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            selective = (flags & 4) != 0;
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
        }
    }

    public static class TL_replyKeyboardForceReply extends ReplyMarkup {
        public static final int constructor = 0x86b40b08;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            if ((flags & 8) != 0) {
                placeholder = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            if ((flags & 8) != 0) {
                stream.writeString(placeholder);
            }
        }
    }

    public static class TL_replyKeyboardMarkup_layer129 extends TL_replyKeyboardMarkup {
        public static final int constructor = 0x3502758c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            resize = (flags & 1) != 0;
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
            rows = Vector.deserialize(stream, TL_keyboardButtonRow::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = resize ? (flags | 1) : (flags &~ 1);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            Vector.serialize(stream, rows);
        }
    }

    public static class TL_replyKeyboardForceReply_layer129 extends TL_replyKeyboardForceReply {
        public static final int constructor = 0xf4108aa0;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            single_use = (flags & 2) != 0;
            selective = (flags & 4) != 0;
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = single_use ? (flags | 2) : (flags &~ 2);
            flags = selective ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
        }
    }

    public static class TL_replyInlineMarkup extends ReplyMarkup {
        public static final int constructor = 0x48a30254;

        public void readParams(InputSerializedData stream, boolean exception) {
            rows = Vector.deserialize(stream, TL_keyboardButtonRow::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, rows);
        }
    }

    public static class WebPageAttribute extends TLObject {
        public int flags;

        private static WebPageAttribute fromConstructor(int constructor) {
            switch (constructor) {
                case TL_webPageAttributeTheme.constructor:              return new TL_webPageAttributeTheme();
                case TL_webPageAttributeStory_layer162.constructor:     return new TL_webPageAttributeStory_layer162();
                case TL_webPageAttributeStory.constructor:              return new TL_webPageAttributeStory();
                case TL_webPageAttributeStickerSet.constructor:         return new TL_webPageAttributeStickerSet();
                case TL_webPageAttributeUniqueStarGift.constructor:     return new TL_webPageAttributeUniqueStarGift();
                case TL_webPageAttributeStarGiftCollection.constructor: return new TL_webPageAttributeStarGiftCollection();
                case TL_webPageAttributeStarGiftAuction.constructor:    return new TL_webPageAttributeStarGiftAuction();
            }
            return null;
        }

        public static WebPageAttribute TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            return TLdeserialize(WebPageAttribute.class, fromConstructor(constructor), stream, constructor, exception);
        }
    }

    public static class TL_webPageAttributeStarGiftAuction extends WebPageAttribute {
        public final static int constructor = 0x34986ab;

        public TL_stars.StarGift gift;
        public int end_date;
        public int center_color;
        public int edge_color;
        public int text_color;

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {
            gift = TL_stars.StarGift.TLdeserialize(stream, stream.readInt32(exception), exception);
            end_date = stream.readInt32(exception);
            center_color = stream.readInt32(exception);
            edge_color = stream.readInt32(exception);
            text_color = stream.readInt32(exception);
        }

        @Override
        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            gift.serializeToStream(stream);
            stream.writeInt32(end_date);
            stream.writeInt32(center_color);
            stream.writeInt32(edge_color);
            stream.writeInt32(text_color);
        }
    }

    public static class TL_webPageAttributeStory extends WebPageAttribute {
        public final static int constructor = 0x2e94c3e7;

        public Peer peer;
        public int id;
        public TL_stories.StoryItem storyItem;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            peer = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                storyItem = TL_stories.StoryItem.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            peer.serializeToStream(stream);
            stream.writeInt32(id);
            if ((flags & 1) != 0) {
                storyItem.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPageAttributeStickerSet extends WebPageAttribute {
        public final static int constructor = 0x50cc03d3;

        public Peer peer;
        public boolean emojis;
        public boolean text_color;
        public ArrayList<TLRPC.Document> stickers = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            emojis = (flags & 1) != 0;
            text_color = (flags & 2) != 0;
            stickers = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = emojis ? (flags | 1) : (flags &~ 1);
            flags = text_color ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            Vector.serialize(stream, stickers);
        }
    }

    public static class TL_webPageAttributeStarGiftCollection extends WebPageAttribute {
        public final static int constructor = 0x31cad303;

        public ArrayList<TLRPC.Document> icons = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            icons = Vector.deserialize(stream, TLRPC.Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, icons);
        }
    }

    public static class TL_webPageAttributeUniqueStarGift extends WebPageAttribute {
        public final static int constructor = 0xcf6f6db8;

        public TL_stars.StarGift gift;

        public void readParams(InputSerializedData stream, boolean exception) {
            gift = TL_stars.StarGift.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            gift.serializeToStream(stream);
        }
    }

    public static class TL_webPageAttributeStory_layer162 extends TL_webPageAttributeStory {
        public static final int constructor = 0x939a4671;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            long user_id = stream.readInt64(exception);
            peer = new TL_peerUser();
            peer.user_id = user_id;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                storyItem = TL_stories.StoryItem.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (storyItem != null) {
                flags |= 1;
            } else {
                flags &= ~1;
            }
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(peer.user_id);
            stream.writeInt32(id);
            if ((flags & 1) != 0) {
                storyItem.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPageAttributeTheme extends WebPageAttribute {
        public static final int constructor = 0x54b56617;

        public ArrayList<Document> documents = new ArrayList<>();
        public ThemeSettings settings;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
            }
            if ((flags & 2) != 0) {
                settings = ThemeSettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                Vector.serialize(stream, documents);
            }
            if ((flags & 2) != 0) {
                settings.serializeToStream(stream);
            }
        }
    }

    public static abstract class contacts_Contacts extends TLObject {
        public ArrayList<TL_contact> contacts = new ArrayList<>();
        public int saved_count;
        public ArrayList<User> users = new ArrayList<>();

        public static contacts_Contacts TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            contacts_Contacts result = null;
            switch (constructor) {
                case 0xb74ba9d2:
                    result = new TL_contacts_contactsNotModified();
                    break;
                case 0xeae87e42:
                    result = new TL_contacts_contacts();
                    break;
            }
            return TLdeserialize(contacts_Contacts.class, result, stream, constructor, exception);
        }
    }

    public static class TL_contacts_contactsNotModified extends contacts_Contacts {
        public static final int constructor = 0xb74ba9d2;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_contacts_contacts extends contacts_Contacts {
        public static final int constructor = 0xeae87e42;

        public void readParams(InputSerializedData stream, boolean exception) {
            contacts = Vector.deserialize(stream, TL_contact::TLdeserialize, exception);
            saved_count = stream.readInt32(exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, contacts);
            stream.writeInt32(saved_count);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class SecureRequiredType extends TLObject {

        public static SecureRequiredType TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            SecureRequiredType result = null;
            switch (constructor) {
                case 0x829d99da:
                    result = new TL_secureRequiredType();
                    break;
                case 0x27477b4:
                    result = new TL_secureRequiredTypeOneOf();
                    break;
            }
            return TLdeserialize(SecureRequiredType.class, result, stream, constructor, exception);
        }
    }

    public static class TL_secureRequiredType extends SecureRequiredType {
        public static final int constructor = 0x829d99da;

        public int flags;
        public boolean native_names;
        public boolean selfie_required;
        public boolean translation_required;
        public SecureValueType type;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            native_names = (flags & 1) != 0;
            selfie_required = (flags & 2) != 0;
            translation_required = (flags & 4) != 0;
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = native_names ? (flags | 1) : (flags &~ 1);
            flags = selfie_required ? (flags | 2) : (flags &~ 2);
            flags = translation_required ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
        }
    }

    public static class TL_secureRequiredTypeOneOf extends SecureRequiredType {
        public static final int constructor = 0x27477b4;

        public ArrayList<SecureRequiredType> types = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            types = Vector.deserialize(stream, SecureRequiredType::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, types);
        }
    }

    public static abstract class InputPrivacyKey extends TLObject {

        public static InputPrivacyKey TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputPrivacyKey result = null;
            switch (constructor) {
                case TL_inputPrivacyKeyStatusTimestamp.constructor:
                    result = new TL_inputPrivacyKeyStatusTimestamp();
                    break;
                case TL_inputPrivacyKeyChatInvite.constructor:
                    result = new TL_inputPrivacyKeyChatInvite();
                    break;
                case TL_inputPrivacyKeyPhoneCall.constructor:
                    result = new TL_inputPrivacyKeyPhoneCall();
                    break;
                case TL_inputPrivacyKeyForwards.constructor:
                    result = new TL_inputPrivacyKeyForwards();
                    break;
                case TL_inputPrivacyKeyProfilePhoto.constructor:
                    result = new TL_inputPrivacyKeyProfilePhoto();
                    break;
                case TL_inputPrivacyKeyPhoneNumber.constructor:
                    result = new TL_inputPrivacyKeyPhoneNumber();
                    break;
                case TL_inputPrivacyKeyAddedByPhone.constructor:
                    result = new TL_inputPrivacyKeyAddedByPhone();
                    break;
                case TL_inputPrivacyKeyVoiceMessages.constructor:
                    result = new TL_inputPrivacyKeyVoiceMessages();
                    break;
                case TL_inputPrivacyKeyPhoneP2P.constructor:
                    result = new TL_inputPrivacyKeyPhoneP2P();
                    break;
                case TL_inputPrivacyKeyAbout.constructor:
                    result = new TL_inputPrivacyKeyAbout();
                    break;
                case TL_inputPrivacyKeyBirthday.constructor:
                    result = new TL_inputPrivacyKeyBirthday();
                    break;
                case TL_inputPrivacyKeyStarGiftsAutoSave.constructor:
                    result = new TL_inputPrivacyKeyStarGiftsAutoSave();
                    break;
                case TL_inputPrivacyKeyNoPaidMessages.constructor:
                    result = new TL_inputPrivacyKeyNoPaidMessages();
                    break;
                case TL_inputPrivacyKeySavedMusic.constructor:
                    result = new TL_inputPrivacyKeySavedMusic();
                    break;
            }
            return TLdeserialize(InputPrivacyKey.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputPrivacyKeyStatusTimestamp extends InputPrivacyKey {
        public static final int constructor = 0x4f96cb18;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyChatInvite extends InputPrivacyKey {
        public static final int constructor = 0xbdfb0426;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneCall extends InputPrivacyKey {
        public static final int constructor = 0xfabadc5f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyForwards extends InputPrivacyKey {
        public static final int constructor = 0xa4dd4c08;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneNumber extends InputPrivacyKey {
        public static final int constructor = 0x352dafa;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyAddedByPhone extends InputPrivacyKey {
        public static final int constructor = 0xd1219bdd;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyVoiceMessages extends InputPrivacyKey {
        public static final int constructor = 0xaee69d68;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyAbout extends InputPrivacyKey {
        public static final int constructor = 0x3823cc40;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyBirthday extends InputPrivacyKey {
        public static final int constructor = 0xd65a11cc;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyStarGiftsAutoSave extends InputPrivacyKey {
        public static final int constructor = 0xe1732341;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyNoPaidMessages extends InputPrivacyKey {
        public static final int constructor = 0xbdc597b4;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeySavedMusic extends InputPrivacyKey {
        public static final int constructor = 0x4dbe9226;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyProfilePhoto extends InputPrivacyKey {
        public static final int constructor = 0x5719bacc;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputPrivacyKeyPhoneP2P extends InputPrivacyKey {
        public static final int constructor = 0xdb9e70d2;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class messages_ExportedChatInvite extends TLObject {

        public ExportedChatInvite invite;
        public ArrayList<User> users = new ArrayList<>();

        public static messages_ExportedChatInvite TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_ExportedChatInvite result = null;
            switch (constructor) {
                case 0x222600ef:
                    result = new TL_messages_exportedChatInviteReplaced();
                    break;
                case 0x1871be50:
                    result = new TL_messages_exportedChatInvite();
                    break;
            }
            return TLdeserialize(messages_ExportedChatInvite.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_exportedChatInviteReplaced extends messages_ExportedChatInvite {
        public static final int constructor = 0x222600ef;

        public ExportedChatInvite new_invite;

        public void readParams(InputSerializedData stream, boolean exception) {
            invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            new_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            invite.serializeToStream(stream);
            new_invite.serializeToStream(stream);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_exportedChatInvite extends messages_ExportedChatInvite {
        public static final int constructor = 0x1871be50;

        public void readParams(InputSerializedData stream, boolean exception) {
            invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            invite.serializeToStream(stream);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class InputTheme extends TLObject {

        public static InputTheme TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputTheme result = null;
            switch (constructor) {
                case 0xf5890df1:
                    result = new TL_inputThemeSlug();
                    break;
                case 0x3c5693e9:
                    result = new TL_inputTheme();
                    break;
            }
            return TLdeserialize(InputTheme.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputThemeSlug extends InputTheme {
        public static final int constructor = 0xf5890df1;

        public String slug;

        public void readParams(InputSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static class TL_inputTheme extends InputTheme {
        public static final int constructor = 0x3c5693e9;

        public long id;
        public long access_hash;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static abstract class photos_Photos extends TLObject {
        public ArrayList<Photo> photos = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();
        public int count;

        public static photos_Photos TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            photos_Photos result = null;
            switch (constructor) {
                case 0x8dca6aa5:
                    result = new TL_photos_photos();
                    break;
                case 0x15051f54:
                    result = new TL_photos_photosSlice();
                    break;
            }
            return TLdeserialize(photos_Photos.class, result, stream, constructor, exception);
        }
    }

    public static class TL_photos_photos extends photos_Photos {
        public static final int constructor = 0x8dca6aa5;

        public void readParams(InputSerializedData stream, boolean exception) {
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_photos_photosSlice extends photos_Photos {
        public static final int constructor = 0x15051f54;

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class ChatFull extends TLObject {
        public long id;
        public ChatParticipants participants;
        public Photo chat_photo;
        public PeerNotifySettings notify_settings;
        public TL_chatInviteExported exported_invite;
        public ArrayList<TL_bots.BotInfo> bot_info = new ArrayList<>();
        public int flags;
        public boolean can_view_participants;
        public boolean can_set_username;
        public boolean has_scheduled;
        public String about;
        public int participants_count;
        public int admins_count;
        public int read_inbox_max_id;
        public int read_outbox_max_id;
        public int unread_count;
        public long migrated_from_chat_id;
        public int migrated_from_max_id;
        public int pinned_msg_id;
        public int kicked_count;
        public int unread_important_count;
        public int folder_id;
        public boolean can_set_stickers;
        public boolean hidden_prehistory;
        public boolean can_view_stats;
        public boolean can_set_location;
        public boolean blocked;
        public int banned_count;
        public int online_count;
        public StickerSet stickerset;
        public int available_min_id;
        public int call_msg_id;
        public long linked_chat_id;
        public ChannelLocation location;
        public int slowmode_seconds;
        public int slowmode_next_send_date;
        public int stats_dc;
        public int pts;
        public InputGroupCall call;
        public int ttl_period;
        public ArrayList<String> pending_suggestions = new ArrayList<>();
        public Peer groupcall_default_join_as;
        public ArrayList<Long> recent_requesters = new ArrayList<>();
        public String theme_emoticon;
        public int requests_pending;
        public Peer default_send_as;
        public ArrayList<String> available_reactions_legacy = new ArrayList<>();
        public int flags2;
        public boolean can_delete_channel;
        public boolean antispam;
        public boolean participants_hidden;
        public boolean translations_disabled;
        public boolean stories_pinned_available;
        public boolean view_forum_as_messages;
        public boolean restricted_sponsored;
        public ChatReactions available_reactions;
        public int reactions_limit;
        public TL_stories.PeerStories stories;
        public WallPaper wallpaper;
        public int boosts_applied;
        public int boosts_unrestrict;
        public StickerSet emojiset;
        public boolean can_view_revenue;
        public boolean can_view_stars_revenue;
        public boolean paid_media_allowed;
        public boolean paid_reactions_available;
        public boolean stargifts_available;
        public boolean paid_messages_available;
        public TL_bots.botVerification bot_verification;
        public int stargifts_count;
        public long send_paid_messages_stars;
        public ProfileTab main_tab;
        public long inviterId; //custom
        public int invitesCount; //custom

        public static ChatFull TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            ChatFull result = null;
            switch (constructor) {
                case TL_chatFull.constructor:
                    result = new TL_chatFull();
                    break;
                case TL_channelFull.constructor:
                    result = new TL_channelFull();
                    break;
                case TL_channelFull_layer212.constructor:
                    result = new TL_channelFull_layer212();
                    break;
                case TL_channelFull_layer204.constructor:
                    result = new TL_channelFull_layer204();
                    break;
                case TL_channelFull_layer197.constructor:
                    result = new TL_channelFull_layer197();
                    break;
                case TL_channelFull_layer195.constructor:
                    result = new TL_channelFull_layer195();
                    break;
                case TL_chatFull_layer177.constructor:
                    result = new TL_chatFull_layer177();
                    break;
                case TL_channelFull_layer177.constructor:
                    result = new TL_channelFull_layer177();
                    break;
                case TL_channelFull_layer176.constructor:
                    result = new TL_channelFull_layer176();
                    break;
                case TL_channelFull_layer167.constructor:
                    result = new TL_channelFull_layer167();
                    break;
                case TL_channelFull_layer173.constructor:
                    result = new TL_channelFull_layer173();
                    break;
                case TL_channelFull_layer162.constructor:
                    result = new TL_channelFull_layer162();
                    break;
                case TL_chatFull_layer144.constructor:
                    result = new TL_chatFull_layer144();
                    break;
                case TL_channelFull_layer144.constructor:
                    result = new TL_channelFull_layer144();
                    break;
                case TL_channelFull_layer139.constructor:
                    result = new TL_channelFull_layer139();
                    break;
                case TL_channelFull_layer135.constructor:
                    result = new TL_channelFull_layer135();
                    break;
                case TL_channelFull_layer134.constructor:
                    result = new TL_channelFull_layer134();
                    break;
                case TL_channelFull_layer98.constructor:
                    result = new TL_channelFull_layer98();
                    break;
                case TL_channelFull_layer99.constructor:
                    result = new TL_channelFull_layer99();
                    break;
                case TL_chatFull_layer87.constructor:
                    result = new TL_chatFull_layer87();
                    break;
                case TL_channelFull_layer122.constructor:
                    result = new TL_channelFull_layer122();
                    break;
                case TL_channelFull_layer121.constructor:
                    result = new TL_channelFull_layer121();
                    break;
                case TL_channelFull_layer110.constructor:
                    result = new TL_channelFull_layer110();
                    break;
                case TL_channelFull_layer103.constructor:
                    result = new TL_channelFull_layer103();
                    break;
                case TL_channelFull_layer101.constructor:
                    result = new TL_channelFull_layer101();
                    break;
                case TL_channelFull_layer71.constructor:
                    result = new TL_channelFull_layer71();
                    break;
                case TL_channelFull_layer72.constructor:
                    result = new TL_channelFull_layer72();
                    break;
                case TL_channelFull_layer52.constructor:
                    result = new TL_channelFull_layer52();
                    break;
                case TL_channelFull_layer67.constructor:
                    result = new TL_channelFull_layer67();
                    break;
                case TL_channelFull_layer48.constructor:
                    result = new TL_channelFull_layer48();
                    break;
                case TL_chatFull_layer122.constructor:
                    result = new TL_chatFull_layer122();
                    break;
                case TL_chatFull_layer123.constructor:
                    result = new TL_chatFull_layer123();
                    break;
                case TL_channelFull_layer123.constructor:
                    result = new TL_channelFull_layer123();
                    break;
                case TL_chatFull_layer124.constructor:
                    result = new TL_chatFull_layer124();
                    break;
                case TL_channelFull_layer124.constructor:
                    result = new TL_channelFull_layer124();
                    break;
                case TL_chatFull_layer131.constructor:
                    result = new TL_chatFull_layer131();
                    break;
                case TL_channelFull_layer131.constructor:
                    result = new TL_channelFull_layer131();
                    break;
                case TL_chatFull_layer132.constructor:
                    result = new TL_chatFull_layer132();
                    break;
                case TL_channelFull_layer132.constructor:
                    result = new TL_channelFull_layer132();
                    break;
                case TL_chatFull_layer133.constructor:
                    result = new TL_chatFull_layer133();
                    break;
                case TL_channelFull_layer133.constructor:
                    result = new TL_channelFull_layer133();
                    break;
                case TL_chatFull_layer121.constructor:
                    result = new TL_chatFull_layer121();
                    break;
                case TL_chatFull_layer98.constructor:
                    result = new TL_chatFull_layer98();
                    break;
                case TL_chatFull_layer92.constructor:
                    result = new TL_chatFull_layer92();
                    break;
                case TL_chatFull_layer135.constructor:
                    result = new TL_chatFull_layer135();
                    break;
                case TL_channelFull_old.constructor:
                    result = new TL_channelFull_old();
                    break;
                case TL_channelFull_layer70.constructor:
                    result = new TL_channelFull_layer70();
                    break;
                case TL_channelFull_layer89.constructor:
                    result = new TL_channelFull_layer89();
                    break;
            }
            result = TLdeserialize(ChatFull.class, result, stream, constructor, exception);
            if (result != null && result.available_reactions == null) {
                if (!result.available_reactions_legacy.isEmpty()) {
                    TL_chatReactionsSome someReactions = new TL_chatReactionsSome();
                    for (int i = 0; i < result.available_reactions_legacy.size(); i++) {
                        TL_reactionEmoji reaction = new TL_reactionEmoji();
                        reaction.emoticon = result.available_reactions_legacy.get(i);
                        someReactions.reactions.add(reaction);
                    }
                    result.available_reactions = someReactions;
                } else {
                    result.available_reactions = new TL_chatReactionsNone();
                }
            }
            return result;
        }
    }

    public static class TL_channelFull_layer110 extends ChatFull {
        public static final int constructor = 0x2d895c74;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_chatFull_layer124 extends TL_chatFull {
        public static final int constructor = 0xf06c4018;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
        }
    }

    public static class TL_channelFull_layer124 extends TL_channelFull {
        public static final int constructor = 0x2548c037;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
        }
    }

    public static class TL_chatFull_layer123 extends TL_chatFull {
        public static final int constructor = 0xf3474af6;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer123 extends TL_channelFull {
        public static final int constructor = 0x7a7de4f7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer131 extends TL_chatFull {
        public static final int constructor = 0x8a1e2983;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer132 extends TL_chatFull {
        public static final int constructor = 0x49a0a5d9;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_channelFull_layer132 extends TL_channelFull {
        public static final int constructor = 0x2f532f3c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull_layer133 extends ChatFull {
        public static final int constructor = 0x4dbdc099;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull_layer135 extends ChatFull {
        public static final int constructor = 0x46a6ffb4;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
        }
    }

    public static class TL_channelFull_layer135 extends ChatFull {
        public static final int constructor = 0x56662e2e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer134 extends ChatFull {
        public static final int constructor = 0x59cff963;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
        }
    }

    public static class TL_channelFull_layer133 extends ChatFull {
        public static final int constructor = 0xe9b27a17;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TLRPC.TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
        }
    }

    public static class TL_chatFull extends ChatFull {
        public static final int constructor = 0x2633421b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 262144) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1048576) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            flags = setFlag(flags, FLAG_16, theme_emoticon != null);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 262144) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags & 1048576) != 0) {
                stream.writeInt32(reactions_limit);
            }
        }
    }

    public static class TL_chatFull_layer177 extends TL_chatFull {
        public static final int constructor = 0xc9d31138;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 262144) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 262144) != 0) {
                available_reactions.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer144 extends ChatFull {
        public static final int constructor = 0xd18ee226;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8192) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 65536) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 131072) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 131072) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 262144) != 0) {
                available_reactions_legacy = Vector.deserializeString(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            if ((flags & 8192) != 0) {
                exported_invite.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 32768) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 65536) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 131072) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 262144) != 0) {
                Vector.serializeString(stream, available_reactions_legacy);
            }
        }
    }

    public static class TL_channelFull extends ChatFull {
        public static final int constructor = 0xe4e0b29d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            paid_media_allowed = (flags2 & 16384) != 0;
            paid_reactions_available = (flags2 & 65536) != 0;
            stargifts_available = (flags2 & 524288) != 0;
            paid_messages_available = (flags2 & 1048576) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 8192) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification = TL_bots.botVerification.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 262144) != 0) {
                stargifts_count = stream.readInt32(exception);
            }
            if (hasFlag(flags2, FLAG_21)) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
            if (hasFlag(flags2, FLAG_22)) {
                main_tab = ProfileTab.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = paid_media_allowed ? (flags2 | 16384) : (flags2 &~ 16384);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            flags2 = paid_reactions_available ? (flags2 | 65536) : (flags2 &~ 65536);
            flags2 = stargifts_available ? (flags2 | 524288) : (flags2 &~ 524288);
            flags2 = paid_messages_available ? (flags2 | 1048576) : (flags2 &~ 1048576);
            flags2 = setFlag(flags2, FLAG_22, main_tab != null);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 8192) != 0) {
                stream.writeInt32(reactions_limit);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification.serializeToStream(stream);
            }
            if ((flags2 & 262144) != 0) {
                stream.writeInt32(stargifts_count);
            }
            if (hasFlag(flags2, FLAG_21)) {
                stream.writeInt64(send_paid_messages_stars);
            }
            if (hasFlag(flags2, FLAG_22)) {
                // Vector.serialize(stream, tabs_order);
                main_tab.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer212 extends TL_channelFull {
        public static final int constructor = 0xe07429de;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            paid_media_allowed = (flags2 & 16384) != 0;
            paid_reactions_available = (flags2 & 65536) != 0;
            stargifts_available = (flags2 & 524288) != 0;
            paid_messages_available = (flags2 & 1048576) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 8192) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification = TL_bots.botVerification.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 262144) != 0) {
                stargifts_count = stream.readInt32(exception);
            }
            if (hasFlag(flags2, FLAG_21)) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = paid_media_allowed ? (flags2 | 16384) : (flags2 &~ 16384);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            flags2 = paid_reactions_available ? (flags2 | 65536) : (flags2 &~ 65536);
            flags2 = stargifts_available ? (flags2 | 524288) : (flags2 &~ 524288);
            flags2 = paid_messages_available ? (flags2 | 1048576) : (flags2 &~ 1048576);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 8192) != 0) {
                stream.writeInt32(reactions_limit);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification.serializeToStream(stream);
            }
            if ((flags2 & 262144) != 0) {
                stream.writeInt32(stargifts_count);
            }
            if (hasFlag(flags2, FLAG_21)) {
                stream.writeInt64(send_paid_messages_stars);
            }
        }
    }

    public static class TL_channelFull_layer204 extends ChatFull {
        public static final int constructor = 0x52D6806B;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            paid_media_allowed = (flags2 & 16384) != 0;
            paid_reactions_available = (flags2 & 65536) != 0;
            stargifts_available = (flags2 & 524288) != 0;
            paid_messages_available = (flags2 & 1048576) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 8192) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification = TL_bots.botVerification.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 262144) != 0) {
                stargifts_count = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = paid_media_allowed ? (flags2 | 16384) : (flags2 &~ 16384);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            flags2 = paid_reactions_available ? (flags2 | 65536) : (flags2 &~ 65536);
            flags2 = stargifts_available ? (flags2 | 524288) : (flags2 &~ 524288);
            flags2 = paid_messages_available ? (flags2 | 1048576) : (flags2 &~ 1048576);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 8192) != 0) {
                stream.writeInt32(reactions_limit);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification.serializeToStream(stream);
            }
            if ((flags2 & 262144) != 0) {
                stream.writeInt32(stargifts_count);
            }
        }
    }

    public static class TL_channelFull_layer197 extends TL_channelFull {
        public static final int constructor = 0x9ff3b858;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            paid_media_allowed = (flags2 & 16384) != 0;
            paid_reactions_available = (flags2 & 65536) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 8192) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification = TL_bots.botVerification.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = paid_media_allowed ? (flags2 | 16384) : (flags2 &~ 16384);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            flags2 = paid_reactions_available ? (flags2 | 65536) : (flags2 &~ 65536);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 8192) != 0) {
                stream.writeInt32(reactions_limit);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
            if ((flags2 & 131072) != 0) {
                bot_verification.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer195 extends TL_channelFull {
        public static final int constructor = 0xbbab348d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            paid_media_allowed = (flags2 & 16384) != 0;
            paid_reactions_available = (flags2 & 65536) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 8192) != 0) {
                reactions_limit = stream.readInt32(exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = paid_media_allowed ? (flags2 | 16384) : (flags2 &~ 16384);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            flags2 = paid_reactions_available ? (flags2 | 65536) : (flags2 &~ 65536);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 8192) != 0) {
                stream.writeInt32(reactions_limit);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer177 extends TL_channelFull {
        public static final int constructor = 0x44c054a7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            restricted_sponsored = (flags2 & 2048) != 0;
            can_view_revenue = (flags2 & 4096) != 0;
            can_view_stars_revenue = (flags2 & 32768) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            flags2 = restricted_sponsored ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = can_view_revenue ? (flags2 | 4096) : (flags2 &~ 4096);
            flags2 = can_view_stars_revenue ? (flags2 | 32768) : (flags2 &~ 32768);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer176 extends TL_channelFull {
        public static final int constructor = 0x680b773c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 256) != 0) {
                boosts_applied = stream.readInt32(exception);
            }
            if ((flags2 & 512) != 0) {
                boosts_unrestrict = stream.readInt32(exception);
            }
            if ((flags2 & 1024) != 0) {
                emojiset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                stream.writeInt32(boosts_applied);
            }
            if ((flags2 & 512) != 0) {
                stream.writeInt32(boosts_unrestrict);
            }
            if ((flags2 & 1024) != 0) {
                emojiset.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer173 extends ChatFull {
        public static final int constructor = 0xf2bcb6f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 128) != 0) {
                wallpaper = WallPaper.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
            if ((flags2 & 128) != 0) {
                wallpaper.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer167 extends TL_channelFull {
        public static final int constructor = 0x723027bd;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            stories_pinned_available = (flags2 & 32) != 0;
            view_forum_as_messages = (flags2 & 64) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 16) != 0) {
                stories = TL_stories.PeerStories.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_pinned_available ? (flags2 | 32) : (flags2 &~ 32);
            flags2 = view_forum_as_messages ? (flags2 | 64) : (flags2 &~ 64);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
            if ((flags2 & 16) != 0) {
                stories.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer162 extends TL_channelFull {
        public static final int constructor = 0xf2355507;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            antispam = (flags2 & 2) != 0;
            participants_hidden = (flags2 & 4) != 0;
            translations_disabled = (flags2 & 8) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions = ChatReactions.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            flags2 = antispam ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = participants_hidden ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = translations_disabled ? (flags2 | 8) : (flags2 &~ 8);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer144 extends ChatFull {
        public static final int constructor = 0xea68a619;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            flags2 = stream.readInt32(exception);
            can_delete_channel = (flags2 & 1) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions_legacy = Vector.deserializeString(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            flags2 = can_delete_channel ? (flags2 | 1) : (flags2 &~ 1);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                Vector.serializeString(stream, available_reactions_legacy);
            }
        }
    }

    public static class TL_channelFull_layer139 extends ChatFull {
        public static final int constructor = 0xe13c3d20;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 134217728) != 0) {
                theme_emoticon = stream.readString(exception);
            }
            if ((flags & 268435456) != 0) {
                requests_pending = stream.readInt32(exception);
            }
            if ((flags & 268435456) != 0) {
                recent_requesters = Vector.deserializeLong(stream, exception);
            }
            if ((flags & 536870912) != 0) {
                default_send_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1073741824) != 0) {
                available_reactions_legacy = Vector.deserializeString(stream, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
            if ((flags & 134217728) != 0) {
                stream.writeString(theme_emoticon);
            }
            if ((flags & 268435456) != 0) {
                stream.writeInt32(requests_pending);
            }
            if ((flags & 268435456) != 0) {
                Vector.serializeLong(stream, recent_requesters);
            }
            if ((flags & 536870912) != 0) {
                default_send_as.serializeToStream(stream);
            }
            if ((flags & 1073741824) != 0) {
                Vector.serializeString(stream, available_reactions_legacy);
            }
        }
    }

    public static class TL_channelFull_layer131 extends TL_channelFull {
        public static final int constructor = 0x548c3f93;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt64(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 8388608) != 0) {
                exported_invite = (TL_chatInviteExported) ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt64(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt64(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16777216) != 0) {
                ttl_period = stream.readInt32(exception);
            }
            if ((flags & 33554432) != 0) {
                pending_suggestions = Vector.deserializeString(stream, exception);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as = Peer.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if ((flags & 8388608) != 0) {
                exported_invite.serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt64(migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt64(linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
            if ((flags & 16777216) != 0) {
                stream.writeInt32(ttl_period);
            }
            if ((flags & 33554432) != 0) {
                Vector.serializeString(stream, pending_suggestions);
            }
            if ((flags & 67108864) != 0) {
                groupcall_default_join_as.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer122 extends TL_channelFull {
        public static final int constructor = 0xef3a6acd;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
            if ((flags & 2097152) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            if (exported_invite != null) {
                exported_invite.serializeToStream(stream);
            } else {
                new TLRPC.TL_chatInviteEmpty_layer122().serializeToStream(stream);
            }
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
            if ((flags & 2097152) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_channelFull_layer121 extends TL_channelFull {
        public static final int constructor = 0xf0e6672a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_set_location = (flags & 65536) != 0;
            has_scheduled = (flags & 524288) != 0;
            can_view_stats = (flags & 1048576) != 0;
            blocked = (flags & 4194304) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 131072) != 0) {
                slowmode_seconds = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                slowmode_next_send_date = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                stats_dc = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            flags = has_scheduled ? (flags | 524288) : (flags &~ 524288);
            flags = can_view_stats ? (flags | 1048576) : (flags &~ 1048576);
            flags = blocked ? (flags | 4194304) : (flags &~ 4194304);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            if ((flags & 131072) != 0) {
                stream.writeInt32(slowmode_seconds);
            }
            if ((flags & 262144) != 0) {
                stream.writeInt32(slowmode_next_send_date);
            }
            if ((flags & 4096) != 0) {
                stream.writeInt32(stats_dc);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer103 extends TL_channelFull {
        public static final int constructor = 0x10916653;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            can_set_location = (flags & 65536) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 16384) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            if ((flags & 32768) != 0) {
                location = ChannelLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            flags = can_set_location ? (flags | 65536) : (flags &~ 65536);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            if ((flags & 32768) != 0) {
                location.serializeToStream(stream);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer101 extends TL_channelFull {
        public static final int constructor = 0x9882e516;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                linked_chat_id = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32((int) linked_chat_id);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer99 extends TL_channelFull {
        public static final int constructor = 0x3648977;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            pts = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            stream.writeInt32(pts);
        }
    }

    public static class TL_channelFull_layer98 extends TL_channelFull {
        public static final int constructor = 0x1c87a71a;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            can_view_stats = (flags & 4096) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            if ((flags & 8192) != 0) {
                online_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            flags = can_view_stats ? (flags | 4096) : (flags &~ 4096);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            if ((flags & 8192) != 0) {
                stream.writeInt32(online_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
        }
    }

    public static class TL_chatFull_layer87 extends TL_chatFull {
        public static final int constructor = 0x2e02a614;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            participants.serializeToStream(stream);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
        }
    }

    public static class TL_channelFull_layer89 extends TL_channelFull {
        public static final int constructor = 0xcbb62890;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                call_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(call_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer72 extends TL_channelFull {
        public static final int constructor = 0x76af5481;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            hidden_prehistory = (flags & 1024) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 512) != 0) {
                available_min_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            flags = hidden_prehistory ? (flags | 1024) : (flags &~ 1024);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
            if ((flags & 512) != 0) {
                stream.writeInt32(available_min_id);
            }
        }
    }

    public static class TL_channelFull_layer71 extends TL_channelFull {
        public static final int constructor = 0x17f45fcf;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            can_set_stickers = (flags & 128) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                stickerset = StickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            flags = can_set_stickers ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 256) != 0) {
                stickerset.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer122 extends TL_chatFull {
        public static final int constructor = 0xdc8c181;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
            if ((flags & 4096) != 0) {
                call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
            if ((flags & 4096) != 0) {
                call.serializeToStream(stream);
            }
        }
    }

    public static class TL_chatFull_layer121 extends TL_chatFull {
        public static final int constructor = 0x1b7c9db3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            has_scheduled = (flags & 256) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
            if ((flags & 2048) != 0) {
                folder_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            flags = has_scheduled ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(folder_id);
            }
        }
    }

    public static class TL_chatFull_layer98 extends TL_chatFull {
        public static final int constructor = 0x22a235da;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_set_username = (flags & 128) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_set_username ? (flags | 128) : (flags &~ 128);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_chatFull_layer92 extends TL_chatFull {
        public static final int constructor = 0xedd2a791;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt32(exception);
            participants = ChatParticipants.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            if ((flags & 8) != 0) {
                bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            }
            if ((flags & 64) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            participants.serializeToStream(stream);
            if ((flags & 4) != 0) {
                chat_photo.serializeToStream(stream);
            }
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            if ((flags & 8) != 0) {
                Vector.serialize(stream, bot_info);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer70 extends TL_channelFull {
        public static final int constructor = 0x95cb5f57;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                banned_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(banned_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer52 extends TL_channelFull {
        public static final int constructor = 0x97bee562;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer67 extends TL_channelFull {
        public static final int constructor = 0xc3d5512f;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            can_set_username = (flags & 64) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            read_outbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
            if ((flags & 32) != 0) {
                pinned_msg_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            flags = can_set_username ? (flags | 64) : (flags &~ 64);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(read_outbox_max_id);
            stream.writeInt32(unread_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
            if ((flags & 32) != 0) {
                stream.writeInt32(pinned_msg_id);
            }
        }
    }

    public static class TL_channelFull_layer48 extends TL_channelFull {
        public static final int constructor = 0x9e341ddf;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
            bot_info = Vector.deserialize(stream, TL_bots.BotInfo::TLdeserialize, exception);
            if ((flags & 16) != 0) {
                migrated_from_chat_id = stream.readInt32(exception);
            }
            if ((flags & 16) != 0) {
                migrated_from_max_id = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
            Vector.serialize(stream, bot_info);
            if ((flags & 16) != 0) {
                stream.writeInt32((int) migrated_from_chat_id);
            }
            if ((flags & 16) != 0) {
                stream.writeInt32(migrated_from_max_id);
            }
        }
    }

    public static class TL_channelFull_old extends TL_channelFull {
        public static final int constructor = 0xfab31aa3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            can_view_participants = (flags & 8) != 0;
            id = stream.readInt32(exception);
            about = stream.readString(exception);
            if ((flags & 1) != 0) {
                participants_count = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                admins_count = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                kicked_count = stream.readInt32(exception);
            }
            read_inbox_max_id = stream.readInt32(exception);
            unread_count = stream.readInt32(exception);
            unread_important_count = stream.readInt32(exception);
            chat_photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            notify_settings = PeerNotifySettings.TLdeserialize(stream, stream.readInt32(exception), exception);
            ExportedChatInvite invite = ExportedChatInvite.TLdeserialize(stream, stream.readInt32(exception), exception);
            if (invite instanceof TL_chatInviteExported) {
                exported_invite = (TL_chatInviteExported) invite;
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = can_view_participants ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            stream.writeString(about);
            if ((flags & 1) != 0) {
                stream.writeInt32(participants_count);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(admins_count);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(kicked_count);
            }
            stream.writeInt32(read_inbox_max_id);
            stream.writeInt32(unread_count);
            stream.writeInt32(unread_important_count);
            chat_photo.serializeToStream(stream);
            notify_settings.serializeToStream(stream);
            exported_invite.serializeToStream(stream);
        }
    }

    public static class TL_inputPeerNotifySettings extends TLObject {
        public static final int constructor = 0xcacb6ae2;

        public int flags;
        public boolean show_previews;
        public boolean silent;
        public int mute_until;
        public NotificationSound sound;
        public boolean stories_muted;
        public boolean stories_hide_sender;
        public NotificationSound stories_sound;

        public static TL_inputPeerNotifySettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_inputPeerNotifySettings result = TL_inputPeerNotifySettings.constructor != constructor ? null : new TL_inputPeerNotifySettings();
            return TLdeserialize(TL_inputPeerNotifySettings.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                show_previews = stream.readBool(exception);
            }
            if ((flags & 2) != 0) {
                silent = stream.readBool(exception);
            }
            if ((flags & 4) != 0) {
                mute_until = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                stories_muted = stream.readBool(exception);
            }
            if ((flags & 128) != 0) {
                stories_hide_sender = stream.readBool(exception);
            }
            if ((flags & 256) != 0) {
                stories_sound = NotificationSound.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(show_previews);
            }
            if ((flags & 2) != 0) {
                stream.writeBool(silent);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(mute_until);
            }
            if ((flags & 8) != 0) {
                sound.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                stream.writeBool(stories_muted);
            }
            if ((flags & 128) != 0) {
                stream.writeBool(stories_hide_sender);
            }
            if ((flags & 256) != 0) {
                stories_sound.serializeToStream(stream);
            }
        }
    }

    public static abstract class auth_LoginToken extends TLObject {

        public static auth_LoginToken TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            auth_LoginToken result = null;
            switch (constructor) {
                case 0x629f1980:
                    result = new TL_auth_loginToken();
                    break;
                case 0x68e9916:
                    result = new TL_auth_loginTokenMigrateTo();
                    break;
                case 0x390d5c5e:
                    result = new TL_auth_loginTokenSuccess();
                    break;
            }
            return TLdeserialize(auth_LoginToken.class, result, stream, constructor, exception);
        }
    }

    public static class TL_auth_loginToken extends auth_LoginToken {
        public static final int constructor = 0x629f1980;

        public int expires;
        public byte[] token;

        public void readParams(InputSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
            token = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
            stream.writeByteArray(token);
        }
    }

    public static class TL_auth_loginTokenMigrateTo extends auth_LoginToken {
        public static final int constructor = 0x68e9916;

        public int dc_id;
        public byte[] token;

        public void readParams(InputSerializedData stream, boolean exception) {
            dc_id = stream.readInt32(exception);
            token = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(dc_id);
            stream.writeByteArray(token);
        }
    }

    public static class TL_auth_loginTokenSuccess extends auth_LoginToken {
        public static final int constructor = 0x390d5c5e;

        public auth_Authorization authorization;

        public void readParams(InputSerializedData stream, boolean exception) {
            authorization = auth_Authorization.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            authorization.serializeToStream(stream);
        }
    }

        public static class TL_codeSettings extends TLObject {
        public static final int constructor = 0xad253d78;

        public int flags;
        public boolean allow_flashcall;
        public boolean current_number;
        public boolean allow_app_hash;
        public boolean allow_missed_call;
        public boolean allow_firebase;
        public boolean unknown_number;
        public ArrayList<byte[]> logout_tokens = new ArrayList<>();
        public String token;
        public boolean app_sandbox;

        public static TL_codeSettings TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_codeSettings result = TL_codeSettings.constructor != constructor ? null : new TL_codeSettings();
            return TLdeserialize(TL_codeSettings.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            allow_flashcall = (flags & 1) != 0;
            current_number = (flags & 2) != 0;
            allow_app_hash = (flags & 16) != 0;
            allow_missed_call = (flags & 32) != 0;
            allow_firebase = (flags & 128) != 0;
            unknown_number = (flags & 512) != 0;
            if ((flags & 64) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    byte[] object = stream.readByteArray(exception);
                    if (object == null) {
                        return;
                    }
                    logout_tokens.add(object);
                }
            }
            if ((flags & 256) != 0) {
                token = stream.readString(exception);
                app_sandbox = stream.readBool(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = allow_flashcall ? (flags | 1) : (flags &~ 1);
            flags = current_number ? (flags | 2) : (flags &~ 2);
            flags = allow_app_hash ? (flags | 16) : (flags &~ 16);
            flags = allow_missed_call ? (flags | 32) : (flags &~ 32);
            flags = allow_firebase ? (flags | 128) : (flags &~ 128);
            flags = unknown_number ? (flags | 512) : (flags &~ 512);
            stream.writeInt32(flags);
            if ((flags & 64) != 0) {
                stream.writeInt32(0x1cb5c415);
                int count = logout_tokens.size();
                stream.writeInt32(count);
                for (int a = 0; a < count; a++) {
                    stream.writeByteArray(logout_tokens.get(a));
                }
            }
            if ((flags & 256) != 0) {
                stream.writeString(token);
                stream.writeBool(app_sandbox);
            }
        }
    }

    public static abstract class BaseTheme extends TLObject {

        public static BaseTheme TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            BaseTheme result = null;
            switch (constructor) {
                case 0x5b11125a:
                    result = new TL_baseThemeArctic();
                    break;
                case 0xb7b31ea8:
                    result = new TL_baseThemeNight();
                    break;
                case 0xc3a12462:
                    result = new TL_baseThemeClassic();
                    break;
                case 0x6d5f77ee:
                    result = new TL_baseThemeTinted();
                    break;
                case 0xfbd81688:
                    result = new TL_baseThemeDay();
                    break;
            }
            return TLdeserialize(BaseTheme.class, result, stream, constructor, exception);
        }
    }

    public static class TL_baseThemeArctic extends BaseTheme {
        public static final int constructor = 0x5b11125a;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeNight extends BaseTheme {
        public static final int constructor = 0xb7b31ea8;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeClassic extends BaseTheme {
        public static final int constructor = 0xc3a12462;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeTinted extends BaseTheme {
        public static final int constructor = 0x6d5f77ee;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_baseThemeDay extends BaseTheme {
        public static final int constructor = 0xfbd81688;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_null extends TLObject {
        public static final int constructor = 0x56730bcc;

        public static TL_null TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_null result = TL_null.constructor != constructor ? null : new TL_null();
            return TLdeserialize(TL_null.class, result, stream, constructor, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static abstract class Page extends TLObject {

        public int flags;
        public boolean part;
        public boolean rtl;
        public String url;
        public ArrayList<PageBlock> blocks = new ArrayList<>();
        public ArrayList<Photo> photos = new ArrayList<>();
        public ArrayList<Document> documents = new ArrayList<>();
        public boolean v2;
        public int views;
        public boolean web; //custom

        public static Page TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            Page result = null;
            switch (constructor) {
                case 0x8dee6c44:
                    result = new TL_pagePart_layer67();
                    break;
                case 0x8e3f9ebe:
                    result = new TL_pagePart_layer82();
                    break;
                case 0xd7a19d69:
                    result = new TL_pageFull_layer67();
                    break;
                case 0x556ec7aa:
                    result = new TL_pageFull_layer82();
                    break;
                case TL_page_layer88.constructor:
                    result = new TL_page_layer88();
                    break;
                case 0xae891bec:
                    result = new TL_page_layer110();
                    break;
                case 0x98657f0d:
                    result = new TL_page();
                    break;
            }
            return TLdeserialize(Page.class, result, stream, constructor, exception);
        }
    }

    public static class TL_pagePart_layer67 extends TL_pagePart_layer82 {
        public static final int constructor = 0x8dee6c44;

        public void readParams(InputSerializedData stream, boolean exception) {
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_pagePart_layer82 extends Page {
        public static final int constructor = 0x8e3f9ebe;

        public void readParams(InputSerializedData stream, boolean exception) {
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_pageFull_layer67 extends TL_page {
        public static final int constructor = 0xd7a19d69;

        public void readParams(InputSerializedData stream, boolean exception) {
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_pageFull_layer82 extends TL_page {
        public static final int constructor = 0x556ec7aa;

        public void readParams(InputSerializedData stream, boolean exception) {
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_page_layer110 extends TL_page {
        public static final int constructor = 0xae891bec;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            part = (flags & 1) != 0;
            rtl = (flags & 2) != 0;
            url = stream.readString(exception);
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = part ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            stream.writeString(url);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_page extends Page {
        public static final int constructor = 0x98657f0d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            part = (flags & 1) != 0;
            rtl = (flags & 2) != 0;
            v2 = (flags & 4) != 0;
            url = stream.readString(exception);
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
            if ((flags & 8) != 0) {
                views = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = part ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 2) : (flags &~ 2);
            flags = v2 ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            stream.writeString(url);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
            if ((flags & 8) != 0) {
                stream.writeInt32(views);
            }
        }
    }

    public static class TL_page_layer88 extends TL_page {
        public static final int constructor = 0xf199a0a8;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            part = (flags & 1) != 0;
            rtl = (flags & 2) != 0;
            v2 = (flags & 4) != 0;
            blocks = Vector.deserialize(stream, PageBlock::TLdeserialize, exception);
            photos = Vector.deserialize(stream, Photo::TLdeserialize, exception);
            documents = Vector.deserialize(stream, Document::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = part ? (flags | 1) : (flags &~ 1);
            flags = rtl ? (flags | 2) : (flags &~ 2);
            flags = v2 ? (flags | 4) : (flags &~ 4);
            stream.writeInt32(flags);
            Vector.serialize(stream, blocks);
            Vector.serialize(stream, photos);
            Vector.serialize(stream, documents);
        }
    }

    public static class TL_topPeerCategoryPeers extends TLObject {
        public static final int constructor = 0xfb834291;

        public TopPeerCategory category;
        public int count;
        public ArrayList<TL_topPeer> peers = new ArrayList<>();

        public static TL_topPeerCategoryPeers TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_topPeerCategoryPeers result = TL_topPeerCategoryPeers.constructor != constructor ? null : new TL_topPeerCategoryPeers();
            return TLdeserialize(TL_topPeerCategoryPeers.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            category = TopPeerCategory.TLdeserialize(stream, stream.readInt32(exception), exception);
            count = stream.readInt32(exception);
            peers = Vector.deserialize(stream, TL_topPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            category.serializeToStream(stream);
            stream.writeInt32(count);
            Vector.serialize(stream, peers);
        }
    }

    public static abstract class InputUser extends TLObject {

        public long user_id;
        public long access_hash;

        public static InputUser TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputUser result = null;
            switch (constructor) {
                case 0xf7c1b13f:
                    result = new TL_inputUserSelf();
                    break;
                case 0xf21158c6:
                    result = new TL_inputUser();
                    break;
                case 0xb98886cf:
                    result = new TL_inputUserEmpty();
                    break;
                case 0x1da448e2:
                    result = new TL_inputUserFromMessage();
                    break;
                case TL_inputUserFromMessage_layer132.constructor:
                    result = new TL_inputUserFromMessage_layer132();
                    break;
                case TL_inputUser_layer132.constructor:
                    result = new TL_inputUser_layer132();
                    break;
            }
            return TLdeserialize(InputUser.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputUserSelf extends InputUser {
        public static final int constructor = 0xf7c1b13f;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputUser extends InputUser {
        public static final int constructor = 0xf21158c6;

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(user_id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputUserEmpty extends InputUser {
        public static final int constructor = 0xb98886cf;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputUserFromMessage extends InputUser {
        public static final int constructor = 0x1da448e2;

        public InputPeer peer;
        public int msg_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            msg_id = stream.readInt32(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(msg_id);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_inputUser_layer132 extends InputUser {
        public static final int constructor = 0xd8292816;

        public void readParams(InputSerializedData stream, boolean exception) {
            user_id = stream.readInt32(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) user_id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputUserFromMessage_layer132 extends InputUser {
        public static final int constructor = 0x2d117597;

        public InputPeer peer;
        public int msg_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            msg_id = stream.readInt32(exception);
            user_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(msg_id);
            stream.writeInt32((int) user_id);
        }
    }

    public static abstract class KeyboardButton extends TLObject {
        public String text;
        public String url;
        public int flags;
        public boolean same_peer;
        public String query;
        public byte[] data;
        public int button_id;
        public boolean request_write_access;
        public InputUser bot;
        public String fwd_text;
        public boolean quiz;
        public boolean requires_password;
        public long user_id;
        public InputUser inputUser;
        public ArrayList<InlineQueryPeerType> peer_types = new ArrayList<>();

        public static KeyboardButton TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            KeyboardButton result = null;
            switch (constructor) {
                case 0xa0c0505c:
                    result = new TL_keyboardButtonSimpleWebView();
                    break;
                case 0xb16a6c29:
                    result = new TL_keyboardButtonRequestPhone();
                    break;
                case 0x50f41ccf:
                    result = new TL_keyboardButtonGame();
                    break;
                case 0x258aff05:
                    result = new TL_keyboardButtonUrl();
                    break;
                case 0x568a748:
                    result = new TL_keyboardButtonSwitchInline_layer157();
                    break;
                case 0xfc796b3f:
                    result = new TL_keyboardButtonRequestGeoLocation();
                    break;
                case 0x10b78d29:
                    result = new TL_keyboardButtonUrlAuth();
                    break;
                case 0xd02e7fd4:
                    result = new TL_inputKeyboardButtonUrlAuth();
                    break;
                case 0xbbc7515d:
                    result = new TL_keyboardButtonRequestPoll();
                    break;
                case 0xafd93fbb:
                    result = new TL_keyboardButtonBuy();
                    break;
                case 0x683a5e46:
                    result = new TL_keyboardButtonCallback_layer117();
                    break;
                case 0x93b9fbb5:
                    result = new TL_keyboardButtonSwitchInline();
                    break;
                case 0xa2fa4880:
                    result = new TL_keyboardButton();
                    break;
                case 0x35bbdb6b:
                    result = new TL_keyboardButtonCallback();
                    break;
                case 0xe988037b:
                    result = new TL_inputKeyboardButtonUserProfile();
                    break;
                case 0x308660c1:
                    result = new TL_keyboardButtonUserProfile();
                    break;
                case 0x13767230:
                    result = new TL_keyboardButtonWebView();
                    break;
                case TL_keyboardButtonRequestPeer_layer168.constructor:
                    result = new TL_keyboardButtonRequestPeer_layer168();
                    break;
                case TL_keyboardButtonRequestPeer.constructor:
                    result = new TL_keyboardButtonRequestPeer();
                    break;
                case TL_keyboardButtonCopy.constructor:
                    result = new TL_keyboardButtonCopy();
                    break;
            }
            return TLdeserialize(KeyboardButton.class, result, stream, constructor, exception);
        }
    }

    public static class TL_keyboardButtonRequestPhone extends KeyboardButton {
        public static final int constructor = 0xb16a6c29;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonGame extends KeyboardButton {
        public static final int constructor = 0x50f41ccf;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonUrl extends KeyboardButton {
        public static final int constructor = 0x258aff05;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeString(url);
        }
    }

    public static class TL_keyboardButtonSwitchInline_layer157 extends KeyboardButton {
        public static final int constructor = 0x568a748;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            same_peer = (flags & 1) != 0;
            text = stream.readString(exception);
            query = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = same_peer ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            stream.writeString(query);
        }
    }

    public static class TL_keyboardButtonSwitchInline extends KeyboardButton {
        public static final int constructor = 0x93b9fbb5;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            same_peer = (flags & 1) != 0;
            text = stream.readString(exception);
            query = stream.readString(exception);
            if ((flags & 2) != 0) {
                peer_types = Vector.deserialize(stream, InlineQueryPeerType::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = same_peer ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            stream.writeString(query);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, peer_types);
            }
        }
    }

    public static class TL_keyboardButtonRequestGeoLocation extends KeyboardButton {
        public static final int constructor = 0xfc796b3f;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonUrlAuth extends KeyboardButton {
        public static final int constructor = 0x10b78d29;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            text = stream.readString(exception);
            if ((flags & 1) != 0) {
                fwd_text = stream.readString(exception);
            }
            url = stream.readString(exception);
            button_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(text);
            if ((flags & 1) != 0) {
                stream.writeString(fwd_text);
            }
            stream.writeString(url);
            stream.writeInt32(button_id);
        }
    }

    public static class TL_inputKeyboardButtonUrlAuth extends KeyboardButton {
        public static final int constructor = 0xd02e7fd4;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            request_write_access = (flags & 1) != 0;
            text = stream.readString(exception);
            if ((flags & 2) != 0) {
                fwd_text = stream.readString(exception);
            }
            url = stream.readString(exception);
            bot = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = request_write_access ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            if ((flags & 2) != 0) {
                stream.writeString(fwd_text);
            }
            stream.writeString(url);
            bot.serializeToStream(stream);
        }
    }

    public static class TL_keyboardButtonRequestPoll extends KeyboardButton {
        public static final int constructor = 0xbbc7515d;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                quiz = stream.readBool(exception);
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeBool(quiz);
            }
            stream.writeString(text);
        }
    }

    public static class TL_keyboardButtonBuy extends KeyboardButton {
        public static final int constructor = 0xafd93fbb;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_inputKeyboardButtonUserProfile extends KeyboardButton {
        public static final int constructor = 0xe988037b;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            inputUser = InputUser.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            inputUser.serializeToStream(stream);
        }
    }

    public static class TL_keyboardButtonUserProfile extends KeyboardButton {
        public static final int constructor = 0x308660c1;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            user_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeInt64(user_id);
        }
    }

    public static class TL_keyboardButtonCallback extends KeyboardButton {
        public static final int constructor = 0x35bbdb6b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            requires_password = (flags & 1) != 0;
            text = stream.readString(exception);
            data = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = requires_password ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(text);
            stream.writeByteArray(data);
        }
    }

    public static class TL_keyboardButtonCallback_layer117 extends TL_keyboardButtonCallback {
        public static final int constructor = 0x683a5e46;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
            data = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
            stream.writeByteArray(data);
        }
    }

    public static class TL_keyboardButton extends KeyboardButton {
        public static final int constructor = 0xa2fa4880;

        public void readParams(InputSerializedData stream, boolean exception) {
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(text);
        }
    }

    public static class TL_videoSizeEmojiMarkup extends VideoSize {
        public static final int constructor = 0xf85c413c;

        public long emoji_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            emoji_id = stream.readInt64(exception);
            background_colors = Vector.deserializeInt(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(emoji_id);
            Vector.serializeInt(stream, background_colors);
        }
    }

    public static class TL_videoSizeStickerMarkup extends VideoSize {
        public static final int constructor = 0xda082fe;

        public InputStickerSet stickerset;
        public long sticker_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            sticker_id = stream.readInt64(exception);
            background_colors = Vector.deserializeInt(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stickerset.serializeToStream(stream);
            stream.writeInt64(sticker_id);
            Vector.serializeInt(stream, background_colors);
        }
    }

    public static abstract class VideoSize extends TLObject {

        public int flags;
        public String type;
        public FileLocation location;
        public int w;
        public int h;
        public int size;
        public double video_start_ts;
        public ArrayList<Integer> background_colors = new ArrayList<>();

        public static VideoSize TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            return TLdeserialize(0, 0, stream, constructor, exception);
        }

        public static VideoSize TLdeserialize(long photo_id, long document_id, InputSerializedData stream, int constructor, boolean exception) {
            VideoSize result = null;
            switch (constructor) {
                case 0x435bb987:
                    result = new TL_videoSize_layer115();
                    break;
                case 0xe831c556:
                    result = new TL_videoSize_layer127();
                    break;
                case 0xde33b094:
                    result = new TL_videoSize();
                    break;
                case 0xda082fe:
                    result = new TL_videoSizeStickerMarkup();
                    break;
                case 0xf85c413c:
                    result = new TL_videoSizeEmojiMarkup();
                    break;
            }
            result = TLdeserialize(VideoSize.class, result, stream, constructor, exception);
            if (result != null && result.location == null) {
                if (!TextUtils.isEmpty(result.type) && (photo_id != 0 || document_id != 0)) {
                    result.location = new TL_fileLocationToBeDeprecated();
                    if (photo_id != 0) {
                        result.location.volume_id = -photo_id;
                        result.location.local_id = result.type.charAt(0);
                    } else {
                        result.location.volume_id = -document_id;
                        result.location.local_id = 1000 + result.type.charAt(0);
                    }
                } else {
                    result.location = new TL_fileLocationUnavailable();
                }
            }
            return result;
        }
    }

    public static class TL_videoSize_layer115 extends TL_videoSize {
        public static final int constructor = 0x435bb987;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = stream.readString(exception);
            location = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(type);
            location.serializeToStream(stream);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
        }
    }

    public static class TL_videoSize_layer127 extends TL_videoSize {
        public static final int constructor = 0xe831c556;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = stream.readString(exception);
            location = FileLocation.TLdeserialize(stream, stream.readInt32(exception), exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(type);
            location.serializeToStream(stream);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
            if ((flags & 1) != 0) {
                stream.writeDouble(video_start_ts);
            }
        }
    }

    public static class TL_videoSize extends VideoSize {
        public static final int constructor = 0xde33b094;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = stream.readString(exception);
            w = stream.readInt32(exception);
            h = stream.readInt32(exception);
            size = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_start_ts = stream.readDouble(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(type);
            stream.writeInt32(w);
            stream.writeInt32(h);
            stream.writeInt32(size);
            if ((flags & 1) != 0) {
                stream.writeDouble(video_start_ts);
            }
        }
    }

    public static abstract class BotInlineMessage extends TLObject {
        public int flags;
        public GeoPoint geo;
        public String title;
        public String address;
        public String provider;
        public String venue_id;
        public String venue_type;
        public ReplyMarkup reply_markup;
        public String message;
        public ArrayList<MessageEntity> entities = new ArrayList<>();
        public String phone_number;
        public String first_name;
        public String last_name;
        public String vcard;
        public boolean no_webpage;
        public boolean invert_media;
        public boolean force_large_media;
        public boolean force_small_media;
        public boolean manual;
        public boolean safe;
        public int period;
        public int heading;
        public int proximity_notification_radius;
        public String url;

        public static BotInlineMessage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            BotInlineMessage result = null;
            switch (constructor) {
                case 0x4366232e:
                    result = new TL_botInlineMessageMediaVenue_layer77();
                    break;
                case 0x8a86659c:
                    result = new TL_botInlineMessageMediaVenue();
                    break;
                case 0x3a8fd8b8:
                    result = new TL_botInlineMessageMediaGeo_layer71();
                    break;
                case 0x764cf810:
                    result = new TL_botInlineMessageMediaAuto();
                    break;
                case 0xa74b15b:
                    result = new TL_botInlineMessageMediaAuto_layer74();
                    break;
                case 0x35edb4d4:
                    result = new TL_botInlineMessageMediaContact_layer81();
                    break;
                case 0x18d1cdc2:
                    result = new TL_botInlineMessageMediaContact();
                    break;
                case 0x8c7f65e2:
                    result = new TL_botInlineMessageText();
                    break;
                case 0xb722de65:
                    result = new TL_botInlineMessageMediaGeo_layer119();
                    break;
                case 0x51846fd:
                    result = new TL_botInlineMessageMediaGeo();
                    break;
                case 0x354a9b09:
                    result = new TL_botInlineMessageMediaInvoice();
                    break;
                case 0x809ad9a6:
                    result = new TL_botInlineMessageMediaWebPage();
                    break;
            }
            return TLdeserialize(BotInlineMessage.class, result, stream, constructor, exception);
        }
    }

    public static class TL_botInlineMessageMediaVenue_layer77 extends TL_botInlineMessageMediaVenue {
        public static final int constructor = 0x4366232e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaVenue extends BotInlineMessage {
        public static final int constructor = 0x8a86659c;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            title = stream.readString(exception);
            address = stream.readString(exception);
            provider = stream.readString(exception);
            venue_id = stream.readString(exception);
            venue_type = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeString(title);
            stream.writeString(address);
            stream.writeString(provider);
            stream.writeString(venue_id);
            stream.writeString(venue_type);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo_layer71 extends TL_botInlineMessageMediaGeo {
        public static final int constructor = 0x3a8fd8b8;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaAuto extends BotInlineMessage {
        public static final int constructor = 0x764cf810;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, entities);
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaAuto_layer74 extends TL_botInlineMessageMediaAuto {
        public static final int constructor = 0xa74b15b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            message = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaContact_layer81 extends TL_botInlineMessageMediaContact {
        public static final int constructor = 0x35edb4d4;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaContact extends BotInlineMessage {
        public static final int constructor = 0x18d1cdc2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            phone_number = stream.readString(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            vcard = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeString(phone_number);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(vcard);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageText extends BotInlineMessage {
        public static final int constructor = 0x8c7f65e2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            no_webpage = (flags & 1) != 0;
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = no_webpage ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, entities);
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo extends BotInlineMessage {
        public static final int constructor = 0x51846fd;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                heading = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                period = stream.readInt32(exception);
            }
            if ((flags & 8) != 0) {
                proximity_notification_radius = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            if ((flags & 1) != 0) {
                stream.writeInt32(heading);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(period);
            }
            if ((flags & 8) != 0) {
                stream.writeInt32(proximity_notification_radius);
            }
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaGeo_layer119 extends TL_botInlineMessageMediaGeo {
        public static final int constructor = 0xb722de65;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            geo = GeoPoint.TLdeserialize(stream, stream.readInt32(exception), exception);
            period = stream.readInt32(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            geo.serializeToStream(stream);
            stream.writeInt32(period);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaWebPage extends BotInlineMessage {
        public static final int constructor = 0x809ad9a6;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            invert_media = (flags & 8) != 0;
            force_large_media = (flags & 16) != 0;
            force_small_media = (flags & 32) != 0;
            manual = (flags & 128) != 0;
            safe = (flags & 256) != 0;
            message = stream.readString(exception);
            if ((flags & 2) != 0) {
                entities = Vector.deserialize(stream, MessageEntity::TLdeserialize, exception);
            }
            url = stream.readString(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        @Override
        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = invert_media ? (flags | 8) : (flags &~ 8);
            flags = force_large_media ? (flags | 16) : (flags &~ 16);
            flags = force_small_media ? (flags | 32) : (flags &~ 32);
            flags = manual ? (flags | 128) : (flags &~ 128);
            flags = safe ? (flags | 256) : (flags &~ 256);
            stream.writeInt32(flags);
            stream.writeString(message);
            if ((flags & 2) != 0) {
                Vector.serialize(stream, entities);
            }
            stream.writeString(url);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_botInlineMessageMediaInvoice extends BotInlineMessage {
        public static final int constructor = 0x354a9b09;

        public boolean shipping_address_requested;
        public boolean test;
        public String description;
        public WebDocument photo;
        public String currency;
        public long total_amount;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            shipping_address_requested = (flags & 2) != 0;
            test = (flags & 8) != 0;
            title = stream.readString(exception);
            description = stream.readString(exception);
            if ((flags & 1) != 0) {
                photo = WebDocument.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            currency = stream.readString(exception);
            total_amount = stream.readInt64(exception);
            if ((flags & 4) != 0) {
                reply_markup = ReplyMarkup.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = shipping_address_requested ? (flags | 2) : (flags &~ 2);
            flags = test ? (flags | 8) : (flags &~ 8);
            stream.writeInt32(flags);
            stream.writeString(title);
            stream.writeString(description);
            if ((flags & 1) != 0) {
                photo.serializeToStream(stream);
            }
            stream.writeString(currency);
            stream.writeInt64(total_amount);
            if ((flags & 4) != 0) {
                reply_markup.serializeToStream(stream);
            }
        }
    }

    public static class TL_keyboardButtonRow extends TLObject {
        public static final int constructor = 0x77608b83;

        public ArrayList<KeyboardButton> buttons = new ArrayList<>();

        public static TL_keyboardButtonRow TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_keyboardButtonRow result = TL_keyboardButtonRow.constructor != constructor ? null : new TL_keyboardButtonRow();
            return TLdeserialize(TL_keyboardButtonRow.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            buttons = Vector.deserialize(stream, KeyboardButton::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, buttons);
        }
    }

    public static abstract class Bool extends TLObject {

        public static Bool TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            Bool result = null;
            switch (constructor) {
                case 0x997275b5:
                    result = new TL_boolTrue();
                    break;
                case 0xbc799737:
                    result = new TL_boolFalse();
                    break;
            }
            return TLdeserialize(Bool.class, result, stream, constructor, exception);
        }
    }

    public static class TL_boolTrue extends Bool {
        public static final int constructor = 0x997275b5;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_boolFalse extends Bool {
        public static final int constructor = 0xbc799737;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_messages_discussionMessage extends TLObject {
        public static final int constructor = 0xa6341782;

        public int flags;
        public ArrayList<Message> messages = new ArrayList<>();
        public int max_id;
        public int read_inbox_max_id;
        public int read_outbox_max_id;
        public int unread_count;
        public ArrayList<Chat> chats = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_discussionMessage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_discussionMessage result = TL_messages_discussionMessage.constructor != constructor ? null : new TL_messages_discussionMessage();
            return TLdeserialize(TL_messages_discussionMessage.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            messages = Vector.deserialize(stream, Message::TLdeserialize, exception);
            if ((flags & 1) != 0) {
                max_id = stream.readInt32(exception);
            }
            if ((flags & 2) != 0) {
                read_inbox_max_id = stream.readInt32(exception);
            }
            if ((flags & 4) != 0) {
                read_outbox_max_id = stream.readInt32(exception);
            }
            unread_count = stream.readInt32(exception);
            chats = Vector.deserialize(stream, Chat::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            Vector.serialize(stream, messages);
            if ((flags & 1) != 0) {
                stream.writeInt32(max_id);
            }
            if ((flags & 2) != 0) {
                stream.writeInt32(read_inbox_max_id);
            }
            if ((flags & 4) != 0) {
                stream.writeInt32(read_outbox_max_id);
            }
            stream.writeInt32(unread_count);
            Vector.serialize(stream, chats);
            Vector.serialize(stream, users);
        }
    }

    public static class TL_messages_searchCounter extends TLObject {
        public static final int constructor = 0xe844ebff;

        public int flags;
        public boolean inexact;
        public MessagesFilter filter;
        public int count;

        public static TL_messages_searchCounter TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_searchCounter result = TL_messages_searchCounter.constructor != constructor ? null : new TL_messages_searchCounter();
            return TLdeserialize(TL_messages_searchCounter.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            inexact = (flags & 2) != 0;
            filter = MessagesFilter.TLdeserialize(stream, stream.readInt32(exception), exception);
            count = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = inexact ? (flags | 2) : (flags &~ 2);
            stream.writeInt32(flags);
            filter.serializeToStream(stream);
            stream.writeInt32(count);
        }
    }

    public static class TL_inputAppEvent extends TLObject {
        public static final int constructor = 0x1d1b1245;

        public double time;
        public String type;
        public long peer;
        public JSONValue data;

        public static TL_inputAppEvent TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_inputAppEvent result = TL_inputAppEvent.constructor != constructor ? null : new TL_inputAppEvent();
            return TLdeserialize(TL_inputAppEvent.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            time = stream.readDouble(exception);
            type = stream.readString(exception);
            peer = stream.readInt64(exception);
            data = JSONValue.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeDouble(time);
            stream.writeString(type);
            stream.writeInt64(peer);
            data.serializeToStream(stream);
        }
    }

    public static class TL_auth_exportedAuthorization extends TLObject {
        public static final int constructor = 0xb434e2b8;

        public long id;
        public byte[] bytes;

        public static TL_auth_exportedAuthorization TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_auth_exportedAuthorization result = TL_auth_exportedAuthorization.constructor != constructor ? null : new TL_auth_exportedAuthorization();
            return TLdeserialize(TL_auth_exportedAuthorization.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            bytes = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeByteArray(bytes);
        }
    }

    public static abstract class WebPage extends TLObject {
        public int flags;
        public boolean has_large_media;
        public boolean video_cover_photo;
        public long id;
        public String url;
        public String display_url;
        public int hash;
        public String type;
        public String site_name;
        public String title;
        public String description;
        public Photo photo;
        public String embed_url;
        public String embed_type;
        public int embed_width;
        public int embed_height;
        public int duration;
        public String author;
        public Document document;
        public Page cached_page;
        public int date;
        public ArrayList<WebPageAttribute> attributes = new ArrayList<>();
        public String displayedText;//custom
        public String archivePath;//custom

        public static WebPage TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            WebPage result = null;
            switch (constructor) {
                case 0xe89c45b2:
                    result = new TL_webPage();
                    break;
                case 0xfa64e172:
                    result = new TL_webPage_layer107();
                    break;
                case 0x5f07b4bc:
                    result = new TL_webPage_layer104();
                    break;
                case 0xa31ea0b5:
                    result = new TL_webPage_old();
                    break;
                case 0x211a1788:
                    result = new TL_webPageEmpty();
                    break;
                case 0xeb1477e8:
                    result = new TL_webPageEmpty_layer165();
                    break;
                case 0xd41a5167:
                    result = new TL_webPageUrlPending();
                    break;
                case 0x7311ca11:
                    result = new TL_webPageNotModified();
                    break;
                case 0xb0d13e47:
                    result = new TL_webPagePending();
                    break;
                case 0xc586da1c:
                    result = new TL_webPagePending_layer165();
                    break;
                case 0x85849473:
                    result = new TL_webPageNotModified_layer110();
                    break;
                case 0xca820ed7:
                    result = new TL_webPage_layer58();
                    break;
            }
            return TLdeserialize(WebPage.class, result, stream, constructor, exception);
        }
    }

    public static class TL_webPage extends WebPage {
        public static final int constructor = 0xe89c45b2;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_large_media = (flags & 8192) != 0;
            video_cover_photo = (flags & 16384) != 0;
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4096) != 0) {
                attributes = Vector.deserialize(stream, WebPageAttribute::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_large_media ? flags | 8192 : flags &~ 8192;
            flags = video_cover_photo ? flags | 16384 : flags &~ 16384;
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
            if ((flags & 4096) != 0) {
                Vector.serialize(stream, attributes);
            }
        }
    }

    public static class TL_webPage_layer107 extends TL_webPage {
        public static final int constructor = 0xfa64e172;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2048) != 0) {
                int magic = stream.readInt32(exception);
                if (magic != 0x1cb5c415) {
                    if (exception) {
                        throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                    }
                    return;
                }
                TL_webPageAttributeTheme attribute = new TL_webPageAttributeTheme();
                int count = stream.readInt32(exception);
                for (int a = 0; a < count; a++) {
                    Document object = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
                    if (object == null) {
                        return;
                    }
                    attribute.documents.add(object);
                }
                attributes.add(attribute);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 2048) != 0) {
                stream.writeInt32(0x1cb5c415);
                stream.writeInt32(0);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPage_layer104 extends TL_webPage {
        public static final int constructor = 0x5f07b4bc;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            hash = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 1024) != 0) {
                cached_page = Page.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            stream.writeInt32(hash);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
            if ((flags & 1024) != 0) {
                cached_page.serializeToStream(stream);
            }
        }
    }

    public static class TL_webPage_old extends TL_webPage {
        public static final int constructor = 0xa31ea0b5;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
        }
    }

    public static class TL_webPageEmpty extends WebPage {
        public static final int constructor = 0x211a1788;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                url = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeString(url);
            }
        }
    }

    public static class TL_webPageEmpty_layer165 extends TL_webPageEmpty {
        public static final int constructor = 0xeb1477e8;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_webPageUrlPending extends WebPage {
        public static final int constructor = 0xd41a5167;

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
        }
    }

    public static class TL_webPagePending_layer165 extends TL_webPagePending {
        public static final int constructor = 0xc586da1c;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt32(date);
        }
    }

    public static class TL_webPagePending extends WebPage {
        public static final int constructor = 0xb0d13e47;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                url = stream.readString(exception);
            }
            date = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeString(url);
            }
            stream.writeInt32(date);
        }
    }

    public static class TL_webPageNotModified_layer110 extends TL_webPageNotModified {
        public static final int constructor = 0x85849473;

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {

        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_webPageNotModified extends WebPage {
        public static final int constructor = 0x7311ca11;

        public int cached_page_views;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                cached_page_views = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            if ((flags & 1) != 0) {
                stream.writeInt32(cached_page_views);
            }
        }
    }

    public static class TL_webPage_layer58 extends TL_webPage {
        public static final int constructor = 0xca820ed7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            id = stream.readInt64(exception);
            url = stream.readString(exception);
            display_url = stream.readString(exception);
            if ((flags & 1) != 0) {
                type = stream.readString(exception);
            }
            if ((flags & 2) != 0) {
                site_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                title = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                description = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 32) != 0) {
                embed_url = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                embed_type = stream.readString(exception);
            }
            if ((flags & 64) != 0) {
                embed_width = stream.readInt32(exception);
            }
            if ((flags & 64) != 0) {
                embed_height = stream.readInt32(exception);
            }
            if ((flags & 128) != 0) {
                duration = stream.readInt32(exception);
            }
            if ((flags & 256) != 0) {
                author = stream.readString(exception);
            }
            if ((flags & 512) != 0) {
                document = Document.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            stream.writeString(url);
            stream.writeString(display_url);
            if ((flags & 1) != 0) {
                stream.writeString(type);
            }
            if ((flags & 2) != 0) {
                stream.writeString(site_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(title);
            }
            if ((flags & 8) != 0) {
                stream.writeString(description);
            }
            if ((flags & 16) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_url);
            }
            if ((flags & 32) != 0) {
                stream.writeString(embed_type);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_width);
            }
            if ((flags & 64) != 0) {
                stream.writeInt32(embed_height);
            }
            if ((flags & 128) != 0) {
                stream.writeInt32(duration);
            }
            if ((flags & 256) != 0) {
                stream.writeString(author);
            }
            if ((flags & 512) != 0) {
                document.serializeToStream(stream);
            }
        }
    }

    public static abstract class messages_FeaturedStickers extends TLObject {

        public static messages_FeaturedStickers TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            messages_FeaturedStickers result = null;
            switch (constructor) {
                case 0xbe382906:
                    result = new TL_messages_featuredStickers();
                    break;
                case 0xc6dc0c66:
                    result = new TL_messages_featuredStickersNotModified();
                    break;
            }
            return TLdeserialize(messages_FeaturedStickers.class, result, stream, constructor, exception);
        }
    }

    public static class TL_messages_featuredStickers extends messages_FeaturedStickers {
        public static final int constructor = 0xbe382906;

        public int flags;
        public boolean premium;
        public long hash;
        public int count;
        public ArrayList<StickerSetCovered> sets = new ArrayList<>();
        public ArrayList<Long> unread = new ArrayList<>();

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            premium = (flags & 1) != 0;
            hash = stream.readInt64(exception);
            count = stream.readInt32(exception);
            sets = Vector.deserialize(stream, StickerSetCovered::TLdeserialize, exception);
            unread = Vector.deserializeLong(stream, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = premium ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            stream.writeInt64(hash);
            stream.writeInt32(count);
            Vector.serialize(stream, sets);
            Vector.serializeLong(stream, unread);
        }
    }

    public static class TL_messages_featuredStickersNotModified extends messages_FeaturedStickers {
        public static final int constructor = 0xc6dc0c66;

        public int count;

        public void readParams(InputSerializedData stream, boolean exception) {
            count = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(count);
        }
    }

    public static abstract class SecureValueError extends TLObject {

        public static SecureValueError TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            SecureValueError result = null;
            switch (constructor) {
                case 0x7a700873:
                    result = new TL_secureValueErrorFile();
                    break;
                case 0xbe3dfa:
                    result = new TL_secureValueErrorFrontSide();
                    break;
                case 0x666220e9:
                    result = new TL_secureValueErrorFiles();
                    break;
                case 0x868a2aa5:
                    result = new TL_secureValueErrorReverseSide();
                    break;
                case 0xa1144770:
                    result = new TL_secureValueErrorTranslationFile();
                    break;
                case 0x869d758f:
                    result = new TL_secureValueError();
                    break;
                case 0xe8a40bd9:
                    result = new TL_secureValueErrorData();
                    break;
                case 0x34636dd8:
                    result = new TL_secureValueErrorTranslationFiles();
                    break;
                case 0xe537ced6:
                    result = new TL_secureValueErrorSelfie();
                    break;
            }
            return TLdeserialize(SecureValueError.class, result, stream, constructor, exception);
        }
    }

    public static class TL_secureValueErrorFile extends SecureValueError {
        public static final int constructor = 0x7a700873;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorFrontSide extends SecureValueError {
        public static final int constructor = 0xbe3dfa;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorFiles extends SecureValueError {
        public static final int constructor = 0x666220e9;

        public SecureValueType type;
        public ArrayList<byte[]> file_hash = new ArrayList<>();
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                file_hash.add(stream.readByteArray(exception));
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = file_hash.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeByteArray(file_hash.get(a));
            }
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorReverseSide extends SecureValueError {
        public static final int constructor = 0x868a2aa5;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorTranslationFile extends SecureValueError {
        public static final int constructor = 0xa1144770;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueError extends SecureValueError {
        public static final int constructor = 0x869d758f;

        public SecureValueType type;
        public byte[] hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorData extends SecureValueError {
        public static final int constructor = 0xe8a40bd9;

        public SecureValueType type;
        public byte[] data_hash;
        public String field;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            data_hash = stream.readByteArray(exception);
            field = stream.readString(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(data_hash);
            stream.writeString(field);
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorTranslationFiles extends SecureValueError {
        public static final int constructor = 0x34636dd8;

        public SecureValueType type;
        public ArrayList<byte[]> file_hash = new ArrayList<>();
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            int magic = stream.readInt32(exception);
            if (magic != 0x1cb5c415) {
                if (exception) {
                    throw new RuntimeException(String.format("wrong Vector magic, got %x", magic));
                }
                return;
            }
            int count = stream.readInt32(exception);
            for (int a = 0; a < count; a++) {
                file_hash.add(stream.readByteArray(exception));
            }
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeInt32(0x1cb5c415);
            int count = file_hash.size();
            stream.writeInt32(count);
            for (int a = 0; a < count; a++) {
                stream.writeByteArray(file_hash.get(a));
            }
            stream.writeString(text);
        }
    }

    public static class TL_secureValueErrorSelfie extends SecureValueError {
        public static final int constructor = 0xe537ced6;

        public SecureValueType type;
        public byte[] file_hash;
        public String text;

        public void readParams(InputSerializedData stream, boolean exception) {
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            file_hash = stream.readByteArray(exception);
            text = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            type.serializeToStream(stream);
            stream.writeByteArray(file_hash);
            stream.writeString(text);
        }
    }

    public static class TL_secureValue extends TLObject {
        public static final int constructor = 0x187fa0ca;

        public int flags;
        public SecureValueType type;
        public TL_secureData data;
        public SecureFile front_side;
        public SecureFile reverse_side;
        public SecureFile selfie;
        public ArrayList<SecureFile> translation = new ArrayList<>();
        public ArrayList<SecureFile> files = new ArrayList<>();
        public SecurePlainData plain_data;
        public byte[] hash;

        public static TL_secureValue TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_secureValue result = TL_secureValue.constructor != constructor ? null : new TL_secureValue();
            return TLdeserialize(TL_secureValue.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            type = SecureValueType.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 1) != 0) {
                data = TL_secureData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 2) != 0) {
                front_side = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 4) != 0) {
                reverse_side = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 8) != 0) {
                selfie = SecureFile.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                translation = Vector.deserialize(stream, SecureFile::TLdeserialize, exception);
            }
            if ((flags & 16) != 0) {
                files = Vector.deserialize(stream, SecureFile::TLdeserialize, exception);
            }
            if ((flags & 32) != 0) {
                plain_data = SecurePlainData.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            hash = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            type.serializeToStream(stream);
            if ((flags & 1) != 0) {
                data.serializeToStream(stream);
            }
            if ((flags & 2) != 0) {
                front_side.serializeToStream(stream);
            }
            if ((flags & 4) != 0) {
                reverse_side.serializeToStream(stream);
            }
            if ((flags & 8) != 0) {
                selfie.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                Vector.serialize(stream, translation);
            }
            if ((flags & 16) != 0) {
                Vector.serialize(stream, files);
            }
            if ((flags & 32) != 0) {
                plain_data.serializeToStream(stream);
            }
            stream.writeByteArray(hash);
        }
    }

    public static abstract class PhoneCallDiscardReason extends TLObject {
        public String slug;

        public static PhoneCallDiscardReason TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            PhoneCallDiscardReason result = null;
            switch (constructor) {
                case TL_phoneCallDiscardReasonHangup.constructor:
                    result = new TL_phoneCallDiscardReasonHangup();
                    break;
                case TL_phoneCallDiscardReasonBusy.constructor:
                    result = new TL_phoneCallDiscardReasonBusy();
                    break;
                case TL_phoneCallDiscardReasonMissed.constructor:
                    result = new TL_phoneCallDiscardReasonMissed();
                    break;
                case TL_phoneCallDiscardReasonDisconnect.constructor:
                    result = new TL_phoneCallDiscardReasonDisconnect();
                    break;
                case TL_phoneCallDiscardReasonMigrateConferenceCall.constructor:
                    result = new TL_phoneCallDiscardReasonMigrateConferenceCall();
                    break;
            }
            return TLdeserialize(PhoneCallDiscardReason.class, result, stream, constructor, exception);
        }
    }

    public static class TL_phoneCallDiscardReasonHangup extends PhoneCallDiscardReason {
        public static final int constructor = 0x57adc690;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonBusy extends PhoneCallDiscardReason {
        public static final int constructor = 0xfaf7e8c9;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonMissed extends PhoneCallDiscardReason {
        public static final int constructor = 0x85e42301;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonDisconnect extends PhoneCallDiscardReason {
        public static final int constructor = 0xe095c1a0;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_phoneCallDiscardReasonMigrateConferenceCall extends PhoneCallDiscardReason {
        public static final int constructor = 0x9fbbf1f7;

        public void readParams(InputSerializedData stream, boolean exception) {
            slug = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(slug);
        }
    }

    public static abstract class DialogFilter extends TLObject {

        public int flags;
        public boolean contacts;
        public boolean non_contacts;
        public boolean groups;
        public boolean broadcasts;
        public boolean bots;
        public boolean exclude_muted;
        public boolean exclude_read;
        public boolean exclude_archived;
        public boolean title_noanimate;
        public int id;
        public TL_textWithEntities title = new TL_textWithEntities();
        public String emoticon;
        public ArrayList<InputPeer> pinned_peers = new ArrayList<>();
        public ArrayList<InputPeer> include_peers = new ArrayList<>();
        public ArrayList<InputPeer> exclude_peers = new ArrayList<>();
        public boolean has_my_invites;
        public int color;

        public static DialogFilter TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            DialogFilter result = null;
            switch (constructor) {
                case TL_dialogFilterDefault.constructor:
                    result = new TL_dialogFilterDefault();
                    break;
                case TL_dialogFilter.constructor:
                    result = new TL_dialogFilter();
                    break;
                case TL_dialogFilter_layer195.constructor:
                    result = new TL_dialogFilter_layer195();
                    break;
                case TL_dialogFilter_layer175.constructor:
                    result = new TL_dialogFilter_layer175();
                    break;
                case TL_dialogFilterChatlist.constructor:
                    result = new TL_dialogFilterChatlist();
                    break;
                case TL_dialogFilterChatlist_layer195.constructor:
                    result = new TL_dialogFilterChatlist_layer195();
                    break;
                case TL_dialogFilterChatlist_layer175.constructor:
                    result = new TL_dialogFilterChatlist_layer175();
                    break;
            }
            return TLdeserialize(DialogFilter.class, result, stream, constructor, exception);
        }
    }

    public static class TL_dialogFilterChatlist extends DialogFilter {
        public static final int constructor = 0x96537bd7;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_my_invites = (flags & 67108864) != 0;
            id = stream.readInt32(exception);
            title = TL_textWithEntities.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            if ((flags & 134217728) != 0) {
                color = stream.readInt32(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_my_invites ? (flags | 67108864) : (flags &~ 67108864);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            title.serializeToStream(stream);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            if ((flags & 134217728) != 0) {
                stream.writeInt32(color);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
        }
    }

    public static class TL_dialogFilterChatlist_layer195 extends TL_dialogFilterChatlist {
        public static final int constructor = 0x9fe28ea4;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_my_invites = (flags & 67108864) != 0;
            id = stream.readInt32(exception);
            title = new TL_textWithEntities();
            title.text = stream.readString(exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            if ((flags & 134217728) != 0) {
                color = stream.readInt32(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_my_invites ? (flags | 67108864) : (flags &~ 67108864);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title == null ? "" : title.text);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            if ((flags & 134217728) != 0) {
                stream.writeInt32(color);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
        }
    }

    public static class TL_dialogFilterChatlist_layer175 extends TL_dialogFilterChatlist {
        public static final int constructor = 0xd64a04a8;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            has_my_invites = (flags & 67108864) != 0;
            id = stream.readInt32(exception);
            title = new TL_textWithEntities();
            title.text = stream.readString(exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = has_my_invites ? (flags | 67108864) : (flags &~ 67108864);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title == null ? "" : title.text);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
        }
    }

    public static class TL_dialogFilterDefault extends DialogFilter {
        public static final int constructor = 0x363293ae;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_dialogFilter extends DialogFilter {
        public static final int constructor = 0xaa472651;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            contacts = (flags & 1) != 0;
            non_contacts = (flags & 2) != 0;
            groups = (flags & 4) != 0;
            broadcasts = (flags & 8) != 0;
            bots = (flags & 16) != 0;
            exclude_muted = (flags & 2048) != 0;
            exclude_read = (flags & 4096) != 0;
            exclude_archived = (flags & 8192) != 0;
            title_noanimate = (flags & 268435456) != 0;
            id = stream.readInt32(exception);
            title = TL_textWithEntities.TLdeserialize(stream, stream.readInt32(exception), exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            if ((flags & 134217728) != 0) {
                color = stream.readInt32(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            exclude_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = contacts ? (flags | 1) : (flags &~ 1);
            flags = non_contacts ? (flags | 2) : (flags &~ 2);
            flags = groups ? (flags | 4) : (flags &~ 4);
            flags = broadcasts ? (flags | 8) : (flags &~ 8);
            flags = bots ? (flags | 16) : (flags &~ 16);
            flags = exclude_muted ? (flags | 2048) : (flags &~ 2048);
            flags = exclude_read ? (flags | 4096) : (flags &~ 4096);
            flags = exclude_archived ? (flags | 8192) : (flags &~ 8192);
            flags = title_noanimate ? (flags | 268435456) : (flags &~ 268435456);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            title.serializeToStream(stream);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            if ((flags & 134217728) != 0) {
                stream.writeInt32(color);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
            Vector.serialize(stream, exclude_peers);
        }
    }

    public static class TL_dialogFilter_layer195 extends TL_dialogFilter {
        public static final int constructor = 0x5fb5523b;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            contacts = (flags & 1) != 0;
            non_contacts = (flags & 2) != 0;
            groups = (flags & 4) != 0;
            broadcasts = (flags & 8) != 0;
            bots = (flags & 16) != 0;
            exclude_muted = (flags & 2048) != 0;
            exclude_read = (flags & 4096) != 0;
            exclude_archived = (flags & 8192) != 0;
            id = stream.readInt32(exception);
            title = new TL_textWithEntities();
            title.text = stream.readString(exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            if ((flags & 134217728) != 0) {
                color = stream.readInt32(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            exclude_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = contacts ? (flags | 1) : (flags &~ 1);
            flags = non_contacts ? (flags | 2) : (flags &~ 2);
            flags = groups ? (flags | 4) : (flags &~ 4);
            flags = broadcasts ? (flags | 8) : (flags &~ 8);
            flags = bots ? (flags | 16) : (flags &~ 16);
            flags = exclude_muted ? (flags | 2048) : (flags &~ 2048);
            flags = exclude_read ? (flags | 4096) : (flags &~ 4096);
            flags = exclude_archived ? (flags | 8192) : (flags &~ 8192);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title == null ? "" : title.text);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            if ((flags & 134217728) != 0) {
                stream.writeInt32(color);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
            Vector.serialize(stream, exclude_peers);
        }
    }

    public static class TL_dialogFilter_layer175 extends TL_dialogFilter {
        public static final int constructor = 0x7438f7e8;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            contacts = (flags & 1) != 0;
            non_contacts = (flags & 2) != 0;
            groups = (flags & 4) != 0;
            broadcasts = (flags & 8) != 0;
            bots = (flags & 16) != 0;
            exclude_muted = (flags & 2048) != 0;
            exclude_read = (flags & 4096) != 0;
            exclude_archived = (flags & 8192) != 0;
            id = stream.readInt32(exception);
            title = new TL_textWithEntities();
            title.text = stream.readString(exception);
            if ((flags & 33554432) != 0) {
                emoticon = stream.readString(exception);
            }
            pinned_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            include_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
            exclude_peers = Vector.deserialize(stream, InputPeer::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = contacts ? (flags | 1) : (flags &~ 1);
            flags = non_contacts ? (flags | 2) : (flags &~ 2);
            flags = groups ? (flags | 4) : (flags &~ 4);
            flags = broadcasts ? (flags | 8) : (flags &~ 8);
            flags = bots ? (flags | 16) : (flags &~ 16);
            flags = exclude_muted ? (flags | 2048) : (flags &~ 2048);
            flags = exclude_read ? (flags | 4096) : (flags &~ 4096);
            flags = exclude_archived ? (flags | 8192) : (flags &~ 8192);
            stream.writeInt32(flags);
            stream.writeInt32(id);
            stream.writeString(title == null ? "" : title.text);
            if ((flags & 33554432) != 0) {
                stream.writeString(emoticon);
            }
            Vector.serialize(stream, pinned_peers);
            Vector.serialize(stream, include_peers);
            Vector.serialize(stream, exclude_peers);
        }
    }

    public static class TL_auth_passwordRecovery extends TLObject {
        public static final int constructor = 0x137948a5;

        public String email_pattern;

        public static TL_auth_passwordRecovery TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_auth_passwordRecovery result = TL_auth_passwordRecovery.constructor != constructor ? null : new TL_auth_passwordRecovery();
            return TLdeserialize(TL_auth_passwordRecovery.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            email_pattern = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(email_pattern);
        }
    }

    public static class TL_exportedContactToken extends TLObject {
        public static final int constructor = 0x41bf109b;

        public String url;
        public int expires;

        public static TL_exportedContactToken TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_exportedContactToken result = TL_exportedContactToken.constructor != constructor ? null : new TL_exportedContactToken();
            return TLdeserialize(TL_exportedContactToken.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            url = stream.readString(exception);
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(url);
            stream.writeInt32(expires);
        }
    }

    public static class TL_botCommand extends TLObject {
        public static final int constructor = 0xc27ac8c7;

        public String command;
        public String description;

        public static TL_botCommand TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_botCommand result = TL_botCommand.constructor != constructor ? null : new TL_botCommand();
            return TLdeserialize(TL_botCommand.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            command = stream.readString(exception);
            description = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeString(command);
            stream.writeString(description);
        }
    }

    public static abstract class InputNotifyPeer extends TLObject {

        public static InputNotifyPeer TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputNotifyPeer result = null;
            switch (constructor) {
                case 0x4a95e84e:
                    result = new TL_inputNotifyChats();
                    break;
                case 0xb8bc5b0c:
                    result = new TL_inputNotifyPeer();
                    break;
                case 0x193b4417:
                    result = new TL_inputNotifyUsers();
                    break;
                case 0xb1db7c7e:
                    result = new TL_inputNotifyBroadcasts();
                    break;
                case 0x5c467992:
                    result = new TL_inputNotifyForumTopic();
                    break;
            }
            return TLdeserialize(InputNotifyPeer.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputNotifyChats extends InputNotifyPeer {
        public static final int constructor = 0x4a95e84e;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyPeer extends InputNotifyPeer {
        public static final int constructor = 0xb8bc5b0c;

        public InputPeer peer;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
        }
    }

    public static class TL_inputNotifyUsers extends InputNotifyPeer {
        public static final int constructor = 0x193b4417;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyBroadcasts extends InputNotifyPeer {
        public static final int constructor = 0xb1db7c7e;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_inputNotifyForumTopic extends InputNotifyPeer {
        public static final int constructor = 0x5c467992;

        public InputPeer peer;
        public int top_msg_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            top_msg_id = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            peer.serializeToStream(stream);
            stream.writeInt32(top_msg_id);
        }
    }

    public static abstract class InputFileLocation extends TLObject {

        public long id;
        public long access_hash;
        public byte[] file_reference;
        public String thumb_size;
        public int flags;
        public long volume_id;
        public int local_id;
        public long secret;

        public static InputFileLocation TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            InputFileLocation result = null;
            switch (constructor) {
                case 0x40181ffe:
                    result = new TL_inputPhotoFileLocation();
                    break;
                case 0xbad07584:
                    result = new TL_inputDocumentFileLocation();
                    break;
                case 0x37257e99:
                    result = new TL_inputPeerPhotoFileLocation();
                    break;
                case 0xf5235d55:
                    result = new TL_inputEncryptedFileLocation();
                    break;
                case 0x9d84f3db:
                    result = new TL_inputStickerSetThumb();
                    break;
                case 0x598a92a:
                    result = new TL_inputGroupCallStream();
                    break;
                case 0xdfdaabe1:
                    result = new TL_inputFileLocation();
                    break;
                case 0xcbc7ee28:
                    result = new TL_inputSecureFileLocation();
                    break;
            }
            return TLdeserialize(InputFileLocation.class, result, stream, constructor, exception);
        }
    }

    public static class TL_inputPhotoFileLocation extends InputFileLocation {
        public static final int constructor = 0x40181ffe;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
            thumb_size = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeByteArray(file_reference);
            stream.writeString(thumb_size);
        }
    }

    public static class TL_inputDocumentFileLocation extends InputFileLocation {
        public static final int constructor = 0xbad07584;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
            thumb_size = stream.readString(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
            stream.writeByteArray(file_reference);
            stream.writeString(thumb_size);
        }
    }

    public static class TL_inputPeerPhotoFileLocation extends InputFileLocation {
        public static final int constructor = 0x37257e99;

        public boolean big;
        public InputPeer peer;
        public long photo_id;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            big = (flags & 1) != 0;
            peer = InputPeer.TLdeserialize(stream, stream.readInt32(exception), exception);
            photo_id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = big ? (flags | 1) : (flags &~ 1);
            stream.writeInt32(flags);
            peer.serializeToStream(stream);
            stream.writeInt64(photo_id);
        }
    }

    public static class TL_inputEncryptedFileLocation extends InputFileLocation {
        public static final int constructor = 0xf5235d55;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_inputStickerSetThumb extends InputFileLocation {
        public static final int constructor = 0x9d84f3db;

        public InputStickerSet stickerset;
        public int thumb_version;

        public void readParams(InputSerializedData stream, boolean exception) {
            stickerset = InputStickerSet.TLdeserialize(stream, stream.readInt32(exception), exception);
            thumb_version = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stickerset.serializeToStream(stream);
            stream.writeInt32(thumb_version);
        }
    }

    public static class TL_inputGroupCallStream extends InputFileLocation {
        public static final int constructor = 0x598a92a;

        public InputGroupCall call;
        public long time_ms;
        public int scale;
        public int video_channel;
        public int video_quality;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            call = InputGroupCall.TLdeserialize(stream, stream.readInt32(exception), exception);
            time_ms = stream.readInt64(exception);
            scale = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                video_channel = stream.readInt32(exception);
            }
            if ((flags & 1) != 0) {
                video_quality = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(flags);
            call.serializeToStream(stream);
            stream.writeInt64(time_ms);
            stream.writeInt32(scale);
            if ((flags & 1) != 0) {
                stream.writeInt32(video_channel);
            }
            if ((flags & 1) != 0) {
                stream.writeInt32(video_quality);
            }
        }
    }

    public static class TL_inputFileLocation extends InputFileLocation {
        public static final int constructor = 0xdfdaabe1;

        public void readParams(InputSerializedData stream, boolean exception) {
            volume_id = stream.readInt64(exception);
            local_id = stream.readInt32(exception);
            secret = stream.readInt64(exception);
            file_reference = stream.readByteArray(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(volume_id);
            stream.writeInt32(local_id);
            stream.writeInt64(secret);
            stream.writeByteArray(file_reference);
        }
    }

    public static class TL_inputSecureFileLocation extends InputFileLocation {
        public static final int constructor = 0xcbc7ee28;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
            access_hash = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
            stream.writeInt64(access_hash);
        }
    }

    public static class TL_photos_photo extends TLObject {
        public static final int constructor = 0x20212ca8;

        public Photo photo;
        public ArrayList<User> users = new ArrayList<>();

        public static TL_photos_photo TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_photos_photo result = TL_photos_photo.constructor != constructor ? null : new TL_photos_photo();
            return TLdeserialize(TL_photos_photo.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            photo = Photo.TLdeserialize(stream, stream.readInt32(exception), exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            photo.serializeToStream(stream);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class help_TermsOfServiceUpdate extends TLObject {

        public static help_TermsOfServiceUpdate TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_TermsOfServiceUpdate result = null;
            switch (constructor) {
                case 0x28ecf961:
                    result = new TL_help_termsOfServiceUpdate();
                    break;
                case 0xe3309f7f:
                    result = new TL_help_termsOfServiceUpdateEmpty();
                    break;
            }
            return TLdeserialize(help_TermsOfServiceUpdate.class, result, stream, constructor, exception);
        }
    }

    public static class TL_help_termsOfServiceUpdate extends help_TermsOfServiceUpdate {
        public static final int constructor = 0x28ecf961;

        public int expires;
        public TL_help_termsOfService terms_of_service;

        public void readParams(InputSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
            terms_of_service = TL_help_termsOfService.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
            terms_of_service.serializeToStream(stream);
        }
    }

    public static class TL_help_termsOfServiceUpdateEmpty extends help_TermsOfServiceUpdate {
        public static final int constructor = 0xe3309f7f;

        public int expires;

        public void readParams(InputSerializedData stream, boolean exception) {
            expires = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32(expires);
        }
    }

    public static class TL_messages_chatAdminsWithInvites extends TLObject {
        public static final int constructor = 0xb69b72d7;

        public ArrayList<TL_chatAdminWithInvites> admins = new ArrayList<>();
        public ArrayList<User> users = new ArrayList<>();

        public static TL_messages_chatAdminsWithInvites TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_messages_chatAdminsWithInvites result = TL_messages_chatAdminsWithInvites.constructor != constructor ? null : new TL_messages_chatAdminsWithInvites();
            return TLdeserialize(TL_messages_chatAdminsWithInvites.class, result, stream, constructor, exception);
        }

        public void readParams(InputSerializedData stream, boolean exception) {
            admins = Vector.deserialize(stream, TL_chatAdminWithInvites::TLdeserialize, exception);
            users = Vector.deserialize(stream, User::TLdeserialize, exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, admins);
            Vector.serialize(stream, users);
        }
    }

    public static abstract class help_CountriesList extends TLObject {

        public static help_CountriesList TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            help_CountriesList result = null;
            switch (constructor) {
                case 0x93cc1f32:
                    result = new TL_help_countriesListNotModified();
                    break;
                case 0x87d0759e:
                    result = new TL_help_countriesList();
                    break;
            }
            return TLdeserialize(help_CountriesList.class, result, stream, constructor, exception);
        }
    }

    public static class TL_help_countriesListNotModified extends help_CountriesList {
        public static final int constructor = 0x93cc1f32;

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
        }
    }

    public static class TL_help_countriesList extends help_CountriesList {
        public static final int constructor = 0x87d0759e;

        public ArrayList<TL_help_country> countries = new ArrayList<>();
        public int hash;

        public void readParams(InputSerializedData stream, boolean exception) {
            countries = Vector.deserialize(stream, TL_help_country::TLdeserialize, exception);
            hash = stream.readInt32(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            Vector.serialize(stream, countries);
            stream.writeInt32(hash);
        }
    }

    public static abstract class User extends TLObject {
        public long id;
        public String first_name;
        public String last_name;
        public String username;
        public long access_hash;
        public String phone;
        public UserProfilePhoto photo;
        public UserStatus status;
        public int flags;
        public int flags2;
        public boolean self;
        public boolean contact;
        public boolean mutual_contact;
        public boolean deleted;
        public boolean bot;
        public boolean bot_chat_history;
        public boolean bot_nochats;
        public boolean bot_can_edit;
        public boolean bot_business;
        public boolean bot_has_main_app;
        public boolean verified;
        public boolean restricted;
        public boolean min;
        public boolean bot_inline_geo;
        public boolean support;
        public boolean scam;
        public boolean apply_min_photo;
        public boolean fake;
        public boolean premium;
        public boolean close_friend;
        public boolean stories_unavailable;
        public boolean stories_hidden;
        public boolean contact_require_premium;
        public int bot_info_version;
        public String bot_inline_placeholder;
        public String lang_code;
        public boolean inactive;
        public boolean explicit_content;
        public ArrayList<RestrictionReason> restriction_reason = new ArrayList<>();
        public boolean bot_attach_menu;
        public boolean bot_menu_webview;
        public boolean attach_menu_enabled;
        public EmojiStatus emoji_status;
        public ArrayList<TL_username> usernames = new ArrayList<>();
        public TL_recentStory stories_max_id;
        public PeerColor color;
        public PeerColor profile_color;
        public int bot_active_users;
        public long bot_verification_icon;
        public long send_paid_messages_stars;
        public boolean bot_forum_view;

        public static User TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            User result = null;
            switch (constructor) {
                case TL_user.constructor:
                    result = new TL_user();
                    break;
                case TL_user_layer216.constructor:
                    result = new TL_user_layer216();
                    break;
                case TL_user_layer199.constructor:
                    result = new TL_user_layer199();
                    break;
                case TL_user_layer196_2.constructor:
                    result = new TL_user_layer196_2();
                    break;
                case TL_user_layer196_1.constructor:
                    result = new TL_user_layer196_1();
                    break;
                case TL_user_layer195.constructor:
                    result = new TL_user_layer195();
                    break;
                case TL_user_layer185.constructor:
                    result = new TL_user_layer185();
                    break;
                case TL_user_layer184.constructor:
                    result = new TL_user_layer184();
                    break;
                case TL_user_layer166.constructor:
                    result = new TL_user_layer166();
                    break;
                case TL_userContact_old2.constructor:
                    result = new TL_userContact_old2();
                    break;
                case TL_userContact_old.constructor:
                    result = new TL_userContact_old();
                    break;
                case TL_userEmpty.constructor:
                    result = new TL_userEmpty();
                    break;
                case TL_user_layer165.constructor:
                    result = new TL_user_layer165();
                    break;
                case TL_user_layer159.constructor:
                    result = new TL_user_layer159();
                    break;
                case TL_user_layer147.constructor:
                    result = new TL_user_layer147();
                    break;
                case TL_user_layer144.constructor:
                    result = new TL_user_layer144();
                    break;
                case TL_user_layer131.constructor:
                    result = new TL_user_layer131();
                    break;
                case TL_user_layer104.constructor:
                    result = new TL_user_layer104();
                    break;
                case TL_userSelf_old.constructor:
                    result = new TL_userSelf_old();
                    break;
                case TL_userSelf_old3.constructor:
                    result = new TL_userSelf_old3();
                    break;
                case TL_userDeleted_old2.constructor:
                    result = new TL_userDeleted_old2();
                    break;
                case TL_userEmpty_layer131.constructor:
                    result = new TL_userEmpty_layer131();
                    break;
                case TL_userRequest_old.constructor:
                    result = new TL_userRequest_old();
                    break;
                case TL_userForeign_old.constructor:
                    result = new TL_userForeign_old();
                    break;
                case TL_userForeign_old2.constructor:
                    result = new TL_userForeign_old2();
                    break;
                case TL_userRequest_old2.constructor:
                    result = new TL_userRequest_old2();
                    break;
                case TL_userDeleted_old.constructor:
                    result = new TL_userDeleted_old();
                    break;
                case TL_user_layer65.constructor:
                    result = new TL_user_layer65();
                    break;
                case TL_user_old.constructor:
                    result = new TL_user_old();
                    break;
                case TL_userSelf_old2.constructor:
                    result = new TL_userSelf_old2();
                    break;
            }
            return TLdeserialize(User.class, result, stream, constructor, exception);
        }
    }

    public static class TL_userContact_old2 extends User {
        public static final int constructor = 0xcab35e18;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            username = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeString(username);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userContact_old extends TL_userContact_old2 {
        public static final int constructor = 0xf2fb8319;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt32(exception);
            first_name = stream.readString(exception);
            last_name = stream.readString(exception);
            access_hash = stream.readInt64(exception);
            phone = stream.readString(exception);
            photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt32((int) id);
            stream.writeString(first_name);
            stream.writeString(last_name);
            stream.writeInt64(access_hash);
            stream.writeString(phone);
            photo.serializeToStream(stream);
            status.serializeToStream(stream);
        }
    }

    public static class TL_userEmpty extends User {
        public static final int constructor = 0xd3bc4b7a;

        public void readParams(InputSerializedData stream, boolean exception) {
            id = stream.readInt64(exception);
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            stream.writeInt64(id);
        }
    }

    public static class TL_recentStory extends TLObject {
        public static final int constructor = 0x711d692d;
        
        public int flags;
        public boolean live;
        public int max_id;

        public static TL_recentStory TLdeserialize(InputSerializedData stream, int constructor, boolean exception) {
            final TL_recentStory result = TL_recentStory.constructor != constructor ? null : new TL_recentStory();
            return TLdeserialize(TL_recentStory.class, result, stream, constructor, exception);
        }

        @Override
        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = setFlag(flags, FLAG_0, live);
            stream.writeInt32(flags);
            if (hasFlag(flags, FLAG_1)) {
                stream.writeInt32(max_id);
            }
        }

        @Override
        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            live = hasFlag(flags, FLAG_0);
            if (hasFlag(flags, FLAG_1)) {
                max_id = stream.readInt32(exception);
            }
        }
    }

    public static class TL_user extends User {
        public static final int constructor = 0x31774388;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            bot_forum_view = hasFlag(flags2, FLAG_16);

            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = TL_recentStory.TLdeserialize(stream, stream.readInt32(exception), exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
            if ((flags2 & 16384) != 0) {
                bot_verification_icon = stream.readInt64(exception);
            }
            if ((flags2 & 32768) != 0) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            flags2 = setFlag(flags2, FLAG_16, bot_forum_view);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stories_max_id.serializeToStream(stream);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
            if ((flags2 & 16384) != 0) {
                stream.writeInt64(bot_verification_icon);
            }
            if ((flags2 & 32768) != 0) {
                stream.writeInt64(send_paid_messages_stars);
            }
        }
    }

    public static class TL_user_layer216 extends TL_user {
        public static final int constructor = 0x20b1422;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            bot_forum_view = hasFlag(flags2, FLAG_16);

            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
            if ((flags2 & 16384) != 0) {
                bot_verification_icon = stream.readInt64(exception);
            }
            if ((flags2 & 32768) != 0) {
                send_paid_messages_stars = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            flags2 = setFlag(flags2, FLAG_16, bot_forum_view);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
            if ((flags2 & 16384) != 0) {
                stream.writeInt64(bot_verification_icon);
            }
            if ((flags2 & 32768) != 0) {
                stream.writeInt64(send_paid_messages_stars);
            }
        }
    }

    public static class TL_user_layer199 extends TL_user {
        public static final int constructor = 0x4b46c37e;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
            if ((flags2 & 16384) != 0) {
                bot_verification_icon = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
            if ((flags2 & 16384) != 0) {
                stream.writeInt64(bot_verification_icon);
            }
        }
    }

    public static class TL_user_layer196_2 extends TL_user {
        public static final int constructor = 0x59476956;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
            if ((flags2 & 16384) != 0) {
                bot_verification_icon = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
            if ((flags2 & 16384) != 0) {
                stream.writeInt64(bot_verification_icon);
            }
        }
    }

    public static class TL_user_layer196_1 extends TL_user {
        public static final int constructor = 0x685f5e21;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
            if ((flags2 & 16384) != 0) {
                bot_verification_icon = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
            if ((flags2 & 16384) != 0) {
                stream.writeInt64(bot_verification_icon);
            }
        }
    }

    public static class TL_user_layer195 extends TL_user {
        public static final int constructor = 0x83314fca;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            bot_has_main_app = (flags2 & 8192) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            flags2 = bot_has_main_app ? (flags2 | 8192) : (flags2 &~ 8192);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
        }
    }

    public static class TL_user_layer185 extends TL_user {
        public static final int constructor = 0x4fe9cce0;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 4096) != 0) {
                bot_active_users = stream.readInt32(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
            if ((flags2 & 4096) != 0) {
                stream.writeInt32(bot_active_users);
            }
        }
    }

    public static class TL_user_layer184 extends TL_user {
        public static final int constructor = 0x215c4438;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            contact_require_premium = (flags2 & 1024) != 0;
            bot_business = (flags2 & 2048) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 256) != 0) {
                color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 512) != 0) {
                profile_color = PeerColor.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            flags2 = contact_require_premium ? (flags2 | 1024) : (flags2 &~ 1024);
            flags2 = bot_business ? (flags2 | 2048) : (flags2 &~ 2048);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 256) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.serializeToStream(stream);
            }
            if ((flags2 & 512) != 0) {
                if (profile_color == null) {
                    profile_color = new PeerColor();
                }
                profile_color.serializeToStream(stream);
            }
        }
    }

    public static class TL_user_layer166 extends TL_user {
        public static final int constructor = 0xeb602f25;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
            if ((flags2 & 128) != 0) {
                color = new PeerColor();
                color.color = stream.readInt32(exception);
            }
            if ((flags2 & 64) != 0) {
                if (color == null) {
                    color = new PeerColor();
                }
                color.background_emoji_id = stream.readInt64(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
            if ((flags2 & 128) != 0) {
                stream.writeInt32(color.color);
            }
            if ((flags2 & 64) != 0) {
                stream.writeInt64(color.background_emoji_id);
            }
        }
    }

    public static class TL_user_layer165 extends User {
        public static final int constructor = 0xabb5f120;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 8) != 0;
            stories_unavailable = (flags2 & 16) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
            try {
                if ((flags2 & 32) != 0) {
                    stories_max_id = new TL_recentStory();
                    stories_max_id.flags |= FLAG_1;
                    stories_max_id.max_id = stream.readInt32(exception);
                }
            } catch (Throwable e) {
                FileLog.e(e);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 8) : (flags2 &~ 8);
            flags2 = stories_unavailable ? (flags2 | 16) : (flags2 &~ 16);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
            if ((flags2 & 32) != 0) {
                stream.writeInt32(stories_max_id == null ? 0 : stories_max_id.max_id);
            }
        }
    }

    public static class TL_user_layer159 extends User {
        public static final int constructor = 0x8f97c628;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            flags2 = stream.readInt32(exception);
            bot_can_edit = (flags2 & 2) != 0;
            close_friend = (flags2 & 4) != 0;
            stories_hidden = (flags2 & 32) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags2 & 1) != 0) {
                usernames = Vector.deserialize(stream, TL_username::TLdeserialize, exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            if (username == null) {
                flags = flags & ~8;
            }
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            flags2 = bot_can_edit ? (flags2 | 2) : (flags2 &~ 2);
            flags2 = close_friend ? (flags2 | 4) : (flags2 &~ 4);
            flags2 = stories_hidden ? (flags2 | 32) : (flags2 &~ 32);
            stream.writeInt32(flags);
            stream.writeInt32(flags2);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
            if ((flags2 & 1) != 0) {
                Vector.serialize(stream, usernames);
            }
        }
    }

    public static class TL_user_layer147 extends User {
        public static final int constructor = 0x5d99adee;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            attach_menu_enabled = (flags & 536870912) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status = EmojiStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            flags = attach_menu_enabled ? (flags | 536870912) : (flags &~ 536870912);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
            if ((flags & 1073741824) != 0) {
                emoji_status.serializeToStream(stream);
            }
        }
    }

    public static class TL_user_layer144 extends User {
        public static final int constructor = 0x3ff6ecb0;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            bot_attach_menu = (flags & 134217728) != 0;
            premium = (flags & 268435456) != 0;
            id = stream.readInt64(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            flags = bot_attach_menu ? (flags | 134217728) : (flags &~ 134217728);
            flags = premium ? (flags | 268435456) : (flags &~ 268435456);
            stream.writeInt32(flags);
            stream.writeInt64(id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
        }
    }

    public static class TL_user_layer131 extends TL_user {
        public static final int constructor = 0x938458c1;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            apply_min_photo = (flags & 33554432) != 0;
            fake = (flags & 67108864) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                username = stream.readString(exception);
            }
            if ((flags & 16) != 0) {
                phone = stream.readString(exception);
            }
            if ((flags & 32) != 0) {
                photo = UserProfilePhoto.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 64) != 0) {
                status = UserStatus.TLdeserialize(stream, stream.readInt32(exception), exception);
            }
            if ((flags & 16384) != 0) {
                bot_info_version = stream.readInt32(exception);
            }
            if ((flags & 262144) != 0) {
                restriction_reason = Vector.deserialize(stream, RestrictionReason::TLdeserialize, exception);
            }
            if ((flags & 524288) != 0) {
                bot_inline_placeholder = stream.readString(exception);
            }
            if ((flags & 4194304) != 0) {
                lang_code = stream.readString(exception);
            }
        }

        public void serializeToStream(OutputSerializedData stream) {
            stream.writeInt32(constructor);
            flags = self ? (flags | 1024) : (flags &~ 1024);
            flags = contact ? (flags | 2048) : (flags &~ 2048);
            flags = mutual_contact ? (flags | 4096) : (flags &~ 4096);
            flags = deleted ? (flags | 8192) : (flags &~ 8192);
            flags = bot ? (flags | 16384) : (flags &~ 16384);
            flags = bot_chat_history ? (flags | 32768) : (flags &~ 32768);
            flags = bot_nochats ? (flags | 65536) : (flags &~ 65536);
            flags = verified ? (flags | 131072) : (flags &~ 131072);
            flags = restricted ? (flags | 262144) : (flags &~ 262144);
            flags = min ? (flags | 1048576) : (flags &~ 1048576);
            flags = bot_inline_geo ? (flags | 2097152) : (flags &~ 2097152);
            flags = support ? (flags | 8388608) : (flags &~ 8388608);
            flags = scam ? (flags | 16777216) : (flags &~ 16777216);
            flags = apply_min_photo ? (flags | 33554432) : (flags &~ 33554432);
            flags = fake ? (flags | 67108864) : (flags &~ 67108864);
            stream.writeInt32(flags);
            stream.writeInt32((int) id);
            if ((flags & 1) != 0) {
                stream.writeInt64(access_hash);
            }
            if ((flags & 2) != 0) {
                stream.writeString(first_name);
            }
            if ((flags & 4) != 0) {
                stream.writeString(last_name);
            }
            if ((flags & 8) != 0) {
                stream.writeString(username);
            }
            if ((flags & 16) != 0) {
                stream.writeString(phone);
            }
            if ((flags & 32) != 0) {
                photo.serializeToStream(stream);
            }
            if ((flags & 64) != 0) {
                status.serializeToStream(stream);
            }
            if ((flags & 16384) != 0) {
                stream.writeInt32(bot_info_version);
            }
            if ((flags & 262144) != 0) {
                Vector.serialize(stream, restriction_reason);
            }
            if ((flags & 524288) != 0) {
                stream.writeString(bot_inline_placeholder);
            }
            if ((flags & 4194304) != 0) {
                stream.writeString(lang_code);
            }
        }
    }

    public static class TL_user_layer104 extends TL_user {
        public static final int constructor = 0x2e13f4c3;

        public void readParams(InputSerializedData stream, boolean exception) {
            flags = stream.readInt32(exception);
            self = (flags & 1024) != 0;
            contact = (flags & 2048) != 0;
            mutual_contact = (flags & 4096) != 0;
            deleted = (flags & 8192) != 0;
            bot = (flags & 16384) != 0;
            bot_chat_history = (flags & 32768) != 0;
            bot_nochats = (flags & 65536) != 0;
            verified = (flags & 131072) != 0;
            restricted = (flags & 262144) != 0;
            min = (flags & 1048576) != 0;
            bot_inline_geo = (flags & 2097152) != 0;
            support = (flags & 8388608) != 0;
            scam = (flags & 16777216) != 0;
            id = stream.readInt32(exception);
            if ((flags & 1) != 0) {
                access_hash = stream.readInt64(exception);
            }
            if ((flags & 2) != 0) {
                first_name = stream.readString(exception);
            }
            if ((flags & 4) != 0) {
                last_name = stream.readString(exception);
            }
            if ((flags & 8) != 0) {
                