import 'dart:async';

import 'package:encointer_wallet/models/index.dart';
import 'package:ew_log/ew_log.dart';
import 'package:ew_polkadart/ew_polkadart.dart';

/// Api to talk to an substrate node via the websocket protocol.
///
/// Once connected, a websocket channel is maintained until closed by either side.
class SubstrateDartApi {
  SubstrateDartApi(this._provider);

  /// Websocket client used to connect to the node.
  final Provider _provider;

  /// Returns the rpc nodes of the connected node or an empty list otherwise.
  Future<RpcMethods> rpcMethods() async {
    return rpc<Map<String, dynamic>>('rpc_methods', []).then(RpcMethods.fromJson);
  }

  Future<void> connect(String endpoint) async {
    try {
      // Sanity check that we are running against valid node with offchain indexing enabled
      if (!(await offchainIndexingEnabled())) {
        Log.d(
          "rpc_methods does not contain 'getReputations'. Are the following flags passed"
              " to the node? \n '--enable-offchain-indexing true --rpc-methods unsafe'",
          'SubstrateDartApi',
        );
      }
    } catch (e) {
      Log.e('RPC error $e', 'SubstrateDartApi');
    }
  }

  Future<bool> offchainIndexingEnabled() async {
    try {
      final communities = await rpc<List<dynamic>>('encointer_getAllCommunities', []);
      if (communities.isEmpty) {
        // Unfortunately, we cannot test the same way for reputations, as empty
        // results are valid.
        Log.e(
          '[offchainIndexingEnabled] Enabled, but no communities found. We have a caching issue!',
          'SubstrateDartApi',
        );
        return Future.value(false);
      }

      return Future.value(true);
    } catch (e) {
      Log.d('[offchainIndexingEnabled] rpc not supported: $e', 'SubstrateDartApi');
      return Future.value(false);
    }
  }

  Future<bool> newTreasuryRpcSupported() async {
    try {
      // Check treasury account. This will return an exception if the rpc is not supported.
      await rpc<String>('encointer_getCommunityTreasuryAccountUnchecked', []);

      return Future.value(true);
    } catch (e) {
      Log.d('[newTreasuryRpcSupported] rpc not supported: $e', 'SubstrateDartApi');
      return Future.value(false);
    }
  }

  /// Queries the rpc of the node.
  ///
  /// Hints:
  /// * account ids must be passed as SS58.
  Future<T> rpc<T>(String method, List<dynamic> params) async {
    Log.d('[DartApi] $method: $params');

    final response = await _provider.send(method, params);
    Log.d('[DartApi] Response Error: ${response.error}');
    Log.d('[DartApi] Response Result: ${response.result}');

    if (response.error != null) throw Exception(response.error);

    final data = response.result! as T;
    return data;
  }
}
