import 'package:flutter/cupertino.dart';
import 'package:flutter/material.dart';
import 'package:iconsax/iconsax.dart';
import 'package:provider/provider.dart';

import 'package:encointer_wallet/common/components/rounded_button.dart';
import 'package:encointer_wallet/page/qr_scan/qr_codes/index.dart';
import 'package:encointer_wallet/modules/modules.dart';
import 'package:encointer_wallet/page/qr_scan/qr_scan_page.dart';
import 'package:encointer_wallet/service/substrate_api/api.dart';
import 'package:encointer_wallet/store/app.dart';
import 'package:encointer_wallet/utils/format.dart';
import 'package:ew_l10n/l10n.dart';
import 'package:ew_keyring/ew_keyring.dart';
import 'package:ew_test_keys/ew_test_keys.dart';

class ContactPage extends StatefulWidget {
  const ContactPage({super.key});

  static const String route = '/profile/contact';

  @override
  State<ContactPage> createState() => _Contact();
}

class _Contact extends State<ContactPage> {
  final _formKey = GlobalKey<FormState>();

  final TextEditingController _addressCtrl = TextEditingController();
  final TextEditingController _nameCtrl = TextEditingController();
  final TextEditingController _memoCtrl = TextEditingController();

  bool? _isObservation = false;

  ContactData? qrScanData;

  bool _submitting = false;

  Future<void> _onSave() async {
    if (_formKey.currentState!.validate()) {
      setState(() {
        _submitting = true;
      });
      final l10n = context.l10n;
      final addr = _addressCtrl.text.replaceAll(' ', '');
      final pubKey = AddressUtils.addressToPubKeyHex(addr);

      final con = {
        'address': addr,
        'name': _nameCtrl.text,
        'memo': _memoCtrl.text,
        'observation': _isObservation,
        'pubKey': pubKey,
      };
      setState(() {
        _submitting = false;
      });
      if (qrScanData == null) {
        // create new contact
        final exist = context.read<AppStore>().settings.contactList.indexWhere((i) => i.pubKey == pubKey);
        if (exist > -1) {
          return showCupertinoDialog<void>(
            context: context,
            builder: (BuildContext context) {
              return CupertinoAlertDialog(
                title: Container(),
                content: Text(l10n.contactAlreadyExists),
                actions: <Widget>[
                  CupertinoButton(
                    child: Text(context.l10n.ok),
                    onPressed: () => Navigator.of(context).pop(),
                  ),
                ],
              );
            },
          );
        } else {
          await context.read<AppStore>().settings.addContact(con);
        }
      } else {
        // edit contact
        await context.read<AppStore>().settings.updateContact(con);
      }

      // get contact info
      if (!_isObservation!) {
        // if this address was used as observation and current account,
        // we need to change current account
        if (pubKey == context.read<AppStore>().account.currentAccountPubKey) {
          await webApi.account.changeCurrentAccount(fetchData: true);
        }
      }
      Navigator.of(context).pop();
    }
  }

  @override
  void dispose() {
    _addressCtrl.dispose();
    _nameCtrl.dispose();
    _memoCtrl.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final qrScanData = ModalRoute.of(context)!.settings.arguments as ContactData?;
    final l10n = context.l10n;
    if (qrScanData != null) {
      _addressCtrl.text = qrScanData.account;
      _nameCtrl.text = qrScanData.label;
    }

    return Scaffold(
      appBar: AppBar(
        title: Text(l10n.addressBook),
      ),
      body: SafeArea(
        child: Column(
          children: <Widget>[
            Expanded(
              child: Form(
                key: _formKey,
                child: ListView(
                  padding: const EdgeInsets.only(top: 8, bottom: 8),
                  children: <Widget>[
                    Padding(
                      padding: const EdgeInsets.only(left: 16, right: 16),
                      child: TextFormField(
                        key: const Key(EWTestKeys.contactAddress),
                        decoration: InputDecoration(
                          hintText: l10n.contactAddress,
                          labelText: l10n.contactAddress,
                        ),
                        controller: _addressCtrl,
                        validator: (v) {
                          if (!Fmt.isAddress(v!.replaceAll(' ', ''))) {
                            return l10n.contactAddressError;
                          }
                          return null;
                        },
                        readOnly: qrScanData != null,
                      ),
                    ),
                    Padding(
                      padding: const EdgeInsets.only(left: 16, right: 16),
                      child: TextFormField(
                        key: const Key(EWTestKeys.contactName),
                        decoration: InputDecoration(
                          hintText: l10n.contactName,
                          labelText: l10n.contactName,
                        ),
                        controller: _nameCtrl,
                        validator: (v) {
                          return v!.trim().isNotEmpty ? null : l10n.contactNameError;
                        },
                      ),
                    ),
                    if (context.select<AppSettings, bool>((store) => store.developerMode))
                      Padding(
                        padding: const EdgeInsets.only(left: 16, right: 16),
                        child: TextFormField(
                          decoration: InputDecoration(
                            hintText: l10n.contactMemo,
                            labelText: l10n.contactMemo,
                          ),
                          controller: _memoCtrl,
                        ),
                      ),
                    if (context.select<AppSettings, bool>((store) => store.developerMode))
                      Row(
                        children: <Widget>[
                          Checkbox(
                            value: _isObservation,
                            onChanged: (v) {
                              setState(() {
                                _isObservation = v;
                              });
                            },
                          ),
                          GestureDetector(
                            child: Text(context.l10n.observe),
                            onTap: () {
                              setState(() {
                                _isObservation = !_isObservation!;
                              });
                            },
                          ),
                          Tooltip(
                            triggerMode: TooltipTriggerMode.tap,
                            message: context.l10n.observeBrief,
                            child: const Padding(
                              padding: EdgeInsets.only(left: 8),
                              child: Icon(Icons.info_outline, size: 16),
                            ),
                          ),
                        ],
                      ),
                    const SizedBox(height: 24),
                    IconButton(
                      key: const Key(EWTestKeys.scanBarcode),
                      iconSize: 48,
                      icon: const Icon(Iconsax.scan_barcode),
                      onPressed: () async {
                        await Navigator.of(context).popAndPushNamed(
                          ScanPage.route,
                          arguments: ScanPageParams(scannerContext: QrScannerContext.contactsPage),
                        );
                      },
                    ),
                    const SizedBox(height: 24),
                  ],
                ),
              ),
            ),
            Container(
              margin: const EdgeInsets.all(16),
              child: RoundedButton(
                key: const Key(EWTestKeys.contactSave),
                submitting: _submitting,
                text: l10n.contactSave,
                onPressed: _onSave,
              ),
            ),
          ],
        ),
      ),
    );
  }
}
