import 'dart:math';

import 'package:ew_l10n/l10n.dart';
import 'package:encointer_wallet/models/communities/community_identifier.dart';
import 'package:encointer_wallet/page-encointer/democracy/utils/asset_id.dart';

import 'package:ew_polkadart/encointer_types.dart' as et;

/// Enum for Scope selection
enum ProposalScope { global, local }

extension ProposalScopeExt on ProposalScope {
  bool get isGlobal => this == ProposalScope.global;

  bool get isLocal => this == ProposalScope.local;

  String localizedStr(AppLocalizations l10n, String community) {
    return switch (this) {
      ProposalScope.global => l10n.proposalScopeGlobal,
      ProposalScope.local => l10n.proposalScopeLocal(community),
    };
  }
}

/// Enum representing different proposal actions
enum ProposalActionIdentifier {
  addLocation,
  removeLocation,
  updateDemurrage,
  updateNominalIncome,
  setInactivityTimeout,
  petition,
  spendNative,
  issueSwapNativeOption,
  spendAsset,
  issueSwapAssetOption,
}

ProposalActionIdentifier proposalActionIdentifierFromPolkadartAction(et.ProposalAction action) {
  return switch (action.runtimeType) {
    et.AddLocation => ProposalActionIdentifier.addLocation,
    et.RemoveLocation => ProposalActionIdentifier.removeLocation,
    et.UpdateDemurrage => ProposalActionIdentifier.updateDemurrage,
    et.UpdateNominalIncome => ProposalActionIdentifier.updateNominalIncome,
    et.SetInactivityTimeout => ProposalActionIdentifier.setInactivityTimeout,
    et.Petition => ProposalActionIdentifier.petition,
    et.SpendNative => ProposalActionIdentifier.spendNative,
    et.IssueSwapNativeOption => ProposalActionIdentifier.issueSwapNativeOption,
    et.SpendAsset => ProposalActionIdentifier.spendAsset,
    et.IssueSwapAssetOption => ProposalActionIdentifier.issueSwapAssetOption,
    _ => throw UnimplementedError('Invalid Proposal Id Type'),
  };
}

class ProposalActionIdWithScope {
  ProposalActionIdWithScope(this.actionId, this.cid);

  factory ProposalActionIdWithScope.fromProposalAction(et.ProposalAction action) {
    return ProposalActionIdWithScope(
        proposalActionIdentifierFromPolkadartAction(action), getCidFromPolkadartAction(action));
  }

  final ProposalActionIdentifier actionId;
  final CommunityIdentifier? cid;
}

CommunityIdentifier? getCidFromPolkadartAction(et.ProposalAction action) {
  final maybeCid = switch (action.runtimeType) {
    et.AddLocation => (action as et.AddLocation).value0,
    et.RemoveLocation => (action as et.RemoveLocation).value0,
    et.UpdateDemurrage => (action as et.UpdateDemurrage).value0,
    et.UpdateNominalIncome => (action as et.UpdateNominalIncome).value0,
    et.SetInactivityTimeout => null,
    et.Petition => (action as et.Petition).value0,
    et.SpendNative => (action as et.SpendNative).value0,
    et.IssueSwapNativeOption => (action as et.IssueSwapNativeOption).value0,
    et.SpendAsset => (action as et.SpendAsset).value0,
    et.IssueSwapAssetOption => (action as et.IssueSwapAssetOption).value0,
    _ => throw UnimplementedError('Invalid Proposal Id Type'),
  };

  return maybeCid != null ? CommunityIdentifier.fromPolkadart(maybeCid) : null;
}

/// Removing locations is not supported, as this needs some more complex logic:
/// https://github.com/encointer/encointer-wallet-flutter/issues/1757
List<ProposalActionIdentifier> supportedProposalIds(bool developerMode) {
  return [
    ProposalActionIdentifier.addLocation,
    ProposalActionIdentifier.updateDemurrage,
    ProposalActionIdentifier.updateNominalIncome,
    ProposalActionIdentifier.setInactivityTimeout,
    ProposalActionIdentifier.petition,
    ProposalActionIdentifier.spendNative,
    if (developerMode) ProposalActionIdentifier.issueSwapNativeOption,
    if (developerMode) ProposalActionIdentifier.spendAsset,
    ProposalActionIdentifier.issueSwapAssetOption,
  ];
}

extension PropsalActionExt on ProposalActionIdentifier {
  /// Returns the allowed proposal policies corresponding
  /// to a specific proposal variant.
  List<ProposalScope> allowedPolicies() {
    return switch (this) {
      // Only global proposal allowed
      ProposalActionIdentifier.setInactivityTimeout => [ProposalScope.global],

      // Only local proposals allowed
      ProposalActionIdentifier.addLocation => [ProposalScope.local],
      ProposalActionIdentifier.removeLocation => [ProposalScope.local],
      ProposalActionIdentifier.updateDemurrage => [ProposalScope.local],
      ProposalActionIdentifier.updateNominalIncome => [ProposalScope.local],
      ProposalActionIdentifier.issueSwapNativeOption => [ProposalScope.local],

      // Global or Local allowed (first value is the default)
      ProposalActionIdentifier.petition => [ProposalScope.local, ProposalScope.global],
      ProposalActionIdentifier.spendNative => [ProposalScope.local, ProposalScope.global],
      ProposalActionIdentifier.spendAsset => [ProposalScope.local, ProposalScope.global],

      // Only local for now, but later global might be possible
      ProposalActionIdentifier.issueSwapAssetOption => [ProposalScope.local],
    };
  }

  String localizedStr(AppLocalizations l10n, String cidSymbol, AssetToSpend asset) {
    return switch (this) {
      ProposalActionIdentifier.addLocation => l10n.proposalTypeAddLocation,
      ProposalActionIdentifier.removeLocation => l10n.proposalTypeRemoveLocation,
      ProposalActionIdentifier.updateDemurrage => l10n.proposalTypeUpdateDemurrage,
      ProposalActionIdentifier.updateNominalIncome => l10n.proposalTypeUpdateNominalIncome,
      ProposalActionIdentifier.setInactivityTimeout => l10n.proposalTypeSetInactivityTimeout,
      ProposalActionIdentifier.petition => l10n.proposalTypePetition,
      ProposalActionIdentifier.spendNative => l10n.proposalTypeSpendNative,
      ProposalActionIdentifier.issueSwapNativeOption => l10n.proposalTypeIssueSwapNativeOption(cidSymbol),
      ProposalActionIdentifier.spendAsset => l10n.proposalTypeSpendAsset(asset.symbol),
      ProposalActionIdentifier.issueSwapAssetOption => l10n.proposalTypeIssueSwapAssetOption(cidSymbol, asset.symbol)
    };
  }

  bool isSwapAction() {
    return this == ProposalActionIdentifier.issueSwapNativeOption ||
        this == ProposalActionIdentifier.issueSwapAssetOption;
  }
}

bool hasSameProposalForSameScope(
    List<ProposalActionIdWithScope> actions, ProposalActionIdentifier id, CommunityIdentifier? scope) {
  return actions.any((action) => action.actionId == id && action.cid == scope);
}

double monthlyDemurragePercentToDemurrage(double monthly, BigInt blockProductionTime) {
  final blocks = blocksPerMonth(blockProductionTime);
  return -log(1 - (monthly / 100)) / blocks;
}

double blocksPerMonth(BigInt blockProductionTime) {
  return (86400 / blockProductionTime.toDouble()) * (365 / 12);
}
