import 'dart:convert';
import 'dart:typed_data';

import 'package:pointycastle/digests/sha256.dart';
import 'package:pointycastle/digests/sha512.dart';
import 'package:pointycastle/key_derivators/api.dart';
import 'package:pointycastle/key_derivators/pbkdf2.dart';
import 'package:pointycastle/macs/hmac.dart';
import 'package:unorm_dart/unorm_dart.dart';

/// BIP39: To create a binary seed from the mnemonic, we use the PBKDF2 function with a mnemonic sentence (in UTF-8 NFKD) used as the password and the string "mnemonic" + passphrase (again in UTF-8 NFKD) used as the salt.
/// The iteration count is set to 2048 and HMAC-SHA512 is used as the pseudo-random function.
/// The length of the derived key is 512 bits (= 64 bytes).
List<int> pbkdf2(String sentence, {String passphrase = ""}) {
  const blockLength = 128;
  const iterationCount = 2048;
  const desiredKeyLength = 64;
  String nfkdPassphrase = nfkd("mnemonic$passphrase");
  String nfkdSentence = nfkd(sentence);
  final salt = Uint8List.fromList(utf8.encode(nfkdPassphrase));
  final derivator = PBKDF2KeyDerivator(HMac(SHA512Digest(), blockLength));
  derivator.reset();
  derivator.init(Pbkdf2Parameters(salt, iterationCount, desiredKeyLength));
  Uint8List result = derivator.process(
    Uint8List.fromList(utf8.encode(nfkdSentence)),
  );
  return result.toList();
}

/// BIP39: A checksum is generated by taking the first bits of its SHA256 hash.
List<int> sha256(List<int> bytes) {
  final digest = SHA256Digest();
  Uint8List result = digest.process(Uint8List.fromList(bytes));
  return result.toList();
}
