package org.eagsoftware.laundrynotes.adapters;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.view.ContextMenu;
import android.view.LayoutInflater;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import org.eagsoftware.laundrynotes.GestoreFileImm;
import org.eagsoftware.laundrynotes.R;
import org.eagsoftware.laundrynotes.Utilities;
import org.eagsoftware.laundrynotes.activities.DetailActivity;
import org.eagsoftware.laundrynotes.activities.MainActivity;
import org.eagsoftware.laundrynotes.data.ArticoloEntity;
import org.eagsoftware.laundrynotes.data.LavanderiaDB;
import com.google.android.material.snackbar.Snackbar;

import java.util.ArrayList;
import java.util.List;

public class MainRecyclerAdapter    extends RecyclerView.Adapter<MainRecyclerAdapter.ViewHolder>
                                    implements View.OnCreateContextMenuListener {
    private final Context activityContext;
    private List<ArticoloEntity> lista;
    private final List<ArticoloEntity> listaCompleta;

    public MainRecyclerAdapter(Context activityContext, List<ArticoloEntity> lista){
        this.activityContext = activityContext;
        this.lista = lista;
        listaCompleta = new ArrayList<>(lista); // crea una copia della lista che rimarrà intatta
    }

    @NonNull
    @Override
    public MainRecyclerAdapter.ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        // Inflate del layout elemento lista
        View view = LayoutInflater.from(parent.getContext()).inflate(R.layout.recycler_list_item, parent,
                false);
        return new ViewHolder(view);
    }

    @Override
    public void onBindViewHolder(@NonNull MainRecyclerAdapter.ViewHolder holder, int position) {
        // Associazione dati con view elemento lista
        ArticoloEntity elementoLista =  lista.get(position);

        holder.txwNome.setText(elementoLista.getNome());
        int iconaID =
                Utilities.getArrayListID(activityContext, R.array.tipologia_icone).get(elementoLista.getTipologia());
        if(iconaID != -1) holder.imwIcona.setImageResource(iconaID);

        // Gestisce il click sull'item
        holder.itemView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                Intent itnShowDetAct = new Intent(activityContext, DetailActivity.class);
                itnShowDetAct.putExtra("objArticolo", elementoLista);
                activityContext.startActivity(itnShowDetAct);
                ((Activity)activityContext).overridePendingTransition(R.anim.slide_in_right, R.anim.fade_out);
            }
        });

        // Salva l'articolo selezionato come tag (per passarlo alla gestione del context menu)
        holder.itemView.setTag(elementoLista);
        // Invoca la creazione del context menu
        holder.itemView.setOnCreateContextMenuListener(this);
    }

    @Override
    public void onCreateContextMenu(ContextMenu contextMenu, View view, ContextMenu.ContextMenuInfo contextMenuInfo) {
        MainActivity mainActivity = (MainActivity) view.getContext();
        MenuInflater iflCtxMenu = mainActivity.getMenuInflater();
        iflCtxMenu.inflate(R.menu.context_recycler, contextMenu);
        ArticoloEntity ctxArtSel = (ArticoloEntity) view.getTag();

        MenuItem mniElimina = contextMenu.findItem(R.id.cmi_rcw_elimina);
        mniElimina.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(@NonNull MenuItem menuItem) {
                View viewRcy = mainActivity.findViewById(R.id.rcw_main_lista);
                Snackbar.make(viewRcy, R.string.conferma_elimina, Snackbar.LENGTH_LONG)
                        .setAction(R.string.conferma, new View.OnClickListener(){
                            @Override
                            public void onClick(View view) {
                                asyncEliminaArticolo(mainActivity, ctxArtSel, viewRcy);
                            }
                        }).show();
                return true;
            }
        });
    }

    @Override
    public int getItemCount() {
        return lista.size();
    }

    private void asyncEliminaArticolo(MainActivity mainActivity, ArticoloEntity ctxArtSel, View viewRcy){
        new Thread(new Runnable() {
            @Override
            public void run() {
                if (ctxArtSel == null) throw new NullPointerException
                        ("Oggetto ArticoloEntity non definito");
                LavanderiaDB.getInstance(mainActivity.getApplicationContext()).getArticoloDAO()
                        .eliminaArticolo(ctxArtSel);
                mainActivity.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        if(ctxArtSel.getImmURI() != null) {
                            // Elimina il file immagine
                            GestoreFileImm gfi = new GestoreFileImm(mainActivity.getApplicationContext(),
                                    mainActivity.getApplicationContext().getPackageName() + ".fileprovider",
                                    mainActivity.getFilesDir(),
                                    mainActivity);
                            gfi.eliminaFile(Uri.parse(ctxArtSel.getImmURI()));
                        }
                        // Aggiorna recyclerView
                        notifyItemRemoved(lista.indexOf(ctxArtSel));
                        listaCompleta.remove(ctxArtSel);
                        lista.remove(ctxArtSel);
                        Snackbar.make(viewRcy, R.string.articolo_eliminato, Snackbar.LENGTH_SHORT).show();
                    }
                });
            }
        }).start();
    }

    @SuppressLint("NotifyDataSetChanged")
    public void filtraByNome(String nome){
        lista.clear();
        if(!nome.isEmpty()) {
            nome = nome.toLowerCase();
            for (ArticoloEntity articolo : listaCompleta){
                if(articolo.getNome().toLowerCase().contains(nome)) lista.add(articolo);
            }
        } else lista = new ArrayList<>(listaCompleta);
        notifyDataSetChanged();
    }

    @SuppressLint("NotifyDataSetChanged")
    public void filtraByTipo(int tipologia){
        lista.clear();
        if(tipologia != 0) {
            for (ArticoloEntity articolo : listaCompleta) {
                if (articolo.getTipologia() == tipologia) lista.add(articolo);
            }
        } else lista = new ArrayList<>(listaCompleta);
        notifyDataSetChanged();
    }


    public static class ViewHolder extends RecyclerView.ViewHolder{
        final ImageView imwIcona;
        final TextView txwNome;

        public ViewHolder(@NonNull View itemView) {
            super(itemView);
            imwIcona = itemView.findViewById(R.id.imw_ril_icona);
            txwNome = itemView.findViewById(R.id.txw_ril_nome);
        }
    }
}
