package org.eagsoftware.laundrynotes;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.media.ExifInterface;
import android.net.Uri;
import android.util.Log;

import androidx.core.content.FileProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Objects;

public class GestoreFileImm {
    private final Context context;
    private final String providerAutority;
    private final File filesDir;
    private final OnErrorListener errorListener;


    public GestoreFileImm(OnErrorListener listener){
        this.errorListener = listener;
        this.context = null;
        this.providerAutority = null;
        this.filesDir = null;
    }


    public GestoreFileImm(Context context, String providerAutority, File filesDir, OnErrorListener listener) {
        this.context = context;
        this.providerAutority = providerAutority;
        Log.d("DEVELOP", providerAutority);
        this.filesDir = filesDir;
        this.errorListener = listener;
    }


    private void notifyError(Exception exc) {
        if (errorListener != null) errorListener.onError(exc);
    }


    public void svuotaCartella(File dir){
        try {
            if (dir.exists() && dir.isDirectory()) {
                File[] files = dir.listFiles();
                if (files != null) {
                    for (File file : files) {
                        if (file.isDirectory()) svuotaCartella(file);
                        else if (!file.delete()) throw new IOException("Impossibile eliminare file");
                    }
                }
            }
        } catch (IOException exc) {notifyError(exc);}
    }


    /** Per la creazione di un file immagine (vuoto) nella directory images */
    public Uri creaFileTemp(String tempPath, String nomeFile) {
        try {
            File storageDir = new File(filesDir, tempPath);
            if (!storageDir.exists()) if (!storageDir.mkdirs())
                throw new IOException("Impossibile creare cartella");
            File tempFile = File.createTempFile(nomeFile + "_", ".jpg", storageDir);
            tempFile.deleteOnExit();    // Elimina il file alla chiusura dell'app
            return FileProvider.getUriForFile(context, providerAutority, tempFile);
        } catch (IOException exc){notifyError(exc);}
        return null;
    }

    public void eliminaFile(Uri fUri){
        try {
            File file = new File(filesDir + Objects.requireNonNull(fUri.getPath()));
            if (file.exists()) {if(!file.delete()) throw new IOException("File non cancellato");}
            else throw new IOException("Il file non esiste");
        } catch (IOException exc) {notifyError(exc);}
    }

    /** Sposta il file  */
    public Uri promuoviImmagine(Uri fUri, String defPath, String defName){
        Uri newUri = null;
        try {
            File tempFile = new File(filesDir, Objects.requireNonNull(fUri.getPath()));
            File storageDir = new File(filesDir, defPath);
            if (!storageDir.exists()) if (!storageDir.mkdirs())
                throw new IOException("Impossibile creare cartella");
            File defFile = new File(storageDir, defName + ".jpg");
            if(!defFile.exists()) if(!defFile.createNewFile()) throw new IOException("File non creato");
            copiaFile(tempFile, defFile);
            eliminaFile(fUri);
            newUri = FileProvider.getUriForFile(context, providerAutority, defFile);
        } catch (IOException exc) {notifyError(exc);}
        return newUri;
    }


    public boolean esisteFile(Uri fUri){
        File file = new File(filesDir, Objects.requireNonNull(fUri.getPath()));
        return file.exists();
    }


    private void copiaFile(File origine, File dest) {
        // try-with-resources chiude automaticamente gli stream
        try (
                FileInputStream in = new FileInputStream(origine);
                FileOutputStream out = new FileOutputStream(dest)
        ) {
            if (origine == null || dest == null)
                throw new IllegalArgumentException("I file origine e destinazione non possono essere null");
            if (!origine.exists())
                throw new FileNotFoundException("Il file di origine non esiste: " + origine.getAbsolutePath());
            if (!dest.exists())
                throw new FileNotFoundException("Il file di destinazione non esiste: " + dest.getAbsolutePath());

            Utilities.copyStream(in, out);
        } catch (Exception e) {
            notifyError(e);
        }
    }


    // Restituisce un budget prudenziale di memoria in byte
    private long getSafeBudgetBytes(@SuppressWarnings("SameParameterValue") float fraction) {
        long maxHeap = Runtime.getRuntime().maxMemory();
        // usa solo una frazione (es. 25%)
        return (long)(maxHeap * fraction);
    }

    // Calcola il numero massimo di pixel consentiti
    private long getMaxPixels(long budgetBytes, @SuppressWarnings("SameParameterValue") boolean useArgb8888) {
        int bpp = useArgb8888 ? 4 : 2;
        long maxPixels = budgetBytes / bpp;
        // tetto massimo di sicurezza (es. 16 Mpx)
        return Math.min(maxPixels, 16_000_000);
    }

    // Calcola dimensioni target coerenti con aspect ratio
    private int[] calculateTargetSize(int origW, int origH, long maxPixels) {
        if (origW <= 0 || origH <= 0) return new int[]{0,0};
        double aspect = (double) origW / (double) origH;

        int targetW = (int)Math.max(1, Math.round(Math.sqrt(maxPixels * aspect)));
        int targetH = (int)Math.max(1, Math.round(Math.sqrt(maxPixels / aspect)));

        // Non superare l’originale
        targetW = Math.min(targetW, origW);
        targetH = Math.min(targetH, origH);

        return new int[]{targetW, targetH};
    }


    private int calculateInSampleSize(BitmapFactory.Options options, int reqWidth, int reqHeight) {
        final int width = options.outWidth;
        final int height = options.outHeight;
        int inSampleSize = 1;

        if (height > reqHeight || width > reqWidth) {
            int halfWidth = width / 2;
            int halfHeight = height / 2;
            while ((halfWidth / inSampleSize) >= reqWidth
                    && (halfHeight / inSampleSize) >= reqHeight) {
                inSampleSize *= 2;
            }
        }
        return Math.max(inSampleSize, 1);
    }


    private Bitmap applyExifOrientation(Uri sourceUri, Bitmap bitmap) throws Exception {
        if (sourceUri == null || bitmap == null)
            throw new IllegalArgumentException("I parametri sourceUri e bitmap non possono essere null");

        try (InputStream in = context.getContentResolver().openInputStream(sourceUri)) {
            if (in == null) throw new IOException("inputStream è null.");
            ExifInterface exif = new ExifInterface(in);
            int orientation = exif.getAttributeInt(
                    ExifInterface.TAG_ORIENTATION,
                    ExifInterface.ORIENTATION_NORMAL);

            Matrix matrix = new Matrix();
            switch (orientation) {
                case ExifInterface.ORIENTATION_ROTATE_90: matrix.postRotate(90); break;
                case ExifInterface.ORIENTATION_ROTATE_180: matrix.postRotate(180); break;
                case ExifInterface.ORIENTATION_ROTATE_270: matrix.postRotate(270); break;
                case ExifInterface.ORIENTATION_FLIP_HORIZONTAL: matrix.postScale(-1, 1); break;
                case ExifInterface.ORIENTATION_FLIP_VERTICAL: matrix.postScale(1, -1); break;
                case ExifInterface.ORIENTATION_TRANSPOSE:
                    matrix.postRotate(90); matrix.postScale(-1, 1);
                    break;
                case ExifInterface.ORIENTATION_TRANSVERSE:
                    matrix.postRotate(270); matrix.postScale(-1, 1);
                    break;
                default:
                    return bitmap; // nessuna trasformazione
            }

            return Bitmap.createBitmap(bitmap, 0, 0,
                    bitmap.getWidth(), bitmap.getHeight(), matrix, true);
        }
    }


    public void copiaContenutoFileConLimiti(Uri sourceUri, Uri destUri) {
        if (sourceUri == null || destUri == null)
            throw new IllegalArgumentException("Gli URI di origine e destinazione non possono essere null");

        try {
            // Step 1: leggi dimensioni originali
            BitmapFactory.Options options = new BitmapFactory.Options();
            options.inJustDecodeBounds = true;
            try (InputStream in = context.getContentResolver().openInputStream(sourceUri)) {
                BitmapFactory.decodeStream(in, null, options);
            }

            int origW = options.outWidth;
            int origH = options.outHeight;
            if (origW <= 0 || origH <= 0)
                throw new IllegalStateException("Dimensioni immagine non valide o impossibile leggere il file");

            // Step 2: calcolo limiti
            long budgetBytes = getSafeBudgetBytes(0.25f); // 25% heap
            long maxPixels = getMaxPixels(budgetBytes, true);
            int[] targetSize = calculateTargetSize(origW, origH, maxPixels);
            int targetW = targetSize[0];
            int targetH = targetSize[1];

            // Step 3: calcola inSampleSize
            int inSampleSize = calculateInSampleSize(options, targetW, targetH);

            // Step 4: decodifica bitmap ridotta
            options.inJustDecodeBounds = false;
            options.inSampleSize = inSampleSize;
            Bitmap bitmap;
            try (InputStream in = context.getContentResolver().openInputStream(sourceUri)) {
                bitmap = BitmapFactory.decodeStream(in, null, options);
            }
            if (bitmap == null) throw new IOException("Impossibile decodificare l'immagine dalla sorgente");

            // Step 5: ridimensiona ulteriormente se serve
            if (bitmap.getWidth() > targetW || bitmap.getHeight() > targetH) {
                bitmap = Bitmap.createScaledBitmap(bitmap, targetW, targetH, true);
            }

            // Step 6: ruota l'immagine se necessario
            bitmap = applyExifOrientation(sourceUri, bitmap);

            // Step 7: scrivi JPEG
            try (OutputStream out = context.getContentResolver().openOutputStream(destUri)) {
                if (out == null) throw new IOException("Impossibile aprire lo stream di destinazione");
                if (!bitmap.compress(Bitmap.CompressFormat.JPEG, 90, out))
                    throw new IOException("Errore durante la compressione/scrittura del file JPEG");
            }

        } catch (Exception e) {
            notifyError(e);
        }
    }

    public interface OnErrorListener{
        void onError(Exception exc);
    }

}
