#!/bin/bash
# Native build script for Android (restic, rclone, PRoot, talloc)
set -eo pipefail

# -------------------------------
#  Banner
# -------------------------------
echo "================================="
echo "Building Native Binaries from Source"
echo "================================="

# -------------------------------
#  Version pins (override via env)
# -------------------------------
RESTIC_VERSION="${RESTIC_VERSION:-0.18.1}"
RCLONE_VERSION="${RCLONE_VERSION:-1.68.2}"
MIN_API_LEVEL=24

# -------------------------------
#  Directories
# -------------------------------
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
BUILD_DIR="$SCRIPT_DIR/build/native-build"
SOURCE_DIR="$BUILD_DIR/sources"
OUTPUT_DIR="$SCRIPT_DIR/app/src/main/jniLibs"

mkdir -p "$BUILD_DIR" "$SOURCE_DIR" "$OUTPUT_DIR"

# -------------------------------
#  Go setup
# -------------------------------
export PATH=$PATH:/usr/local/go/bin

# -------------------------------
#  Reproducible build setup
# -------------------------------
# Set SOURCE_DATE_EPOCH for reproducible builds (F-Droid requirement)
export SOURCE_DATE_EPOCH="${SOURCE_DATE_EPOCH:-$(date +%s)}"
echo "Using SOURCE_DATE_EPOCH: $SOURCE_DATE_EPOCH ($(date -r $SOURCE_DATE_EPOCH 2>/dev/null || date -d @$SOURCE_DATE_EPOCH))"

# Set Go build environment for reproducible builds
export GOFLAGS="-trimpath -ldflags=-buildid="
export CGO_CFLAGS="-g0 -O2"
export CGO_LDFLAGS="-s -w"

# -------------------------------
#  Android NDK setup (provided by Docker)
# -------------------------------
export ANDROID_NDK_HOME="${ANDROID_NDK_HOME:-/opt/android-ndk}"
export NDK="$ANDROID_NDK_HOME"
export PREBUILT_TAG="linux-x86_64"

echo "Using NDK: $NDK"

# -------------------------------
#  Architecture mappings (ARM64 only)
# -------------------------------
GO_ARCHS_arm64_v8a="arm64"
NDK_ARCH_ABI_arm64_v8a="aarch64-linux-android"

# -------------------------------
#  Utility: download sources
# -------------------------------
download_source() {
  local name="$1" url="$2" target_dir="$3"
  echo "📦 Downloading $name..."
  if [ -d "$target_dir" ]; then
    echo "Already present: $target_dir"
    return
  fi
  local tmp="$(mktemp)"
  curl -sSfL "$url" -o "$tmp"
  mkdir -p "$target_dir"
  tar -xzf "$tmp" -C "$target_dir" --strip-components=1
  rm "$tmp"
}

# -------------------------------
#  Utility: clone git repository
# -------------------------------
clone_repo() {
  local name="$1" url="$2" target_dir="$3"
  echo "📦 Cloning $name repository..."
  if [ -d "$target_dir" ]; then
    echo "Already present: $target_dir"
    return
  fi
  git clone --depth 1 "$url" "$target_dir"
}

# -------------------------------
#  Build Go-based tools
# -------------------------------
build_go_binary() {
  local name="$1" src="$2" out="$3" arch="$4"
  local arch_var="${arch//-/_}"
  local go_arch_var="GO_ARCHS_${arch_var}"
  local go_arch=$(eval echo "\$$go_arch_var")
  local ndk_arch_var="NDK_ARCH_ABI_${arch_var}"
  local ndk_arch=$(eval echo "\$$ndk_arch_var")
  local out_dir="$OUTPUT_DIR/$arch"
  mkdir -p "$out_dir"

  export GOOS=android
  export GOARCH="$go_arch"
  export CGO_ENABLED=1
  export CC="$NDK/toolchains/llvm/prebuilt/$PREBUILT_TAG/bin/${ndk_arch}${MIN_API_LEVEL}-clang"

  # Additional reproducible build environment variables
  export CGO_CFLAGS="-g0 -O2 -Wno-unused-command-line-argument"
  export CGO_LDFLAGS="-s -w"

  echo "🏗️  Building $name for $arch..."
  pushd "$src" >/dev/null

  # Build with reproducible flags - SOURCE_DATE_EPOCH ensures consistent timestamps
  local ldflags="-s -w -buildid="
  if [ -n "$SOURCE_DATE_EPOCH" ]; then
    # Format SOURCE_DATE_EPOCH as RFC3339 for Go builds
    local build_date=$(date -u -d "@$SOURCE_DATE_EPOCH" +"%Y-%m-%dT%H:%M:%SZ" 2>/dev/null || date -u -r "$SOURCE_DATE_EPOCH" +"%Y-%m-%dT%H:%M:%SZ" 2>/dev/null || echo "")
    if [ -n "$build_date" ]; then
      ldflags="$ldflags -X main.version=$build_date"
    fi
  fi

  case "$name" in
    restic) go build -trimpath -ldflags="$ldflags" -o "$out_dir/$out" ./cmd/restic ;;
    rclone) go build -trimpath -ldflags="$ldflags" -o "$out_dir/$out" . ;;
  esac
  popd >/dev/null
  [ -f "$out_dir/$out" ] || { echo "✗ Failed to build $name ($arch)"; exit 1; }

  # Strip any remaining build metadata for reproducible builds
  if command -v strip >/dev/null 2>&1; then
    strip --strip-all "$out_dir/$out" 2>/dev/null || true
  fi

  echo "✅ Built $name → $out_dir/$out"
}


# -------------------------------
#  Main build pipeline
# -------------------------------
main() {
  echo "📥 Step 1: Downloading sources"
  download_source "restic" "https://github.com/restic/restic/archive/refs/tags/v${RESTIC_VERSION}.tar.gz" "$SOURCE_DIR/restic"
  download_source "rclone" "https://github.com/rclone/rclone/archive/refs/tags/v${RCLONE_VERSION}.tar.gz" "$SOURCE_DIR/rclone"

  echo "⚙️  Step 2: Building PRoot"
  # Instead of hardcoded /build/build-native-binaries.sh
  if [ -f "/build/build-native-binaries.sh" ]; then
      /build/build-native-binaries.sh  # Docker
  else
      ./build-native-binaries.sh      # Direct execution
  fi

  echo "💻 Step 3: Building Go binaries (restic & rclone)"
  for arch in arm64-v8a; do
    build_go_binary "restic" "$SOURCE_DIR/restic" "libdata_restic.so" "$arch"
    build_go_binary "rclone" "$SOURCE_DIR/rclone" "libdata_rclone.so" "$arch"
  done

  echo "🎉 All native libraries built successfully!"
}

main "$@"
