/*
 * Copyright © 2016 Mozilla Foundation
 *
 * This program is made available under an ISC-style license.  See the
 * accompanying file LICENSE for details.
 */

/* libcubeb api/function test. Record the mic and check there is sound. */
#include "gtest/gtest.h"
#if !defined(_XOPEN_SOURCE)
#define _XOPEN_SOURCE 600
#endif
#include "cubeb/cubeb.h"
#include <atomic>
#include <math.h>
#include <memory>
#include <stdio.h>
#include <stdlib.h>

// #define ENABLE_NORMAL_LOG
// #define ENABLE_VERBOSE_LOG
#include "common.h"

#define SAMPLE_FREQUENCY 48000
#define STREAM_FORMAT CUBEB_SAMPLE_FLOAT32LE

struct user_state_record {
  std::atomic<int> invalid_audio_value{0};
};

long
data_cb_record(cubeb_stream * stream, void * user, const void * inputbuffer,
               void * outputbuffer, long nframes)
{
  user_state_record * u = reinterpret_cast<user_state_record *>(user);
  float * b = (float *)inputbuffer;

  if (stream == NULL || inputbuffer == NULL || outputbuffer != NULL) {
    return CUBEB_ERROR;
  }

  for (long i = 0; i < nframes; i++) {
    if (b[i] <= -1.0 || b[i] >= 1.0) {
      u->invalid_audio_value = 1;
      break;
    }
  }

  return nframes;
}

void
state_cb_record(cubeb_stream * stream, void * /*user*/, cubeb_state state)
{
  if (stream == NULL)
    return;

  switch (state) {
  case CUBEB_STATE_STARTED:
    fprintf(stderr, "stream started\n");
    break;
  case CUBEB_STATE_STOPPED:
    fprintf(stderr, "stream stopped\n");
    break;
  case CUBEB_STATE_DRAINED:
    fprintf(stderr, "stream drained\n");
    break;
  default:
    fprintf(stderr, "unknown stream state %d\n", state);
  }

  return;
}

TEST(cubeb, record)
{
  if (cubeb_set_log_callback(CUBEB_LOG_DISABLED, nullptr /*print_log*/) !=
      CUBEB_OK) {
    fprintf(stderr, "Set log callback failed\n");
  }
  cubeb * ctx;
  cubeb_stream * stream;
  cubeb_stream_params params;
  int r;
  user_state_record stream_state;

  r = common_init(&ctx, "Cubeb record example");
  ASSERT_EQ(r, CUBEB_OK) << "Error initializing cubeb library";

  std::unique_ptr<cubeb, decltype(&cubeb_destroy)> cleanup_cubeb_at_exit(
      ctx, cubeb_destroy);

  /* This test needs an available input device, skip it if this host does not
   * have one. */
  if (!can_run_audio_input_test(ctx)) {
    return;
  }

  params.format = STREAM_FORMAT;
  params.rate = SAMPLE_FREQUENCY;
  params.channels = 1;
  params.layout = CUBEB_LAYOUT_UNDEFINED;
  params.prefs = CUBEB_STREAM_PREF_NONE;

  r = cubeb_stream_init(ctx, &stream, "Cubeb record (mono)", NULL, &params,
                        NULL, nullptr, 4096, data_cb_record, state_cb_record,
                        &stream_state);
  ASSERT_EQ(r, CUBEB_OK) << "Error initializing cubeb stream";

  std::unique_ptr<cubeb_stream, decltype(&cubeb_stream_destroy)>
      cleanup_stream_at_exit(stream, cubeb_stream_destroy);

  cubeb_stream_start(stream);
  delay(500);
  cubeb_stream_stop(stream);

#ifdef __linux__
  // user callback does not arrive in Linux, silence the error
  fprintf(stderr, "Check is disabled in Linux\n");
#else
  ASSERT_FALSE(stream_state.invalid_audio_value.load());
#endif
}

#undef SAMPLE_FREQUENCY
#undef STREAM_FORMAT
