package org.cryptomator.data.executor;

import org.cryptomator.domain.executor.ThreadExecutor;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.inject.Inject;
import javax.inject.Singleton;

/**
 * Decorated {@link java.util.concurrent.ThreadPoolExecutor}
 */
@Singleton
public class JobExecutor implements ThreadExecutor {

	private static final int INITIAL_POOL_SIZE = 3;
	private static final int MAX_POOL_SIZE = 5;

	// Sets the amount of time an idle thread waits before terminating
	private static final int KEEP_ALIVE_TIME = 10;

	// Sets the Time Unit to seconds
	private static final TimeUnit KEEP_ALIVE_TIME_UNIT = TimeUnit.SECONDS;

	private final ThreadPoolExecutor threadPoolExecutor;

	@Inject
	public JobExecutor() {
		BlockingQueue<Runnable> workQueue = new LinkedBlockingQueue<>();
		ThreadFactory threadFactory = new JobThreadFactory();
		this.threadPoolExecutor = new ThreadPoolExecutor(INITIAL_POOL_SIZE, MAX_POOL_SIZE, KEEP_ALIVE_TIME, KEEP_ALIVE_TIME_UNIT, workQueue, threadFactory);
	}

	@Override
	public void execute(@NotNull Runnable runnable) {
		if (runnable == null) {
			throw new IllegalArgumentException("Runnable to execute cannot be null");
		}
		this.threadPoolExecutor.execute(runnable);
	}

	private static class JobThreadFactory implements ThreadFactory {

		private static final String THREAD_NAME = "android_";
		private int counter = 0;

		@Override
		public Thread newThread(@NotNull Runnable runnable) {
			return new Thread(runnable, THREAD_NAME + counter++);
		}
	}
}
