/**
 * License: GPL-3.0-or-later
 * Author: veto@myridia.com
 * Description: Main Activity Screen
 */
package org.calantas.mygeo;

import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.location.GnssStatus;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuItem;

import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.core.app.ActivityCompat;

import java.text.MessageFormat;


public class Main extends AppCompatActivity {
    LocationManager lm;
    int log_counter = 1;
    boolean app_ok = false;

    boolean gps;
    boolean net;
    int satellites = 0;
    Lib lib;

    MediaPlayer amp;
    LocationListener listen_callback = new LocationListener() {
        @Override
        public void onLocationChanged(Location loc) {
            //log_me("...location changed");
            Double last_lat = loc.getLatitude();
            Double last_long = loc.getLongitude();
            set_status(last_lat, last_long);
            stop_graphic_status_animation();
            save_cord_data(String.valueOf(last_lat), String.valueOf(last_long));
            lib.save_date_data(getApplicationContext());
        }
    };

    private void log_me(String message) {
        log_counter += 1;
        lib.log_me(log_counter + ". " + message, findViewById(R.id.msg), true);
    }


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);     // set the main window
        lib = new Lib(getApplicationContext()); //init general used function class Lib.java
        //log_me("...successfully created main activity windows ");
        Toolbar toolbar = findViewById(R.id.my_toolbar); //find the toolbar from the xml layout
        toolbar.setTitle("myGeo - Main");
        setSupportActionBar(toolbar); // add toolbar to the main screen

        toolbar.setNavigationOnClickListener(v -> log_me("...click logo icon"));



        boolean p = lib.allPermissionsGranted();
        if (p) {
            log_me("...permission ok, activate my-geo");
            start_geo_location(); // init the geo location
            start_graphic_status_animation(); // animate the status icons
            share_click_listen(); // listen on graphic clicks
        } else {

            lib.request_permission(this);
        }


    }

    @Override
    public void onRequestPermissionsResult(int request_code, @NonNull String[] permissions, @NonNull int[] grant_results) {
        //log_me("...request code " + String.valueOf(request_code));
        super.onRequestPermissionsResult(request_code, permissions, grant_results);
        boolean p = lib.allPermissionsGranted();
        if (p && !app_ok) {
            log_me("...permission ok, activate my-geo");
            app_ok = true;
            start_geo_location(); // init the geo location
            start_graphic_status_animation(); // animate the status icons
            share_click_listen(); // listen on graphic clicks

        }
    }

    @Override
    protected void onStart() {
        super.onStart();
        //log_me("...start main activity");
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(@NonNull MenuItem item) {
        return lib.drop_menu(item, this);
    }

    @Override
    protected void onRestart() {
        super.onRestart();
        //log_me("...restart Activity");
    }

    @Override
    protected void onResume() {
        super.onResume();
        //log_me("...resume Activity");
    }

    @Override
    protected void onPause() {
        super.onPause();
        //log_me("...pause Activity");
    }

    @Override
    protected void onStop() {
        super.onStop();
        //log_me("...stop Activity");
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
    }

    private void start_geo_location() {
        //log_me("...start geo location");

        lm = (LocationManager) getSystemService(LOCATION_SERVICE); //init location manager
        gps = lm.isProviderEnabled(LocationManager.GPS_PROVIDER); // check if can use GPS to get the cords
        net = lm.isProviderEnabled(LocationManager.NETWORK_PROVIDER); // check if can use Net to get the cords

        if (net) {
            //log_me("...net service is activated, use it");
            if (ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED && ActivityCompat.checkSelfPermission(this, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                return;
            }
            lm.requestLocationUpdates(LocationManager.NETWORK_PROVIDER, 1000, 0, listen_callback);
        } else if (gps) {
            //log_me("...gps service is activated, use it");
            lm.requestLocationUpdates(LocationManager.GPS_PROVIDER, 1000, 0, listen_callback);
            Gnss_callback gps_callback = new Gnss_callback(this.satellites, lm);
            lm.registerGnssStatusCallback(gps_callback);
        } else {
            log_me("...none Geo Service is activated, cannot use this app");
        }

    }

    public void alert() {
        Boolean alert = lib.get_alert(getApplicationContext());
        if (alert && amp == null) {
            Uri sound = Uri.parse("android.resource://" + getPackageName() + "/" + R.raw.sos);
            amp = MediaPlayer.create(getApplicationContext(), sound);
            amp.setVolume(100, 100);
            amp.start();
        }
    }


    public String get_data(String _key) {
        Context context = getApplicationContext();
        SharedPreferences pref = context.getSharedPreferences(getString(R.string.preference_file_key), Context.MODE_PRIVATE);
        return pref.getString(_key, "");
    }

    public void save_cord_data(String _lat, String _lon) {
        Context context = getApplicationContext();
        SharedPreferences pref = context.getSharedPreferences(getString(R.string.preference_file_key), Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = pref.edit();
        editor.putString("lat", _lat);
        editor.putString("lon", _lon);
        editor.apply();
    }

    public void set_status(Double _lat, Double _long) {
        TextView l = findViewById(R.id.last_fix);
        String latX = String.valueOf(_lat);
        String longX = String.valueOf(_long);
        String last_fix = String.format("Last Fix: %s, %s",latX,longX);
        l.setText(last_fix);
    }

    public void start_graphic_status_animation() {
        ImageView image = findViewById(R.id.earth);
        Animation ani = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.rotate);
        image.startAnimation(ani);

        ImageView image1 = findViewById(R.id.sat1);
        Animation ani1 = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.blink);
        image1.startAnimation(ani1);

        ImageView image2 = findViewById(R.id.sat2);
        Animation ani2 = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.blink);
        image2.startAnimation(ani2);
    }

    public void stop_graphic_status_animation() {
        ImageView image = findViewById(R.id.earth);
        //Animation ani = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.rotate);
        image.clearAnimation();

        ImageView image1 = findViewById(R.id.sat1);
        //Animation ani1 = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.blink);
        image1.clearAnimation();

        ImageView image2 = findViewById(R.id.sat2);
        //Animation ani2 = AnimationUtils.loadAnimation(getApplicationContext(), R.anim.blink);
        image2.clearAnimation();
    }

    public void share_click_listen() {



        final Button btn_share_my_template = findViewById(R.id.btn_share_template);
        btn_share_my_template.setOnClickListener(v -> {
            log_me("...click share my template");
            String last_lat = get_data("lat");
            String last_long = get_data("lon");
            String last_date = lib.get_saved_date_data(getApplicationContext());
            Context context = getApplicationContext();
            SharedPreferences pref = context.getSharedPreferences(getString(R.string.preference_file_key), Context.MODE_PRIVATE);
            String draft = pref.getString("template", "");
            String subject = MessageFormat.format("{0},{1}", last_lat, last_long);
            String body = MessageFormat.format(draft, last_lat, last_long, last_date);

            Intent intentX = new Intent(Intent.ACTION_SEND);
            intentX.setType("text/plain");
            intentX.putExtra(Intent.EXTRA_SUBJECT, subject);
            intentX.putExtra(Intent.EXTRA_TEXT, body);
            //log_me(draft);
            startActivity(Intent.createChooser(intentX, "Share Via"));
        });

        final Button btn_share_cords = findViewById(R.id.btn_share_coordinates);
        btn_share_cords.setOnClickListener(v -> {
            log_me("...click share cords");
            String last_lat = get_data("lat");
            String last_long = get_data("lon");
            String last_date = lib.get_saved_date_data(getApplicationContext());
            String subject = MessageFormat.format("geo:{0},{1}", last_lat, last_long);
            String body = MessageFormat.format("geo:{0},{1} \n\ndate: {2}", last_lat, last_long, last_date);

            Intent intentX = new Intent(Intent.ACTION_SEND);
            intentX.setType("text/plain");
            intentX.putExtra(Intent.EXTRA_SUBJECT, subject);
            intentX.putExtra(Intent.EXTRA_TEXT, body);
            //log_me(body);
            startActivity(Intent.createChooser(intentX, "Share Via"));
        });


        final Button btn_share_camera = findViewById(R.id.btn_share_camera);
        btn_share_camera.setOnClickListener(v -> {
            log_me("...click share camera");
            Context context = getApplicationContext();
            Intent cam = new Intent(context, Camerax.class);
            cam.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            context.startActivity(cam);
        });


    }


    private class Gnss_callback extends GnssStatus.Callback {
        int satellites;
        //TextView s = findViewById(R.id.sat_count);

        Gnss_callback(int sat, LocationManager _lm) {
            satellites = sat;
            lm = _lm;
        }



        @Override
        public void onFirstFix(int sec) {
            //log_me("...got first fix");
            LocationManager loc_m = (LocationManager) getSystemService(LOCATION_SERVICE); //init location manager
            Context c = getApplicationContext();
           if (ActivityCompat.checkSelfPermission(c, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED && ActivityCompat.checkSelfPermission(c, Manifest.permission.ACCESS_COARSE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                return;
            }
            Location ll = loc_m.getLastKnownLocation(LocationManager.GPS_PROVIDER);
            assert ll != null;
            Double last_lat = ll.getLatitude();
            Double last_long = ll.getLongitude();
            set_status(last_lat, last_long);
            stop_graphic_status_animation();
            save_cord_data(String.valueOf(last_lat), String.valueOf(last_long));
            lib.save_date_data(getApplicationContext());
            alert();
        }

        @Override
        public void onSatelliteStatusChanged(GnssStatus status) {
            if (satellites != status.getSatelliteCount()) {
                satellites = status.getSatelliteCount();
                String count = String.valueOf(status.getSatelliteCount());
                String msg = String.format("Satellites: %s",count);
                TextView s = findViewById(R.id.sat_count);
                s.setText(msg);
                //log_me("...satellite Counts " + status.getSatelliteCount());
            }
        }
    }


}


