import 'dart:collection';
import 'dart:convert';

import 'package:archive/archive_io.dart';

/// Returns a list of Unicode emoji.
List<Emoji> getUnicodeEmojiList() => _data;

enum EmojiStatus { component, fullyQualified, minimallyQualified, unqualified }

/// Represents information about Emoji.
class Emoji {
  final List<int> codePoints;

  final String group;

  final String name;

  final EmojiStatus status;

  final String subgroup;

  final String version;

  Emoji({
    required this.codePoints,
    required this.group,
    required this.name,
    required this.status,
    required this.subgroup,
    required this.version,
  });

  @override
  String toString() {
    return String.fromCharCodes(codePoints);
  }

  /// Searches for emoji by name.
  ///
  /// The search is performed using the binary search algorithm.
  ///
  /// If a [status] value is specified, an emoji with the specified [status] is
  /// searched for.
  ///
  /// If no [status] value is specified, the emoji with the best [status] is
  /// searched for.
  ///
  /// In either case, if the [name] lookup succeeds and the [status] lookup
  /// fails, the best value found is returned.
  ///
  /// Since searching is a resource-intensive operation, it is recommended to
  /// call this method only to initialize static variables.
  static Emoji? findByName(String name, {EmojiStatus? status}) {
    var left = 0;
    var right = _data.length;
    int middle;
    while (left < right) {
      middle = (left + right) >> 1;
      final element = _data[middle];
      final result = name.compareTo(element.name);
      if (result > 0) {
        left = middle + 1;
      } else if (result < 0) {
        right = middle;
      } else if (result == 0) {
        if (element.status == status) {
          return element;
        }

        final elements = [element];
        for (var i = middle + 1; i < right; i++) {
          final value = _data[i];
          if (value.name != name) {
            break;
          }

          if (value.status == status) {
            return value;
          }

          elements.add(value);
        }

        for (var i = middle - 1; i >= 0; i--) {
          final value = _data[i];
          if (value.name != name) {
            break;
          }

          if (value.status == status) {
            return value;
          }

          elements.add(value);
        }

        elements.sort((a, b) {
          int getRank(EmojiStatus s) {
            switch (s) {
              case EmojiStatus.component:
                return 3;
              case EmojiStatus.fullyQualified:
                return 0;
              case EmojiStatus.minimallyQualified:
                return 1;
              case EmojiStatus.unqualified:
                return 2;
            }
          }

          return getRank(a.status).compareTo(getRank(b.status));
        });

        return elements.first;
      }
    }

    return null;
  }
}

List<Emoji> _build(String data) {
  final result = <Emoji>[];
  final bytes = base64.decode(data);
  final charCodes = GZipDecoder().decodeBytes(bytes);
  final source = String.fromCharCodes(charCodes);
  final cache = <String, String>{};
  final lines = const LineSplitter().convert(source);

  String fromCache(String s) {
    cache[s] ??= s;
    return cache[s]!;
  }

  for (var i = 0; i < lines.length; i++) {
    final line = lines[i];
    final parts = line.split(';');
    final sequence =
        parts[0].split(' ').map((e) => int.parse(e, radix: 16)).toList();
    final status = fromCache(parts[1]);
    final name = fromCache(parts[2]);
    final group = fromCache(parts[3]);
    final subgroup = fromCache(parts[4]);
    final version = fromCache(parts[5]);

    EmojiStatus findStatus(String name) {
      switch (name) {
        case 'fully-qualified':
          return EmojiStatus.fullyQualified;
        case 'minimally-qualified':
          return EmojiStatus.minimallyQualified;
        case 'unqualified':
          return EmojiStatus.unqualified;
        case 'component':
          return EmojiStatus.component;
        default:
          throw StateError('Internal error: unknown emoji status ($status)');
      }
    }

    final emoji = Emoji(
        codePoints: UnmodifiableListView(sequence),
        group: group,
        name: name,
        status: findStatus(status),
        subgroup: subgroup,
        version: version);
    result.add(emoji);
  }

  result.sort((a, b) {
    return a.name.compareTo(b.name);
  });
  return UnmodifiableListView(result);
}

final _data = _build('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');
