part of 'internal_parts.dart';

/// Used to hold a list of [Dependency]s from
/// a single dependency section in the pubspec.yaml
///
/// e.g. the list of dependancies for the 'dependencies' key or
/// the 'dev_dependencies' in pubspec.yaml
class Dependencies {
  /// The [Document] [Section] that holds the dependencies
  late SectionImpl _section;

  /// The name of the dependency section such as
  /// dev_dependencies
  late final String name;

  /// reference to the pubspec that has these dependencies.
  final PubSpec _pubspec;

  final _dependencies = <Dependency>[];

  Dependencies._missing(this._pubspec, this.name)
      : _section = SectionImpl.missing(_pubspec.document, name);

  Dependencies._fromLine(this._pubspec, LineImpl line) {
    name = line.key;
    _section = SectionImpl.fromLine(line);
  }

  /// List of the dependencies as an unmodifiable list.
  ///
  /// To modify the list of dependencies use [add], [remove] and [removeAll].
  List<Dependency> get list => List.unmodifiable(_dependencies);

  /// The number of dependencies in this section
  int get length => _dependencies.length;

  /// Returns the [Dependency] with the given [name]
  /// if it exists in this section's list of dependencies.
  /// Returns null if it doesn't exist.
  Dependency? operator [](String name) {
    for (final dependency in _dependencies) {
      if (dependency.name == name) {
        return dependency;
      }
    }
    return null;
  }

  /// Add the [Dependency] generated by the provided [dependencyBuilder] to
  /// the PubSpec and return the generated
  /// [Dependency] to allow calls to be chained.
  /// A future version will sort the list of depenendancies as
  /// they are added.
  ///
  /// See: [remove] and [removeAll]
  Dependency add(DependencyBuilder dependencyBuilder) {
    // if we don't have a dependencies section then create it.
    if (_section.missing) {
      _section = SectionImpl.fromLine(
          _section.document.append(LineDetached('$name:')));
    }

    Line lineBefore = _section.headerLine;
    if (_dependencies.isNotEmpty) {
      lineBefore = _dependencies.last._section.lines.last;
    }
    final attached = dependencyBuilder._attach(this, _pubspec, lineBefore);

    _dependencies.add(attached);

    return attached;
  }

  /// register a dependency that is already attached.
  Dependency _appendAttached(Dependency dependency) {
    _dependencies.add(dependency);
    return dependency;
  }

  /// Remove all dependencies from the section.
  void removeAll() {
    _dependencies.removeWhere((value) => true);
  }

  /// Remove a dependency with the given [name] from the section.
  /// Throws a [DependencyNotFound] exception if the
  /// dependency doesn't exist.
  void remove(String name) {
    final dependency = this[name];

    if (dependency == null) {
      throw DependencyNotFound(
          _pubspec.document, '$name not found in the ${this.name} section');
    }

    _dependencies.remove(dependency);
    final lines = dependency._section.lines;
    for (final line in lines) {
      _section._removeChild(line);
    }
  }

  /// Returns true if the list of dependencies contains a dependency
  /// with the given name.
  bool exists(String name) => this[name] != null;

  /// List of comments associated with this dependencies section.
  Comments get comments => _section.comments;
}
