#!/bin/bash

set -o errexit
set -o nounset
set -o pipefail

if [[ ${TRACE-0} == "1" ]]; then set -o xtrace; fi

trap 'echo "Error occurred at line $LINENO. Command: $BASH_COMMAND"' ERR

# Get current dir
CURRENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" >/dev/null 2>&1 && pwd)"
readonly CURRENT_DIR

# Game grids file (dart GameData object)
GAME_GRIDS_FILE="${CURRENT_DIR}/../lib/data/game_data.dart"
# Grids as raw strings
GRIDS="$(cat "${GAME_GRIDS_FILE}" | grep "',$" | sed "s/ //g" | sed "s/,//g" | sed "s/'//g")"

readonly GAME_GRIDS_FILE
readonly GRIDS

# Parameters
ALLOWED_BLOCK_SIZE_VALUES="${1:-}"
if [[ -z "${ALLOWED_BLOCK_SIZE_VALUES}" ]]; then
  ALLOWED_BLOCK_SIZE_VALUES="2x2 3x2 3x3"
fi
MAX_GRIDS_COUNT_PER_LEVEL=300

readonly ALLOWED_BLOCK_SIZE_VALUES
readonly MAX_GRIDS_COUNT_PER_LEVEL

MAX_GRIDS_COUNT_PER_SIZE="$(echo "3*${MAX_GRIDS_COUNT_PER_LEVEL}" | bc)"
readonly MAX_GRIDS_COUNT_PER_SIZE

# Get block size from grid string
function get_block_size() {
  local GRID="${1:-}"
  local LOCAL_GRID_LENGTH

  LOCAL_GRID_LENGTH="$(echo -n "${GRID}" | wc -c)"
  case "${LOCAL_GRID_LENGTH}" in
    "16")
        echo "2x2"
        ;;
    "36")
        echo "3x2"
        ;;
    "81")
        echo "3x3"
        ;;
    *)
        echo "0x0"
        ;;
  esac
}

# Count "0" in grid, compute "emptiness" ratio
function get_emptiness() {
  local LOCAL_GRID="${1:-}"
  local LOCAL_CELLS_COUNT
  local LOCAL_STRING
  local LOCAL_ZEROS_COUNT
  local LOCAL_RATIO

  LOCAL_CELLS_COUNT=$(echo "($(get_block_size "${LOCAL_GRID}"))^2" | sed 's/x/\*/g' | bc)
  LOCAL_STRING="${LOCAL_GRID//[^0]}"
  LOCAL_ZEROS_COUNT="${#LOCAL_STRING}"
  LOCAL_RATIO=$(echo "100*${LOCAL_ZEROS_COUNT}/${LOCAL_CELLS_COUNT}" | bc)

  printf "%02d" "${LOCAL_RATIO}"
}

echo "Will keep max ${MAX_GRIDS_COUNT_PER_LEVEL} grids per level."
echo "Will keep max ${MAX_GRIDS_COUNT_PER_SIZE} grids per size."
echo ""

for BLOCK_SIZE in ${ALLOWED_BLOCK_SIZE_VALUES}; do
  echo "Sorting grids [${BLOCK_SIZE}]"

  # Pick only grids with expected "block size", compute emptiness
  GRIDS_FOR_CURRENT_SIZE=""
  for GRID in ${GRIDS}; do
    GRID_BLOCK_SIZE="$(get_block_size "${GRID}")"
    if [[ "${GRID_BLOCK_SIZE}" == "${BLOCK_SIZE}" ]]; then
      # compute emptyness
      EMPTYNESS="$(get_emptiness "${GRID}")"
      # keep only grid if > 40%
      if [[ "${EMPTYNESS}" -gt 40 ]]; then
        # add grid to selected grids for this size
        GRIDS_FOR_CURRENT_SIZE="$(echo -e "${GRIDS_FOR_CURRENT_SIZE}\n${EMPTYNESS};${GRID}" | sed '/^$/d' | sort | uniq)"
      fi
    fi
  done

  GRIDS_COUNT=$(echo "${GRIDS_FOR_CURRENT_SIZE}" | wc -l | awk '{print $1}')
  echo " found ${GRIDS_COUNT} grids"

  # keep only 3xMAX_GRIDS_COUNT_PER_LEVEL grids (randomly picked)
  TRUNCATED_GRIDS_LIST="$(echo "${GRIDS_FOR_CURRENT_SIZE}" | shuf --head-count="${MAX_GRIDS_COUNT_PER_SIZE}")"
  GRIDS_COUNT=$(echo "${TRUNCATED_GRIDS_LIST}" | wc -l | awk '{print $1}')
  echo " truncated to ${GRIDS_COUNT} grids"

  # sort grids by "emptiness", keep only grid part
  SORTED_GRIDS="$(echo "${TRUNCATED_GRIDS_LIST}" | sort | cut -d";" -f2)"

  # split grids per level
  GRIDS_COUNT_PER_LEVEL="$(echo "${GRIDS_COUNT} / 3" | bc)"
  TMP_FOLDER="$(mktemp -d)"
  echo "${SORTED_GRIDS}" | split --lines="${GRIDS_COUNT_PER_LEVEL}" - "${TMP_FOLDER}/grids-"
  GRIDS_LEVEL_EASY="$(cat "${TMP_FOLDER}/grids-aa")"
  GRIDS_LEVEL_MEDIUM="$(cat "${TMP_FOLDER}/grids-ab")"
  GRIDS_LEVEL_HARD="$(cat "${TMP_FOLDER}/grids-ac")"
  rm -rf "${TMP_FOLDER}"

  # delete all existing grids for this size
  CELLS_COUNT="$(echo "(${BLOCK_SIZE})^2" | sed 's/x/\*/g' | bc)"
  sed -i "/'[0-9A-G]\{${CELLS_COUNT}\}',/d" "${GAME_GRIDS_FILE}"

  # restore grids
  # (sort --reverse <- lines are inserted backwards)
  echo " saving grids..."
  for GRID in $(echo "${GRIDS_LEVEL_EASY}" | sort --reverse | uniq); do
    sed -i "/\/\/ ${BLOCK_SIZE}-easy grids/a \ \ \ \ \ \ \ \ '${GRID}'," "${GAME_GRIDS_FILE}"
  done
  for GRID in $(echo "${GRIDS_LEVEL_MEDIUM}" | sort --reverse | uniq); do
    sed -i "/\/\/ ${BLOCK_SIZE}-medium grids/a \ \ \ \ \ \ \ \ '${GRID}'," "${GAME_GRIDS_FILE}"
  done
  for GRID in $(echo "${GRIDS_LEVEL_HARD}" | sort --reverse | uniq); do
    sed -i "/\/\/ ${BLOCK_SIZE}-hard grids/a \ \ \ \ \ \ \ \ '${GRID}'," "${GAME_GRIDS_FILE}"
  done

  echo "done for ${BLOCK_SIZE}."
  echo ""
done

echo "Grids saved in ${GAME_GRIDS_FILE}"
echo "Ok, done."
